package com.google.appengine.api.labs.modules;

import com.google.appengine.api.labs.modules.ModulesServicePb.GetDefaultVersionRequest;
import com.google.appengine.api.labs.modules.ModulesServicePb.GetDefaultVersionResponse;
import com.google.appengine.api.labs.modules.ModulesServicePb.GetHostnameRequest;
import com.google.appengine.api.labs.modules.ModulesServicePb.GetHostnameResponse;
import com.google.appengine.api.labs.modules.ModulesServicePb.GetModulesRequest;
import com.google.appengine.api.labs.modules.ModulesServicePb.GetModulesResponse;
import com.google.appengine.api.labs.modules.ModulesServicePb.GetNumInstancesRequest;
import com.google.appengine.api.labs.modules.ModulesServicePb.GetNumInstancesResponse;
import com.google.appengine.api.labs.modules.ModulesServicePb.GetVersionsRequest;
import com.google.appengine.api.labs.modules.ModulesServicePb.GetVersionsResponse;
import com.google.appengine.api.labs.modules.ModulesServicePb.ModulesServiceError.ErrorCode;
import com.google.appengine.api.labs.modules.ModulesServicePb.SetNumInstancesRequest;
import com.google.appengine.api.labs.modules.ModulesServicePb.SetNumInstancesResponse;
import com.google.appengine.api.labs.modules.ModulesServicePb.StartModuleRequest;
import com.google.appengine.api.labs.modules.ModulesServicePb.StartModuleResponse;
import com.google.appengine.api.labs.modules.ModulesServicePb.StopModuleRequest;
import com.google.appengine.api.labs.modules.ModulesServicePb.StopModuleResponse;
import com.google.appengine.api.utils.FutureWrapper;
import com.google.apphosting.api.ApiProxy;
import com.google.apphosting.api.ApiProxy.Environment;
import com.google.common.base.Splitter;
import com.google.common.collect.Sets;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;

import java.lang.reflect.UndeclaredThrowableException;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

class ModulesServiceImpl implements ModulesService {
  protected static final String PACKAGE = "modules";

  /**
   * Environment attribute key where the instance id is stored.
   *
   * @see ModulesService#getCurrentInstanceId()
   */
  private static final String INSTANCE_ID_ENV_ATTRIBUTE = "com.google.appengine.instance.id";

  @Override
  public String getCurrentModule() {
    return ApiProxy.getCurrentEnvironment().getModuleId();
  }

  @Override
  public String getCurrentVersion() {
    Environment env = ApiProxy.getCurrentEnvironment();
    return Splitter.on('.').split(env.getVersionId()).iterator().next();
  }

  private static Map<String, Object> getThreadLocalAttributes() {
    return ApiProxy.getCurrentEnvironment().getAttributes();
  }

  @Override
  public String getCurrentInstanceId() {
    Map<String, Object> env = getThreadLocalAttributes();
    if (!env.containsKey(INSTANCE_ID_ENV_ATTRIBUTE)) {
      throw new ModulesException("No valid instance id for this instance.");
    }
    String instanceId = (String) getThreadLocalAttributes().get(INSTANCE_ID_ENV_ATTRIBUTE);
    if (instanceId == null) {
      throw new ModulesException("No valid instance id for this instance.");
    }
    return instanceId;
  }

  /**
   * Returns the result from the provided future in the form suitable for a synchronous call.
   * <p>
   * If {@link Future#get} throws an {@link ExecutionException}
   * <ol>
   * <li> if {@link ExecutionException#getCause()} is an unchecked exception
   * including {@link ModulesException} this throws the unchecked exception.
   * </li>
   * <li> otherwise {@link UndeclaredThrowableException} with the checked
   * {@link ExecutionException#getCause()} as the cause.
   * </li>
   * </ol>
   *
   * If {@link Future#get} throws an {@link InterruptedException} this throws a
   * {@link ModulesException} with the cause set to the InterruptedException.
   */
  <V> V getAsyncResult(Future<V> asyncResult) {
    try {
      return asyncResult.get();
    } catch (InterruptedException ie) {
      throw new ModulesException("Unexpected failure", ie);
    } catch (ExecutionException ee) {
      if (ee.getCause() instanceof RuntimeException) {
        throw (RuntimeException) ee.getCause();
      } else if (ee.getCause() instanceof Error) {
        throw (Error) ee.getCause();
      } else {
        throw new UndeclaredThrowableException(ee.getCause());
      }
    }
  }

  @Override
  public Set<String> getModules() {
    return getAsyncResult(getModulesAsync());
  }

  @Override
  public Future<Set<String>> getModulesAsync() {
    GetModulesRequest.Builder requestBuilder = GetModulesRequest.newBuilder();
    Future<Set<String>> result =
        new ModulesServiceFutureWrapper<Set<String>>("GetModules", requestBuilder) {

      @Override
      protected Set<String> wrap(byte[] key) throws InvalidProtocolBufferException {
        GetModulesResponse.Builder responseBuilder = GetModulesResponse.newBuilder();
        responseBuilder.mergeFrom(key);
        return Sets.newHashSet(responseBuilder.getModuleList());
      }
    };
    return result;
  }

  @Override
  public Set<String> getVersions(String module) {
    return getAsyncResult(getVersionsAsync(module));
  }

  @Override
  public Future<Set<String>> getVersionsAsync(String module) {
    GetVersionsRequest.Builder requestBuilder = GetVersionsRequest.newBuilder();
    requestBuilder.setModule(module);
    Future<Set<String>> result =
        new ModulesServiceFutureWrapper<Set<String>>("GetVersions", requestBuilder) {

      @Override
      protected Set<String> wrap(byte[] key) throws InvalidProtocolBufferException {
        GetVersionsResponse.Builder responseBuilder = GetVersionsResponse.newBuilder();
        responseBuilder.mergeFrom(key);
        return Sets.newHashSet(responseBuilder.getVersionList());
      }
    };
    return result;
  }

  @Override
  public String getDefaultVersion(String module) {
    return getAsyncResult(getDefaultVersionAsync(module));
  }

  @Override
  public Future<String> getDefaultVersionAsync(String module) {
    GetDefaultVersionRequest.Builder requestBuilder = GetDefaultVersionRequest.newBuilder();
    requestBuilder.setModule(module);
    Future<String> result =
        new ModulesServiceFutureWrapper<String>("GetDefaultVersion", requestBuilder) {

      @Override
      protected String wrap(byte[] key) throws InvalidProtocolBufferException {
        GetDefaultVersionResponse.Builder responseBuilder = GetDefaultVersionResponse.newBuilder();
        responseBuilder.mergeFrom(key);
        return responseBuilder.getVersion();
      }
    };
    return result;
  }

  @Override
  public long getNumInstances(String module, String version) {
    return getAsyncResult(getNumInstancesAsync(module, version));
  }

  @Override
  public Future<Long> getNumInstancesAsync(String module, String version) {
    GetNumInstancesRequest.Builder requestBuilder = GetNumInstancesRequest.newBuilder();
    requestBuilder.setModule(module);
    requestBuilder.setVersion(version);
    Future<Long> result = new ModulesServiceFutureWrapper<Long>("GetNumInstances", requestBuilder) {

      @Override
      protected Long wrap(byte[] key) throws InvalidProtocolBufferException {
        GetNumInstancesResponse.Builder responseBuilder = GetNumInstancesResponse.newBuilder();
        responseBuilder.mergeFrom(key);
        return responseBuilder.getInstances();
      }
    };
    return result;
  }

  @Override
  public void setNumInstances(String module, String version, long instances) {
    getAsyncResult(setNumInstancesAsync(module, version, instances));
  }

  @Override
  public Future<Void> setNumInstancesAsync(String module, String version, long instances) {
    SetNumInstancesRequest.Builder requestBuilder = SetNumInstancesRequest.newBuilder();
    requestBuilder.setModule(module);
    requestBuilder.setVersion(version);
    requestBuilder.setInstances(instances);
    Future<Void> result = new ModulesServiceFutureWrapper<Void>("SetNumInstances", requestBuilder) {

      @Override
      protected Void wrap(byte[] key) throws InvalidProtocolBufferException {
        SetNumInstancesResponse.Builder responseBuilder = SetNumInstancesResponse.newBuilder();
        responseBuilder.mergeFrom(key);
        return null;
      }
    };
    return result;
  }

  @Override
  public void startModule(String module, String version) {
    getAsyncResult(startModuleAsync(module, version));
  }

  @Override
  public Future<Void> startModuleAsync(String module, String version) {
    StartModuleRequest.Builder requestBuilder = StartModuleRequest.newBuilder();
    requestBuilder.setModule(module);
    requestBuilder.setVersion(version);
    Future<Void> result = new ModulesServiceFutureWrapper<Void>("StartModule", requestBuilder) {

      @Override
      protected Void wrap(byte[] key) throws InvalidProtocolBufferException {
        StartModuleResponse.Builder responseBuilder = StartModuleResponse.newBuilder();
        responseBuilder.mergeFrom(key);
        return null;
      }
    };
    return result;
  }

  @Override
  public void stopModule(String module, String version) {
    getAsyncResult(stopModuleAsync(module, version));
  }

  @Override
  public Future<Void> stopModuleAsync(String module, String version) {
    StopModuleRequest.Builder requestBuilder = StopModuleRequest.newBuilder();
    requestBuilder.setModule(module);
    requestBuilder.setVersion(version);
    Future<Void> result = new ModulesServiceFutureWrapper<Void>("StopModule", requestBuilder) {

      @Override
      protected Void wrap(byte[] key) throws InvalidProtocolBufferException {
        StopModuleResponse.Builder responseBuilder = StopModuleResponse.newBuilder();
        responseBuilder.mergeFrom(key);
        return null;
      }
    };
    return result;
  }

  private Future<String> getHostnameAsync(GetHostnameRequest.Builder requestBuilder) {
    Future<String> result =
        new ModulesServiceFutureWrapper<String>("GetHostname", requestBuilder) {

      @Override
      protected String wrap(byte[] key) throws InvalidProtocolBufferException {
        GetHostnameResponse.Builder responseBuilder = GetHostnameResponse.newBuilder();
        responseBuilder.mergeFrom(key);
        return responseBuilder.getHostname();
      }
    };
    return result;
  }

  private GetHostnameRequest.Builder newGetHostnameRequestBuilder(String module, String version) {
    GetHostnameRequest.Builder builder = GetHostnameRequest.newBuilder();
    if (module != null) {
      builder.setModule(module);
    }
    if (version != null) {
      builder.setVersion(version);
    }
    return builder;
  }

  @Override
  public String getModuleHostname(String module, String version) {
    return getAsyncResult(getModuleHostnameAsync(module, version));
  }

  @Override
  public Future<String> getModuleHostnameAsync(String module, String version) {
    GetHostnameRequest.Builder requestBuilder = newGetHostnameRequestBuilder(module, version);
    return getHostnameAsync(requestBuilder);
  }

  @Override
  public String getModuleHostname(String module, String version, int instance) {
    return getAsyncResult(getModuleHostnameAsync(module, version, instance));
  }

  @Override
  public Future<String> getModuleHostnameAsync(String module, String version, int instance) {
    GetHostnameRequest.Builder requestBuilder = newGetHostnameRequestBuilder(module, version);
    requestBuilder.setInstance(Integer.toString(instance));
    return getHostnameAsync(requestBuilder);
  }

  ModulesServiceImpl() { }

  private abstract static class ModulesServiceFutureWrapper<V> extends FutureWrapper<byte[], V> {
    private final String method;

    public ModulesServiceFutureWrapper(String method, Message.Builder request) {
      super(ApiProxy.makeAsyncCall(PACKAGE, method, request.build().toByteArray()));
      this.method = method;
    }

    @Override
    protected Throwable convertException(Throwable cause) {
      if (cause instanceof ApiProxy.ApplicationException) {
        return convertApplicationException(method, (ApiProxy.ApplicationException) cause);
      } else if (cause instanceof InvalidProtocolBufferException){
        return new ModulesException("Unexpected failure", cause);
      } else {
        return cause;
      }
    }

    private RuntimeException convertApplicationException(String method,
        ApiProxy.ApplicationException e) {
      switch(ErrorCode.valueOf(e.getApplicationError())) {
        case INVALID_MODULE:
          return new InvalidModuleException("Given module is not known.");
        case INVALID_VERSION:
          return new InvalidVersionException("Given module version is not known.");
        case INVALID_INSTANCES:
          return new InvalidInstanceException("Given instances value is invalid.");
        case UNEXPECTED_STATE:
          if (method.equals("StartModule")) {
            throw new ModuleAlreadyStartedException("Given module version is already started.");
          } else if (method.equals("StopModule")) {
            throw new ModuleAlreadyStoppedException("Given module version is already stopped.");
          }
        default:
          throw new ModulesException("Unknown error occurred.");
      }
    }
  }
}
