/**********************************************************************
Copyright (c) 2003 Andy Jefferson and others. All rights reserved. 
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.expression;

import java.util.Arrays;

import org.datanucleus.store.mapped.mapping.JavaTypeMapping;

/**
 * Literal for binary object.
 */
public class BinaryLiteral extends BinaryExpression implements Literal
{
    private final byte[] value;

    /**
     * Creates a binary literal
     * @param qs the QueryExpression
     * @param mapping the mapping
     * @param value the binary value
     */
    public BinaryLiteral(QueryExpression qs, JavaTypeMapping mapping, byte[] value)
    {
        super(qs);
        this.mapping = mapping;
        this.value = value;
        st.appendParameter(qs.getStoreManager().getMappingManager().getMappingWithDatastoreMapping(
            byte[].class, false, false, qs.getClassLoaderResolver()), value);
    }

    public BooleanExpression eq(ScalarExpression expr)
    {
        if (expr instanceof BinaryLiteral)
        {
            return new BooleanLiteral(qs, mapping, Arrays.equals(value, ((BinaryLiteral)expr).value));
        }
        else
        {
            return super.eq(expr);
        }
    }

    public BooleanExpression noteq(ScalarExpression expr)
    {
        if (expr instanceof BinaryLiteral)
        {
            return new BooleanLiteral(qs, mapping, !Arrays.equals(value, ((BinaryLiteral)expr).value));
        }
        else
        {
            return super.noteq(expr);
        }
    }

    public Object getValue()
    {
        return value;
    }

    /**
     * Method to save a "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @param val The raw value
     */
    public void setRawValue(Object val)
    { 
        // Dont support raw value for binary literals
    }

    /**
     * Accessor for the "raw" value that this literal represents.
     * This value differs from the literal value since that is of the same type as this literal.
     * @return The raw value
     */
    public Object getRawValue()
    {
        return null; // Dont support raw value for binary literals
    }
}