/**********************************************************************
Copyright (c) 2006 Andy Jefferson and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.fieldmanager;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.StateManager;
import org.datanucleus.api.ApiAdapter;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.metadata.Relation;
import org.datanucleus.sco.SCO;

/**
 * Field manager that perists all unpersisted PC objects referenced from the source object.
 * If any collection/map fields are not currently using SCO wrappers they will be converted to do so.
 * Effectively provides "persistence-by-reachability" (at insert/update).
 */
public class PersistFieldManager extends AbstractFieldManager
{
    /** StateManager for the owning object. */
    private final StateManager sm;

    /** Whether this manager will replace any SCO fields with SCO wrappers. */
    private final boolean replaceSCOsWithWrappers;

    /**
     * Constructor.
     * @param sm The state manager for the object.
     * @param replaceSCOsWithWrappers Whether to swap any SCO field objects for SCO wrappers
     **/
    public PersistFieldManager(StateManager sm, boolean replaceSCOsWithWrappers)
    {
        this.sm = sm;
        this.replaceSCOsWithWrappers = replaceSCOsWithWrappers;
    }

    /**
     * Utility method to process the passed persistable object.
     * @param pc The PC object
     * @param ownerFieldNum Field number of owner where this is embedded
     * @param objectType Type of object (see org.datanucleus.StateManager)
     */
    protected void processPersistable(Object pc, int ownerFieldNum, int objectType)
    {
        // TODO Consider adding more of the functionality in SCOUtils.validateObjectForWriting
        ApiAdapter adapter = sm.getObjectManager().getApiAdapter();
        if (!adapter.isPersistent(pc) ||
            (adapter.isPersistent(pc) && adapter.isDeleted(pc)))
        {
            // Object is TRANSIENT/DETACHED and being persisted, or P_NEW_DELETED and being re-persisted
            if (objectType != StateManager.PC)
            {
                sm.getObjectManager().persistObjectInternal(pc, null, sm, ownerFieldNum, objectType);
            }
            else
            {
                sm.getObjectManager().persistObjectInternal(pc, null, null, -1, objectType);
            }
        }
    }

    /**
     * Method to store an object field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeObjectField(int fieldNumber, Object value)
    {
        if (value != null)
        {
            AbstractMemberMetaData mmd = sm.getClassMetaData().getMetaDataForManagedMemberAtAbsolutePosition(fieldNumber);
            boolean persistCascade = mmd.isCascadePersist();
            ApiAdapter api = sm.getObjectManager().getApiAdapter();
            ClassLoaderResolver clr = sm.getObjectManager().getClassLoaderResolver();
            int relationType = mmd.getRelationType(clr);

            if (replaceSCOsWithWrappers)
            {
                // Replace any SCO field that isn't already a wrapper, with its wrapper object
                boolean[] secondClassMutableFieldFlags = sm.getClassMetaData().getSCOMutableMemberFlags();
                if (secondClassMutableFieldFlags[fieldNumber] && !(value instanceof SCO))
                {
                    // Replace the field with a SCO wrapper
                    sm.wrapSCOField(fieldNumber, value, false, true, true);
                }
            }

            if (persistCascade)
            {
                switch (relationType)
                {
                    case Relation.ONE_TO_ONE_BI:
                    case Relation.ONE_TO_ONE_UNI:
                    case Relation.MANY_TO_ONE_BI:
                        if (api.isPersistable(value))
                        {
                            // Process PC fields
                            if (mmd.isEmbedded() || mmd.isSerialized())
                            {
                                processPersistable(value, fieldNumber, StateManager.EMBEDDED_PC);
                            }
                            else
                            {
                                processPersistable(value, -1, StateManager.PC);
                            }
                        }
                        break;

                    case Relation.ONE_TO_MANY_BI:
                    case Relation.ONE_TO_MANY_UNI:
                    case Relation.MANY_TO_MANY_BI:
                        if (mmd.hasCollection())
                        {
                            // Process all elements of the Collection that are PC
                            Collection coll = (Collection)value;
                            Iterator iter = coll.iterator();
                            while (iter.hasNext())
                            {
                                Object element = iter.next();
                                if (api.isPersistable(element))
                                {
                                    if (mmd.getCollection().isEmbeddedElement() || mmd.getCollection().isSerializedElement())
                                    {
                                        processPersistable(element, fieldNumber, StateManager.EMBEDDED_COLLECTION_ELEMENT_PC);
                                    }
                                    else
                                    {
                                        processPersistable(element, -1, StateManager.PC);
                                    }
                                }
                            }
                        }
                        else if (mmd.hasMap())
                        {
                            // Process all keys, values of the Map that are PC
                            Map map = (Map)value;

                            Set keys = map.keySet();
                            Iterator iter = keys.iterator();
                            while (iter.hasNext())
                            {
                                Object mapKey = iter.next();
                                if (api.isPersistable(mapKey))
                                {
                                    if (mmd.getMap().isEmbeddedKey() || mmd.getMap().isSerializedKey())
                                    {
                                        processPersistable(mapKey, fieldNumber, StateManager.EMBEDDED_MAP_KEY_PC);
                                    }
                                    else
                                    {
                                        processPersistable(mapKey, -1, StateManager.PC);
                                    }
                                }
                            }

                            Collection values = map.values();
                            iter = values.iterator();
                            while (iter.hasNext())
                            {
                                Object mapValue = iter.next();
                                if (api.isPersistable(mapValue))
                                {
                                    if (mmd.getMap().isEmbeddedValue() || mmd.getMap().isSerializedValue())
                                    {
                                        processPersistable(mapValue, fieldNumber, StateManager.EMBEDDED_MAP_VALUE_PC);
                                    }
                                    else
                                    {
                                        processPersistable(mapValue, -1, StateManager.PC);
                                    }
                                }
                            }
                        }
                        else if (mmd.hasArray())
                        {
                            if (value instanceof Object[])
                            {
                                Object[] array = (Object[]) value;
                                for (int i=0;i<array.length;i++)
                                {
                                    Object element = array[i];
                                    if (api.isPersistable(element))
                                    {
                                        if (mmd.getArray().isEmbeddedElement() || mmd.getArray().isSerializedElement())
                                        {
                                            // TODO This should be ARRAY_ELEMENT_PC but we haven't got that yet
                                            processPersistable(element, fieldNumber, StateManager.EMBEDDED_COLLECTION_ELEMENT_PC);
                                        }
                                        else
                                        {
                                            processPersistable(element, -1, StateManager.PC);
                                        }
                                    }
                                }
                            }
                            else
                            {
                                // primitive array
                            }
                        }
                        break;

                    default :
                        break;
                }
            }
        }
    }

    /**
     * Method to store a boolean field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeBooleanField(int fieldNumber, boolean value)
    {
        // Do nothing
    }

    /**
     * Method to store a byte field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeByteField(int fieldNumber, byte value)
    {
        // Do nothing
    }

    /**
     * Method to store a char field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeCharField(int fieldNumber, char value)
    {
        // Do nothing
    }

    /**
     * Method to store a double field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeDoubleField(int fieldNumber, double value)
    {
        // Do nothing
    }

    /**
     * Method to store a float field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeFloatField(int fieldNumber, float value)
    {
        // Do nothing
    }

    /**
     * Method to store an int field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeIntField(int fieldNumber, int value)
    {
        // Do nothing
    }

    /**
     * Method to store a long field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeLongField(int fieldNumber, long value)
    {
        // Do nothing
    }

    /**
     * Method to store a short field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeShortField(int fieldNumber, short value)
    {
        // Do nothing
    }

    /**
     * Method to store a string field.
     * @param fieldNumber Number of the field (absolute)
     * @param value Value of the field
     */
    public void storeStringField(int fieldNumber, String value)
    {
        // Do nothing
    }
}