/**********************************************************************
Copyright (c) 2007 Thomas Marti and others. All rights reserved.
Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

Contributors:
    ...
**********************************************************************/
package org.datanucleus.store.mapped.mapping;

import java.awt.geom.RoundRectangle2D;

import org.datanucleus.ClassLoaderResolver;
import org.datanucleus.ClassNameConstants;
import org.datanucleus.ObjectManager;
import org.datanucleus.metadata.AbstractMemberMetaData;
import org.datanucleus.store.mapped.DatastoreAdapter;
import org.datanucleus.store.mapped.DatastoreContainerObject;

/**
 * Mapping for java.awt.geom.RoundRectangle2D.Float, maps the x, y, width, height, 
 * arc-width and arc-height values to float-precision datastore fields.
 */
public class RoundRectangle2dFloatMapping extends SingleFieldMultiMapping
{
    private static final RoundRectangle2D.Float sampleValue = new RoundRectangle2D.Float(0, 0, 1, 1, 1, 1);

	/* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#initialize()
     */
    public void initialize(DatastoreAdapter dba, AbstractMemberMetaData fmd, DatastoreContainerObject container, ClassLoaderResolver clr)
    {
		super.initialize(dba, fmd, container, clr);

        addDatastoreField(ClassNameConstants.FLOAT); // X
        addDatastoreField(ClassNameConstants.FLOAT); // Y
        addDatastoreField(ClassNameConstants.FLOAT); // Width
        addDatastoreField(ClassNameConstants.FLOAT); // Height
        addDatastoreField(ClassNameConstants.FLOAT); // Arc-Width
        addDatastoreField(ClassNameConstants.FLOAT); // Arc-Height
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getJavaType()
     */
    public Class getJavaType()
    {
        return RoundRectangle2D.Float.class;
    }

    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getSampleValue()
     */
    public Object getSampleValue(ClassLoaderResolver clr)
    {
        return sampleValue;
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#setObject(org.datanucleus.ObjectManager, java.lang.Object, int[], java.lang.Object)
     */
    public void setObject(ObjectManager om, Object preparedStatement, int[] exprIndex, Object value)
    {
    	RoundRectangle2D.Float roundRectangle = (RoundRectangle2D.Float)value;
        if (roundRectangle == null)
        {
    		for (int i = 0; i < exprIndex.length; i++) 
    		{
    			getDataStoreMapping(i).setObject(preparedStatement, exprIndex[i], null);					
			}
        }
        else
        {
            getDataStoreMapping(0).setFloat(preparedStatement,exprIndex[0],roundRectangle.x);
            getDataStoreMapping(1).setFloat(preparedStatement,exprIndex[1],roundRectangle.y);
            getDataStoreMapping(2).setFloat(preparedStatement,exprIndex[2],roundRectangle.width);
            getDataStoreMapping(3).setFloat(preparedStatement,exprIndex[3],roundRectangle.height);
            getDataStoreMapping(4).setFloat(preparedStatement,exprIndex[4],roundRectangle.arcwidth);
            getDataStoreMapping(5).setFloat(preparedStatement,exprIndex[5],roundRectangle.archeight);
        }
    }
    
    /* (non-Javadoc)
     * @see org.datanucleus.store.mapping.JavaTypeMapping#getObject(org.datanucleus.ObjectManager, java.lang.Object, int[])
     */
    public Object getObject(ObjectManager om, Object resultSet, int[] exprIndex)
    {
        // Check for null entries
        if (getDataStoreMapping(0).getObject(resultSet, exprIndex[0]) == null)
        {
            return null;
        }

        float x = getDataStoreMapping(0).getFloat(resultSet,exprIndex[0]);
        float y = getDataStoreMapping(1).getFloat(resultSet,exprIndex[1]);
        float width  = getDataStoreMapping(2).getFloat(resultSet,exprIndex[2]);
        float height = getDataStoreMapping(3).getFloat(resultSet,exprIndex[3]);
        float arcwidth  = getDataStoreMapping(4).getFloat(resultSet,exprIndex[4]);
        float archeight = getDataStoreMapping(5).getFloat(resultSet,exprIndex[5]);
        return new RoundRectangle2D.Float(x, y, width, height, arcwidth, archeight);
    }
}