/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner;

import com.google.cloud.Timestamp;
import com.google.cloud.grpc.GrpcTransportOptions;
import com.google.cloud.spanner.DatabaseId;
import com.google.cloud.spanner.ErrorCode;
import com.google.cloud.spanner.ForwardingResultSet;
import com.google.cloud.spanner.Key;
import com.google.cloud.spanner.KeySet;
import com.google.cloud.spanner.Mutation;
import com.google.cloud.spanner.Options;
import com.google.cloud.spanner.ReadContext;
import com.google.cloud.spanner.ReadOnlyTransaction;
import com.google.cloud.spanner.ResultSet;
import com.google.cloud.spanner.Session;
import com.google.cloud.spanner.SessionPoolOptions;
import com.google.cloud.spanner.SpannerException;
import com.google.cloud.spanner.SpannerExceptionFactory;
import com.google.cloud.spanner.SpannerImpl;
import com.google.cloud.spanner.SpannerOptions;
import com.google.cloud.spanner.Statement;
import com.google.cloud.spanner.Struct;
import com.google.cloud.spanner.TimestampBound;
import com.google.cloud.spanner.TransactionContext;
import com.google.cloud.spanner.TransactionManager;
import com.google.cloud.spanner.TransactionRunner;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;
import com.google.common.util.concurrent.SettableFuture;
import com.google.common.util.concurrent.Uninterruptibles;
import io.opencensus.trace.Annotation;
import io.opencensus.trace.AttributeValue;
import io.opencensus.trace.Span;
import io.opencensus.trace.Tracing;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import org.threeten.bp.Duration;
import org.threeten.bp.Instant;
import org.threeten.bp.temporal.TemporalAmount;

final class SessionPool {
    private static final Logger logger = Logger.getLogger(SessionPool.class.getName());
    private final SessionPoolOptions options;
    private final DatabaseId db;
    private final SpannerImpl spanner;
    private final ScheduledExecutorService executor;
    private final GrpcTransportOptions.ExecutorFactory<ScheduledExecutorService> executorFactory;
    final PoolMaintainer poolMaintainer;
    private final Clock clock;
    private final Object lock = new Object();
    @GuardedBy(value="lock")
    private int pendingClosure;
    @GuardedBy(value="lock")
    private SettableFuture<Void> closureFuture;
    @GuardedBy(value="lock")
    private final Queue<PooledSession> readSessions = new LinkedList<PooledSession>();
    @GuardedBy(value="lock")
    private final Queue<PooledSession> writePreparedSessions = new LinkedList<PooledSession>();
    @GuardedBy(value="lock")
    private final Queue<Waiter> readWaiters = new LinkedList<Waiter>();
    @GuardedBy(value="lock")
    private final Queue<Waiter> readWriteWaiters = new LinkedList<Waiter>();
    @GuardedBy(value="lock")
    private int numSessionsBeingPrepared = 0;
    @GuardedBy(value="lock")
    private int numSessionsBeingCreated = 0;
    @GuardedBy(value="lock")
    private int numSessionsInUse = 0;
    @GuardedBy(value="lock")
    private int maxSessionsInUse = 0;
    @GuardedBy(value="lock")
    private final Set<PooledSession> allSessions = new HashSet<PooledSession>();

    static SessionPool createPool(SpannerOptions spannerOptions, DatabaseId db, SpannerImpl spanner) {
        return SessionPool.createPool(spannerOptions.getSessionPoolOptions(), (GrpcTransportOptions.ExecutorFactory<ScheduledExecutorService>)((GrpcTransportOptions)spannerOptions.getTransportOptions()).getExecutorFactory(), db, spanner);
    }

    static SessionPool createPool(SessionPoolOptions poolOptions, GrpcTransportOptions.ExecutorFactory<ScheduledExecutorService> executorFactory, DatabaseId db, SpannerImpl spanner) {
        return SessionPool.createPool(poolOptions, executorFactory, db, spanner, new Clock());
    }

    static SessionPool createPool(SessionPoolOptions poolOptions, GrpcTransportOptions.ExecutorFactory<ScheduledExecutorService> executorFactory, DatabaseId db, SpannerImpl spanner, Clock clock) {
        SessionPool pool = new SessionPool(poolOptions, executorFactory, (ScheduledExecutorService)executorFactory.get(), db, spanner, clock);
        pool.initPool();
        return pool;
    }

    private SessionPool(SessionPoolOptions options, GrpcTransportOptions.ExecutorFactory<ScheduledExecutorService> executorFactory, ScheduledExecutorService executor, DatabaseId db, SpannerImpl spanner, Clock clock) {
        this.options = options;
        this.executorFactory = executorFactory;
        this.executor = executor;
        this.db = db;
        this.spanner = spanner;
        this.clock = clock;
        this.poolMaintainer = new PoolMaintainer();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void initPool() {
        Object object = this.lock;
        synchronized (object) {
            this.poolMaintainer.init();
            for (int i = 0; i < this.options.getMinSessions(); ++i) {
                this.createSession();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean isClosed() {
        Object object = this.lock;
        synchronized (object) {
            return this.closureFuture != null;
        }
    }

    private void handleException(SpannerException e, PooledSession session) {
        if (this.isSessionNotFound(e)) {
            this.invalidateSession(session);
        } else {
            this.releaseSession(session);
        }
    }

    private boolean isSessionNotFound(SpannerException e) {
        return e.getErrorCode() == ErrorCode.NOT_FOUND && e.getMessage().contains("Session not found");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void invalidateSession(PooledSession session) {
        Object object = this.lock;
        synchronized (object) {
            if (this.isClosed()) {
                return;
            }
            this.allSessions.remove(session);
            this.createSession();
        }
    }

    private PooledSession findSessionToKeepAlive(Queue<PooledSession> queue, Instant keepAliveThreshold) {
        Iterator iterator = queue.iterator();
        while (iterator.hasNext()) {
            PooledSession session = (PooledSession)iterator.next();
            if (!session.lastUseTime.isBefore(keepAliveThreshold)) continue;
            iterator.remove();
            return session;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Session getReadSession() throws SpannerException {
        Span span = Tracing.getTracer().getCurrentSpan();
        span.addAnnotation("Acquiring session");
        Waiter waiter = null;
        PooledSession sess = null;
        Object object = this.lock;
        synchronized (object) {
            if (this.closureFuture != null) {
                span.addAnnotation("Pool has been closed");
                throw new IllegalStateException("Pool has been closed");
            }
            sess = this.readSessions.poll();
            if (sess == null) {
                sess = this.writePreparedSessions.poll();
                if (sess == null) {
                    span.addAnnotation("No session available");
                    this.maybeCreateSession();
                    waiter = new Waiter();
                    this.readWaiters.add(waiter);
                } else {
                    span.addAnnotation("Acquired read write session");
                }
            } else {
                span.addAnnotation("Acquired read only session");
            }
        }
        if (waiter != null) {
            logger.log(Level.FINE, "No session available in the pool. Blocking for one to become available/created");
            span.addAnnotation("Waiting for read only session to be available");
            sess = waiter.take();
        }
        sess.markBusy();
        this.incrementNumSessionsInUse();
        span.addAnnotation(this.sessionAnnotation(sess));
        return sess;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    Session getReadWriteSession() {
        Span span = Tracing.getTracer().getCurrentSpan();
        span.addAnnotation("Acquiring read write session");
        Waiter waiter = null;
        PooledSession sess = null;
        Object object = this.lock;
        synchronized (object) {
            if (this.closureFuture != null) {
                throw new IllegalStateException("Pool has been closed");
            }
            sess = this.writePreparedSessions.poll();
            if (sess == null) {
                if (this.numSessionsBeingPrepared <= this.readWriteWaiters.size()) {
                    PooledSession readSession = this.readSessions.poll();
                    if (readSession != null) {
                        span.addAnnotation("Acquired read only session. Preparing for read write transaction");
                        this.prepareSession(readSession);
                    } else {
                        span.addAnnotation("No session available");
                        this.maybeCreateSession();
                    }
                }
                waiter = new Waiter();
                this.readWriteWaiters.add(waiter);
            } else {
                span.addAnnotation("Acquired read write session");
            }
        }
        if (waiter != null) {
            logger.log(Level.FINE, "No session available in the pool. Blocking for one to become available/created");
            span.addAnnotation("Waiting for read write session to be available");
            sess = waiter.take();
        }
        sess.markBusy();
        this.incrementNumSessionsInUse();
        span.addAnnotation(this.sessionAnnotation(sess));
        return sess;
    }

    private Annotation sessionAnnotation(Session session) {
        AttributeValue sessionId = AttributeValue.stringAttributeValue((String)session.getName());
        return Annotation.fromDescriptionAndAttributes((String)"Using Session", (Map)ImmutableMap.of((Object)"sessionId", (Object)sessionId));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void incrementNumSessionsInUse() {
        Object object = this.lock;
        synchronized (object) {
            if (this.maxSessionsInUse < ++this.numSessionsInUse) {
                this.maxSessionsInUse = this.numSessionsInUse;
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void maybeCreateSession() {
        Span span = Tracing.getTracer().getCurrentSpan();
        Object object = this.lock;
        synchronized (object) {
            if (this.numWaiters() >= this.numSessionsBeingCreated) {
                if (this.canCreateSession()) {
                    span.addAnnotation("Creating session");
                    this.createSession();
                } else if (this.options.isFailIfPoolExhausted()) {
                    span.addAnnotation("Pool exhausted. Failing");
                    throw SpannerExceptionFactory.newSpannerException(ErrorCode.RESOURCE_EXHAUSTED, "No session available in the pool. Maximum number of sessions in the pool can be overridden by invoking SessionPoolOptions#Builder#setMaxSessions. Client can be made to block rather than fail by setting SessionPoolOptions#Builder#setBlockIfPoolExhausted.");
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void releaseSession(PooledSession session) {
        Preconditions.checkNotNull((Object)session);
        Object object = this.lock;
        synchronized (object) {
            if (this.closureFuture != null) {
                return;
            }
            if (this.readWaiters.size() == 0 && this.numSessionsBeingPrepared >= this.readWriteWaiters.size()) {
                if (this.shouldPrepareSession()) {
                    this.prepareSession(session);
                } else {
                    this.readSessions.add(session);
                }
            } else if (this.shouldUnblockReader()) {
                this.readWaiters.poll().put(session);
            } else {
                this.prepareSession(session);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handleCreateSessionFailure(SpannerException e) {
        Object object = this.lock;
        synchronized (object) {
            if (this.readWaiters.size() > 0) {
                this.readWaiters.poll().put(e);
            } else if (this.readWriteWaiters.size() > 0) {
                this.readWriteWaiters.poll().put(e);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void handlePrepareSessionFailure(SpannerException e, PooledSession session) {
        Object object = this.lock;
        synchronized (object) {
            if (this.isSessionNotFound(e)) {
                this.invalidateSession(session);
            } else if (this.readWriteWaiters.size() > 0) {
                this.readWriteWaiters.poll().put(e);
            } else {
                this.releaseSession(session);
            }
        }
    }

    private void decrementPendingClosures() {
        --this.pendingClosure;
        if (this.pendingClosure == 0) {
            this.closureFuture.set(null);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    ListenableFuture<Void> closeAsync() {
        SettableFuture<Void> retFuture = null;
        Object object = this.lock;
        synchronized (object) {
            if (this.closureFuture != null) {
                throw new IllegalStateException("Close has already been invoked");
            }
            Waiter waiter = this.readWaiters.poll();
            while (waiter != null) {
                waiter.put(SpannerExceptionFactory.newSpannerException(ErrorCode.INTERNAL, "Client has been closed"));
                waiter = this.readWaiters.poll();
            }
            waiter = this.readWriteWaiters.poll();
            while (waiter != null) {
                waiter.put(SpannerExceptionFactory.newSpannerException(ErrorCode.INTERNAL, "Client has been closed"));
                waiter = this.readWriteWaiters.poll();
            }
            this.closureFuture = SettableFuture.create();
            retFuture = this.closureFuture;
            this.pendingClosure = this.totalSessions() + this.numSessionsBeingCreated + 1;
            this.poolMaintainer.close();
            this.readSessions.clear();
            this.writePreparedSessions.clear();
            for (PooledSession session : ImmutableList.copyOf(this.allSessions)) {
                if (session.leakedException != null) {
                    logger.log(Level.WARNING, "Leaked session", session.leakedException);
                }
                if (session.state == SessionState.CLOSING) continue;
                this.closeSessionAsync(session);
            }
        }
        retFuture.addListener(new Runnable(){

            @Override
            public void run() {
                SessionPool.this.executorFactory.release((ExecutorService)SessionPool.this.executor);
            }
        }, MoreExecutors.directExecutor());
        return retFuture;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean shouldUnblockReader() {
        Object object = this.lock;
        synchronized (object) {
            int numWriteWaiters = this.readWriteWaiters.size() - this.numSessionsBeingPrepared;
            return this.readWaiters.size() > numWriteWaiters;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean shouldPrepareSession() {
        Object object = this.lock;
        synchronized (object) {
            int preparedSessions = this.writePreparedSessions.size() + this.numSessionsBeingPrepared;
            return (double)preparedSessions < Math.floor(this.options.getWriteSessionsFraction() * (float)this.totalSessions());
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int numWaiters() {
        Object object = this.lock;
        synchronized (object) {
            return this.readWaiters.size() + this.readWriteWaiters.size();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private int totalSessions() {
        Object object = this.lock;
        synchronized (object) {
            return this.allSessions.size();
        }
    }

    private void closeSessionAsync(final PooledSession sess) {
        this.executor.submit(new Runnable(){

            @Override
            public void run() {
                SessionPool.this.closeSession(sess);
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeSession(PooledSession sess) {
        try {
            sess.delegate.close();
        }
        catch (SpannerException e) {
            if (logger.isLoggable(Level.FINE)) {
                logger.log(Level.FINE, "Failed to close session: " + sess.getName(), (Throwable)((Object)e));
            }
        }
        finally {
            Object object = this.lock;
            synchronized (object) {
                this.allSessions.remove(sess);
                if (this.isClosed()) {
                    this.decrementPendingClosures();
                    return;
                }
                if (this.numWaiters() > this.numSessionsBeingCreated) {
                    this.createSession();
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void prepareSession(final PooledSession sess) {
        Object object = this.lock;
        synchronized (object) {
            ++this.numSessionsBeingPrepared;
        }
        this.executor.submit(new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                try {
                    logger.log(Level.FINE, "Preparing session");
                    sess.prepareReadWriteTransaction();
                    logger.log(Level.FINE, "Session prepared");
                    Object object = SessionPool.this.lock;
                    synchronized (object) {
                        SessionPool.this.numSessionsBeingPrepared--;
                        if (!SessionPool.this.isClosed()) {
                            if (SessionPool.this.readWriteWaiters.size() > 0) {
                                ((Waiter)SessionPool.this.readWriteWaiters.poll()).put(sess);
                            } else if (SessionPool.this.readWaiters.size() > 0) {
                                ((Waiter)SessionPool.this.readWaiters.poll()).put(sess);
                            } else {
                                SessionPool.this.writePreparedSessions.add(sess);
                            }
                        }
                    }
                }
                catch (Throwable t) {
                    Object object = SessionPool.this.lock;
                    synchronized (object) {
                        SessionPool.this.numSessionsBeingPrepared--;
                        if (!SessionPool.this.isClosed()) {
                            SessionPool.this.handlePrepareSessionFailure(SpannerExceptionFactory.newSpannerException(t), sess);
                        }
                    }
                }
            }
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean canCreateSession() {
        Object object = this.lock;
        synchronized (object) {
            return this.totalSessions() + this.numSessionsBeingCreated < this.options.getMaxSessions();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void createSession() {
        logger.log(Level.FINE, "Creating session");
        Object object = this.lock;
        synchronized (object) {
            ++this.numSessionsBeingCreated;
            this.executor.submit(new Runnable(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void run() {
                    Session session = null;
                    try {
                        session = SessionPool.this.spanner.createSession(SessionPool.this.db);
                        logger.log(Level.FINE, "Session created");
                    }
                    catch (Throwable t) {
                        Object object = SessionPool.this.lock;
                        synchronized (object) {
                            SessionPool.this.numSessionsBeingCreated--;
                            if (SessionPool.this.isClosed()) {
                                SessionPool.this.decrementPendingClosures();
                            }
                            SessionPool.this.handleCreateSessionFailure(SpannerExceptionFactory.newSpannerException(t));
                        }
                        return;
                    }
                    boolean closeSession = false;
                    PooledSession pooledSession = null;
                    Object object = SessionPool.this.lock;
                    synchronized (object) {
                        pooledSession = new PooledSession(session);
                        SessionPool.this.numSessionsBeingCreated--;
                        if (SessionPool.this.closureFuture != null) {
                            closeSession = true;
                        } else {
                            Preconditions.checkState((SessionPool.this.totalSessions() <= SessionPool.this.options.getMaxSessions() - 1 ? 1 : 0) != 0);
                            SessionPool.this.allSessions.add(pooledSession);
                            SessionPool.this.releaseSession(pooledSession);
                        }
                    }
                    if (closeSession) {
                        SessionPool.this.closeSession(pooledSession);
                    }
                }
            });
        }
    }

    final class PoolMaintainer {
        private final Duration windowLength = Duration.ofMillis((long)TimeUnit.MINUTES.toMillis(10L));
        @VisibleForTesting
        static final long LOOP_FREQUENCY = 10000L;
        @VisibleForTesting
        final long numClosureCycles = this.windowLength.toMillis() / 10000L;
        private final Duration keepAliveMilis = Duration.ofMillis((long)TimeUnit.MINUTES.toMillis(SessionPool.access$1400(SessionPool.this).getKeepAliveIntervalMinutes()));
        @VisibleForTesting
        final long numKeepAliveCycles = this.keepAliveMilis.toMillis() / 10000L;
        Instant lastResetTime = Instant.ofEpochMilli((long)0L);
        int numSessionsToClose = 0;
        int sessionsToClosePerLoop = 0;
        @GuardedBy(value="lock")
        ScheduledFuture<?> scheduledFuture;
        @GuardedBy(value="lock")
        boolean running;

        PoolMaintainer() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void init() {
            Object object = SessionPool.this.lock;
            synchronized (object) {
                this.scheduledFuture = SessionPool.this.executor.scheduleAtFixedRate(new Runnable(){

                    @Override
                    public void run() {
                        PoolMaintainer.this.maintainPool();
                    }
                }, 10000L, 10000L, TimeUnit.MILLISECONDS);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void close() {
            Object object = SessionPool.this.lock;
            synchronized (object) {
                this.scheduledFuture.cancel(false);
                if (!this.running) {
                    SessionPool.this.decrementPendingClosures();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void maintainPool() {
            Object object = SessionPool.this.lock;
            synchronized (object) {
                if (SessionPool.this.isClosed()) {
                    return;
                }
                this.running = true;
            }
            Instant currTime = SessionPool.this.clock.instant();
            this.closeIdleSessions(currTime);
            this.keepAliveSessions(currTime);
            this.replenishPool();
            Object object2 = SessionPool.this.lock;
            synchronized (object2) {
                this.running = false;
                if (SessionPool.this.isClosed()) {
                    SessionPool.this.decrementPendingClosures();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void closeIdleSessions(Instant currTime) {
            LinkedList<PooledSession> sessionsToClose = new LinkedList<PooledSession>();
            Iterator iterator = SessionPool.this.lock;
            synchronized (iterator) {
                if (currTime.isAfter(this.lastResetTime.plus((TemporalAmount)this.windowLength))) {
                    int sessionsToKeep = Math.max(SessionPool.this.options.getMinSessions(), SessionPool.this.maxSessionsInUse + SessionPool.this.options.getMaxIdleSessions());
                    this.numSessionsToClose = SessionPool.this.totalSessions() - sessionsToKeep;
                    this.sessionsToClosePerLoop = (int)Math.ceil((double)this.numSessionsToClose / (double)this.numClosureCycles);
                    SessionPool.this.maxSessionsInUse = 0;
                    this.lastResetTime = currTime;
                }
                if (this.numSessionsToClose > 0) {
                    while (sessionsToClose.size() < Math.min(this.numSessionsToClose, this.sessionsToClosePerLoop)) {
                        PooledSession sess;
                        PooledSession pooledSession = sess = SessionPool.this.readSessions.size() > 0 ? (PooledSession)SessionPool.this.readSessions.poll() : (PooledSession)SessionPool.this.writePreparedSessions.poll();
                        if (sess == null) break;
                        if (sess.state == SessionState.CLOSING) continue;
                        sess.markClosing();
                        sessionsToClose.add(sess);
                    }
                    this.numSessionsToClose -= sessionsToClose.size();
                }
            }
            for (PooledSession sess : sessionsToClose) {
                logger.log(Level.FINE, "Closing session {0}", sess.getName());
                SessionPool.this.closeSession(sess);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void keepAliveSessions(Instant currTime) {
            long numSessionsToKeepAlive = 0L;
            Object object = SessionPool.this.lock;
            synchronized (object) {
                numSessionsToKeepAlive = (long)Math.ceil((double)SessionPool.this.totalSessions() / (double)this.numKeepAliveCycles);
            }
            Instant keepAliveThreshold = currTime.minus((TemporalAmount)this.keepAliveMilis);
            while (numSessionsToKeepAlive > 0L) {
                PooledSession sessionToKeepAlive = null;
                Object object2 = SessionPool.this.lock;
                synchronized (object2) {
                    sessionToKeepAlive = SessionPool.this.findSessionToKeepAlive(SessionPool.this.readSessions, keepAliveThreshold);
                    if (sessionToKeepAlive == null) {
                        sessionToKeepAlive = SessionPool.this.findSessionToKeepAlive(SessionPool.this.writePreparedSessions, keepAliveThreshold);
                    }
                }
                if (sessionToKeepAlive == null) break;
                try {
                    logger.log(Level.FINE, "Keeping alive session " + sessionToKeepAlive.getName());
                    --numSessionsToKeepAlive;
                    sessionToKeepAlive.keepAlive();
                    SessionPool.this.releaseSession(sessionToKeepAlive);
                }
                catch (SpannerException e) {
                    SessionPool.this.handleException(e, sessionToKeepAlive);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void replenishPool() {
            Object object = SessionPool.this.lock;
            synchronized (object) {
                for (int i = 0; i < SessionPool.this.options.getMinSessions() - (SessionPool.this.totalSessions() + SessionPool.this.numSessionsBeingCreated); ++i) {
                    SessionPool.this.createSession();
                }
            }
        }
    }

    private static final class Waiter {
        private final SynchronousQueue<SessionOrError> waiter = new SynchronousQueue();

        private Waiter() {
        }

        private void put(PooledSession session) {
            Uninterruptibles.putUninterruptibly(this.waiter, (Object)new SessionOrError(session));
        }

        private void put(SpannerException e) {
            Uninterruptibles.putUninterruptibly(this.waiter, (Object)new SessionOrError(e));
        }

        private PooledSession take() throws SpannerException {
            SessionOrError s = (SessionOrError)Uninterruptibles.takeUninterruptibly(this.waiter);
            if (s.e != null) {
                throw SpannerExceptionFactory.newSpannerException((Throwable)((Object)s.e));
            }
            return s.session;
        }
    }

    private static final class SessionOrError {
        private final PooledSession session;
        private final SpannerException e;

        SessionOrError(PooledSession session) {
            this.session = session;
            this.e = null;
        }

        SessionOrError(SpannerException e) {
            this.session = null;
            this.e = e;
        }
    }

    final class PooledSession
    implements Session {
        @VisibleForTesting
        final Session delegate;
        private volatile Instant lastUseTime;
        private volatile SpannerException lastException;
        private volatile LeakedSessionException leakedException;
        @GuardedBy(value="lock")
        private SessionState state;

        private PooledSession(Session delegate) {
            this.delegate = delegate;
            this.state = SessionState.AVAILABLE;
            this.markUsed();
        }

        private void markBusy() {
            this.state = SessionState.BUSY;
            this.leakedException = new LeakedSessionException();
        }

        private void markClosing() {
            this.state = SessionState.CLOSING;
        }

        @Override
        public Timestamp write(Iterable<Mutation> mutations) throws SpannerException {
            try {
                this.markUsed();
                Timestamp timestamp = this.delegate.write(mutations);
                return timestamp;
            }
            catch (SpannerException e) {
                this.lastException = e;
                throw this.lastException;
            }
            finally {
                this.close();
            }
        }

        @Override
        public long executePartitionedUpdate(Statement stmt) throws SpannerException {
            try {
                this.markUsed();
                long l = this.delegate.executePartitionedUpdate(stmt);
                return l;
            }
            catch (SpannerException e) {
                this.lastException = e;
                throw this.lastException;
            }
            finally {
                this.close();
            }
        }

        @Override
        public Timestamp writeAtLeastOnce(Iterable<Mutation> mutations) throws SpannerException {
            try {
                this.markUsed();
                Timestamp timestamp = this.delegate.writeAtLeastOnce(mutations);
                return timestamp;
            }
            catch (SpannerException e) {
                this.lastException = e;
                throw this.lastException;
            }
            finally {
                this.close();
            }
        }

        @Override
        public ReadContext singleUse() {
            try {
                return new AutoClosingReadContext(this.delegate.singleUse(), this, true);
            }
            catch (Exception e) {
                this.close();
                throw e;
            }
        }

        @Override
        public ReadContext singleUse(TimestampBound bound) {
            try {
                return new AutoClosingReadContext(this.delegate.singleUse(bound), this, true);
            }
            catch (Exception e) {
                this.close();
                throw e;
            }
        }

        @Override
        public ReadOnlyTransaction singleUseReadOnlyTransaction() {
            try {
                return new AutoClosingReadTransaction(this.delegate.singleUseReadOnlyTransaction(), this, true);
            }
            catch (Exception e) {
                this.close();
                throw e;
            }
        }

        @Override
        public ReadOnlyTransaction singleUseReadOnlyTransaction(TimestampBound bound) {
            try {
                return new AutoClosingReadTransaction(this.delegate.singleUseReadOnlyTransaction(bound), this, true);
            }
            catch (Exception e) {
                this.close();
                throw e;
            }
        }

        @Override
        public ReadOnlyTransaction readOnlyTransaction() {
            try {
                return new AutoClosingReadTransaction(this.delegate.readOnlyTransaction(), this, false);
            }
            catch (Exception e) {
                this.close();
                throw e;
            }
        }

        @Override
        public ReadOnlyTransaction readOnlyTransaction(TimestampBound bound) {
            try {
                return new AutoClosingReadTransaction(this.delegate.readOnlyTransaction(bound), this, false);
            }
            catch (Exception e) {
                this.close();
                throw e;
            }
        }

        @Override
        public TransactionRunner readWriteTransaction() {
            final TransactionRunner runner = this.delegate.readWriteTransaction();
            return new TransactionRunner(){

                @Override
                @Nullable
                public <T> T run(TransactionRunner.TransactionCallable<T> callable) {
                    try {
                        T result;
                        PooledSession.this.markUsed();
                        T t = result = runner.run(callable);
                        return t;
                    }
                    catch (SpannerException e) {
                        throw PooledSession.this.lastException = e;
                    }
                    finally {
                        PooledSession.this.close();
                    }
                }

                @Override
                public Timestamp getCommitTimestamp() {
                    return runner.getCommitTimestamp();
                }

                @Override
                public TransactionRunner allowNestedTransaction() {
                    runner.allowNestedTransaction();
                    return runner;
                }
            };
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void close() {
            Object object = SessionPool.this.lock;
            synchronized (object) {
                SessionPool.this.numSessionsInUse--;
            }
            this.leakedException = null;
            if (this.lastException != null && SessionPool.this.isSessionNotFound(this.lastException)) {
                SessionPool.this.invalidateSession(this);
            } else {
                this.lastException = null;
                if (this.state != SessionState.CLOSING) {
                    this.state = SessionState.AVAILABLE;
                }
                SessionPool.this.releaseSession(this);
            }
        }

        @Override
        public String getName() {
            return this.delegate.getName();
        }

        @Override
        public void prepareReadWriteTransaction() {
            this.markUsed();
            this.delegate.prepareReadWriteTransaction();
        }

        private void keepAlive() {
            this.markUsed();
            this.delegate.singleUse(TimestampBound.ofMaxStaleness(60L, TimeUnit.SECONDS)).executeQuery(Statement.newBuilder("SELECT 1").build(), new Options.QueryOption[0]).next();
        }

        private void markUsed() {
            this.lastUseTime = SessionPool.this.clock.instant();
        }

        @Override
        public TransactionManager transactionManager() {
            this.markUsed();
            return new AutoClosingTransactionManager(this.delegate.transactionManager(), this);
        }
    }

    private static enum SessionState {
        AVAILABLE,
        BUSY,
        CLOSING;

    }

    private final class LeakedSessionException
    extends RuntimeException {
        private static final long serialVersionUID = 1451131180314064914L;

        private LeakedSessionException() {
            super("Session was checked out from the pool at " + SessionPool.this.clock.instant());
        }
    }

    private static class AutoClosingTransactionManager
    implements TransactionManager {
        final TransactionManager delegate;
        final PooledSession session;
        private boolean closed;

        AutoClosingTransactionManager(TransactionManager delegate, PooledSession session) {
            this.delegate = delegate;
            this.session = session;
        }

        @Override
        public TransactionContext begin() {
            return this.delegate.begin();
        }

        @Override
        public void commit() {
            try {
                this.delegate.commit();
            }
            finally {
                if (this.getState() != TransactionManager.TransactionState.ABORTED) {
                    this.close();
                }
            }
        }

        @Override
        public void rollback() {
            try {
                this.delegate.rollback();
            }
            finally {
                this.close();
            }
        }

        @Override
        public TransactionContext resetForRetry() {
            return this.delegate.resetForRetry();
        }

        @Override
        public Timestamp getCommitTimestamp() {
            return this.delegate.getCommitTimestamp();
        }

        @Override
        public void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            try {
                this.delegate.close();
            }
            finally {
                this.session.close();
            }
        }

        @Override
        public TransactionManager.TransactionState getState() {
            return this.delegate.getState();
        }
    }

    private static class AutoClosingReadTransaction
    extends AutoClosingReadContext
    implements ReadOnlyTransaction {
        private final ReadOnlyTransaction txn;

        AutoClosingReadTransaction(ReadOnlyTransaction txn, PooledSession session, boolean isSingleUse) {
            super(txn, session, isSingleUse);
            this.txn = txn;
        }

        @Override
        public Timestamp getReadTimestamp() {
            return this.txn.getReadTimestamp();
        }
    }

    private static class AutoClosingReadContext
    implements ReadContext {
        private final ReadContext delegate;
        private final PooledSession session;
        private final boolean isSingleUse;
        private boolean closed;

        private AutoClosingReadContext(ReadContext delegate, PooledSession session, boolean isSingleUse) {
            this.delegate = delegate;
            this.session = session;
            this.isSingleUse = isSingleUse;
        }

        private ResultSet wrap(ResultSet resultSet) {
            this.session.markUsed();
            if (!this.isSingleUse) {
                return resultSet;
            }
            return new ForwardingResultSet(resultSet){

                @Override
                public boolean next() throws SpannerException {
                    try {
                        boolean ret = super.next();
                        if (!ret) {
                            this.close();
                        }
                        return ret;
                    }
                    catch (SpannerException e) {
                        if (!AutoClosingReadContext.this.closed) {
                            AutoClosingReadContext.this.session.lastException = e;
                            AutoClosingReadContext.this.close();
                        }
                        throw e;
                    }
                }

                @Override
                public void close() {
                    super.close();
                    AutoClosingReadContext.this.close();
                }
            };
        }

        @Override
        public ResultSet read(String table, KeySet keys, Iterable<String> columns, Options.ReadOption ... options) {
            return this.wrap(this.delegate.read(table, keys, columns, options));
        }

        @Override
        public ResultSet readUsingIndex(String table, String index, KeySet keys, Iterable<String> columns, Options.ReadOption ... options) {
            return this.wrap(this.delegate.readUsingIndex(table, index, keys, columns, options));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        @Nullable
        public Struct readRow(String table, Key key, Iterable<String> columns) {
            try {
                this.session.markUsed();
                Struct struct = this.delegate.readRow(table, key, columns);
                return struct;
            }
            finally {
                if (this.isSingleUse) {
                    this.close();
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        @Nullable
        public Struct readRowUsingIndex(String table, String index, Key key, Iterable<String> columns) {
            try {
                this.session.markUsed();
                Struct struct = this.delegate.readRowUsingIndex(table, index, key, columns);
                return struct;
            }
            finally {
                if (this.isSingleUse) {
                    this.close();
                }
            }
        }

        @Override
        public ResultSet executeQuery(Statement statement, Options.QueryOption ... options) {
            return this.wrap(this.delegate.executeQuery(statement, options));
        }

        @Override
        public ResultSet analyzeQuery(Statement statement, ReadContext.QueryAnalyzeMode queryMode) {
            return this.wrap(this.delegate.analyzeQuery(statement, queryMode));
        }

        @Override
        public void close() {
            if (this.closed) {
                return;
            }
            this.closed = true;
            this.delegate.close();
            this.session.close();
        }
    }

    static class Clock {
        Clock() {
        }

        Instant instant() {
            return Instant.now();
        }
    }
}

