/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner.connection;

import com.google.cloud.Timestamp;
import com.google.cloud.spanner.connection.ReadOnlyStalenessUtil;
import com.google.cloud.spanner.connection.StatementParser;
import com.google.cloud.spanner.connection.StatementResult;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.regex.Pattern;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.Assert;

public abstract class AbstractSqlScriptVerifier {
    private static final Pattern VERIFY_PATTERN = Pattern.compile("(?is)\\s*(?:@EXPECT)\\s+(?<type>NO_RESULT|RESULT_SET\\s*(?<column>'.*?'(?<value>,.*?)?)?|UPDATE_COUNT\\s*(?<count>-?\\d{1,19})|EXCEPTION\\s*(?<exception>(?<code>CANCELLED|UNKNOWN|INVALID_ARGUMENT|DEADLINE_EXCEEDED|NOT_FOUND|ALREADY_EXISTS|PERMISSION_DENIED|UNAUTHENTICATED|RESOURCE_EXHAUSTED|FAILED_PRECONDITION|ABORTED|OUT_OF_RANGE|UNIMPLEMENTED|INTERNAL|UNAVAILABLE|DATA_LOSS)(?:\\s*)(?<messagePrefix>'.*?')?)|EQUAL\\s+(?<variable1>'.+?')\\s*,\\s*(?<variable2>'.+?'))(\\n(?<statement>.*))?");
    private static final String PUT_CONDITION = "@PUT can only be used in combination with a statement that returns a result set containing exactly one row and one column";
    private static final Pattern PUT_PATTERN = Pattern.compile("(?is)\\s*(?:@PUT)\\s+(?<variable>'.*?')\\n(?<statement>.*)");
    private final GenericConnectionProvider connectionProvider;
    private final Map<String, Object> variables = new HashMap<String, Object>();
    private final boolean logStatements;
    private static final Pattern INT64_PATTERN = Pattern.compile("\\d{1,19}");
    private static final Pattern ARRAY_INT64_PATTERN = Pattern.compile("\\[\\s*\\d{1,19}(\\s*,\\s*\\d{1,19})*\\s*\\]");
    private static final Pattern FLOAT64_PATTERN = Pattern.compile("\\d{1,19}.\\d{1,19}");
    private static final String TS_PREFIX = "ts'";
    private static final String TS_SUFFIX = "'";
    private static final Pattern BOOLEAN_PATTERN = Pattern.compile("(?is)true|false");

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static List<String> readStatementsFromFile(String filename, Class<?> resourceClass) {
        try (InputStream is = resourceClass.getResourceAsStream(filename);){
            StringBuilder builder = new StringBuilder();
            try (Scanner scanner = new Scanner(is);){
                while (scanner.hasNextLine()) {
                    String line = scanner.nextLine();
                    builder.append(line).append("\n");
                }
                scanner.close();
            }
            String script = builder.toString().replaceAll("\\/\\*\n \\* Copyright \\d{4} Google LLC\n \\*\n \\* Licensed under the Apache License, Version 2.0 \\(the \"License\"\\);\n \\* you may not use this file except in compliance with the License.\n \\* You may obtain a copy of the License at\n \\*\n \\*       http://www.apache.org/licenses/LICENSE-2.0\n \\*\n \\* Unless required by applicable law or agreed to in writing, software\n \\* distributed under the License is distributed on an \"AS IS\" BASIS,\n \\* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\n \\* See the License for the specific language governing permissions and\n \\* limitations under the License.\n \\*\\/\n", "");
            String[] array = script.split(";");
            ArrayList<String> res = new ArrayList<String>(array.length);
            for (String statement : array) {
                if (statement == null || statement.trim().length() <= 0) continue;
                res.add(statement);
            }
            ArrayList<String> arrayList = res;
            return arrayList;
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public AbstractSqlScriptVerifier() {
        this(null);
    }

    public AbstractSqlScriptVerifier(GenericConnectionProvider provider) {
        this.connectionProvider = provider;
        this.logStatements = Boolean.parseBoolean(System.getProperty("log_sql_statements", "false"));
    }

    public void verifyStatementsInFile(String filename, Class<?> resourceClass) throws Exception {
        this.verifyStatementsInFile(this.connectionProvider.getConnection(), filename, resourceClass);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void verifyStatementsInFile(GenericConnection connection, String filename, Class<?> resourceClass) throws Exception {
        try {
            List<String> statements = AbstractSqlScriptVerifier.readStatementsFromFile(filename, resourceClass);
            for (String statement : statements) {
                String sql = statement.trim();
                if (this.logStatements) {
                    System.out.println("\n------------------------------------------------------\n" + new Date() + " ---- verifying statement:");
                    System.out.println(sql);
                }
                if (sql.equalsIgnoreCase("NEW_CONNECTION")) {
                    connection.close();
                    connection = this.connectionProvider.getConnection();
                    this.variables.clear();
                    continue;
                }
                this.verifyStatement(connection, sql);
            }
        }
        finally {
            if (connection != null) {
                connection.close();
            }
        }
    }

    private void verifyStatement(GenericConnection connection, String statement) throws Exception {
        statement = this.replaceVariables(statement);
        String statementWithoutComments = StatementParser.removeCommentsAndTrim((String)statement);
        java.util.regex.Matcher verifyMatcher = VERIFY_PATTERN.matcher(statementWithoutComments);
        java.util.regex.Matcher putMatcher = PUT_PATTERN.matcher(statementWithoutComments);
        if (verifyMatcher.matches()) {
            int endIndex;
            String sql = verifyMatcher.group("statement");
            String typeName = verifyMatcher.group("type");
            ExpectedResultType type = ExpectedResultType.valueOf(typeName.substring(0, endIndex = this.getFirstSpaceChar(typeName)));
            if (type == ExpectedResultType.EXCEPTION) {
                String code = verifyMatcher.group("code");
                String messagePrefix = verifyMatcher.group("messagePrefix");
                try {
                    connection.execute(sql);
                    Assert.fail((String)("expected exception: " + sql));
                }
                catch (Exception e) {
                    this.verifyExpectedException(statementWithoutComments, e, code, messagePrefix);
                }
            } else if (type == ExpectedResultType.EQUAL) {
                String variable1 = verifyMatcher.group("variable1");
                String variable2 = verifyMatcher.group("variable2");
                variable1 = variable1.substring(1, variable1.length() - 1);
                variable2 = variable2.substring(1, variable2.length() - 1);
                MatcherAssert.assertThat((String)("No variable with name " + variable1), (Object)this.variables.containsKey(variable1), (Matcher)CoreMatchers.is((Object)true));
                MatcherAssert.assertThat((String)("No variable with name " + variable2), (Object)this.variables.containsKey(variable2), (Matcher)CoreMatchers.is((Object)true));
                Object value1 = this.variables.get(variable1);
                Object value2 = this.variables.get(variable2);
                if (value1 instanceof Timestamp && value2 instanceof Timestamp) {
                    Timestamp ts1 = (Timestamp)value1;
                    Timestamp ts2 = (Timestamp)value2;
                    value1 = Timestamp.ofTimeSecondsAndNanos((long)ts1.getSeconds(), (int)(ts1.getNanos() / 1000 * 1000));
                    value2 = Timestamp.ofTimeSecondsAndNanos((long)ts2.getSeconds(), (int)(ts2.getNanos() / 1000 * 1000));
                }
                MatcherAssert.assertThat((Object)value1, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)value2)));
            } else {
                GenericStatementResult result = connection.execute(sql);
                MatcherAssert.assertThat((String)statement, (Object)result.getResultType(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)type.getStatementResultType())));
                switch (type.getStatementResultType()) {
                    case NO_RESULT: {
                        break;
                    }
                    case RESULT_SET: {
                        String column = verifyMatcher.group("column");
                        if (column == null) {
                            this.verifyActualVsExpectedResultSet(statement, result.getResultSet());
                            break;
                        }
                        String value = verifyMatcher.group("value");
                        if (value != null) {
                            String[] parts = column.split(",", 2);
                            column = parts[0].trim();
                            value = parts[1].trim();
                            column = column.substring(1, column.length() - 1);
                            this.verifyResultSetValue(statement, result.getResultSet(), column, this.parseValue(value));
                            break;
                        }
                        column = column.substring(1, column.length() - 1);
                        this.verifyResultSetColumnNotNull(statement, result.getResultSet(), column);
                        break;
                    }
                    case UPDATE_COUNT: {
                        long expectedUpdateCount = Long.valueOf(verifyMatcher.group("count").trim());
                        MatcherAssert.assertThat((String)statement, (Object)result.getUpdateCount(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)expectedUpdateCount)));
                    }
                }
            }
        } else if (putMatcher.matches()) {
            String sql = putMatcher.group("statement");
            String variable = putMatcher.group("variable");
            variable = variable.substring(1, variable.length() - 1);
            GenericStatementResult result = connection.execute(sql);
            MatcherAssert.assertThat((String)PUT_CONDITION, (Object)result.getResultType(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)StatementResult.ResultType.RESULT_SET)));
            GenericResultSet rs = result.getResultSet();
            MatcherAssert.assertThat((String)PUT_CONDITION, (Object)rs.next(), (Matcher)CoreMatchers.is((Object)true));
            MatcherAssert.assertThat((String)PUT_CONDITION, (Object)rs.getColumnCount(), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)1)));
            this.variables.put(variable, rs.getFirstValue());
            MatcherAssert.assertThat((String)PUT_CONDITION, (Object)rs.next(), (Matcher)CoreMatchers.is((Object)false));
        } else {
            connection.execute(statement);
        }
    }

    private String replaceVariables(String sql) {
        for (String key : this.variables.keySet()) {
            sql = sql.replaceAll("%%" + key + "%%", this.variables.get(key).toString());
        }
        return sql;
    }

    protected abstract void verifyExpectedException(String var1, Exception var2, String var3, String var4);

    private Object parseValue(String valueString) {
        if (valueString == null || "".equals(valueString) || "null".equalsIgnoreCase(valueString)) {
            return null;
        }
        if (valueString.startsWith(TS_SUFFIX) && valueString.endsWith(TS_SUFFIX)) {
            return valueString.substring(1, valueString.length() - 1);
        }
        if (INT64_PATTERN.matcher(valueString).matches()) {
            return Long.valueOf(valueString);
        }
        if (ARRAY_INT64_PATTERN.matcher(valueString).matches()) {
            String[] stringArray = valueString.substring(1, valueString.length() - 1).split(",");
            ArrayList<Long> res = new ArrayList<Long>();
            for (int i = 0; i < stringArray.length; ++i) {
                res.add(Long.valueOf(stringArray[i]));
            }
            return res;
        }
        if (FLOAT64_PATTERN.matcher(valueString).matches()) {
            return Double.valueOf(valueString);
        }
        if (valueString.startsWith(TS_PREFIX) && valueString.endsWith(TS_SUFFIX)) {
            try {
                return ReadOnlyStalenessUtil.parseRfc3339((String)valueString.substring(TS_PREFIX.length(), valueString.length() - TS_SUFFIX.length()));
            }
            catch (IllegalArgumentException illegalArgumentException) {
                // empty catch block
            }
        }
        if (BOOLEAN_PATTERN.matcher(valueString).matches()) {
            return Boolean.valueOf(valueString);
        }
        return valueString;
    }

    private int getFirstSpaceChar(String input) {
        for (int index = 0; index < input.length(); ++index) {
            if (!Character.isWhitespace(input.charAt(index))) continue;
            return index;
        }
        return input.length();
    }

    private void verifyResultSetColumnNotNull(String statement, GenericResultSet rs, String column) throws Exception {
        int count = 0;
        while (rs.next()) {
            MatcherAssert.assertThat((String)statement, (Object)this.getValue(rs, column), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.notNullValue()));
            ++count;
        }
        MatcherAssert.assertThat((Object)count, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.not((Matcher)CoreMatchers.equalTo((Object)0))));
    }

    private void verifyResultSetValue(String statement, GenericResultSet rs, String column, Object value) throws Exception {
        int count = 0;
        while (rs.next()) {
            if (value == null) {
                MatcherAssert.assertThat((String)statement, (Object)this.getValue(rs, column), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.nullValue()));
            } else {
                Assert.assertEquals((String)statement, (Object)this.getValue(rs, column), (Object)value);
            }
            ++count;
        }
        MatcherAssert.assertThat((Object)count, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.not((Matcher)CoreMatchers.equalTo((Object)0))));
    }

    private void verifyActualVsExpectedResultSet(String statement, GenericResultSet rs) throws Exception {
        int count = 0;
        while (rs.next()) {
            MatcherAssert.assertThat((String)statement, (Object)this.getValue(rs, "ACTUAL"), (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)this.getValue(rs, "EXPECTED"))));
            ++count;
        }
        MatcherAssert.assertThat((Object)count, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.not((Matcher)CoreMatchers.equalTo((Object)0))));
    }

    private Object getValue(GenericResultSet rs, String col) throws Exception {
        return rs.getValue(col);
    }

    public static interface GenericConnectionProvider {
        public GenericConnection getConnection();
    }

    protected static abstract class GenericResultSet {
        protected GenericResultSet() {
        }

        protected abstract boolean next() throws Exception;

        protected abstract Object getValue(String var1) throws Exception;

        protected abstract int getColumnCount() throws Exception;

        protected abstract Object getFirstValue() throws Exception;
    }

    public static abstract class GenericConnection
    implements AutoCloseable {
        protected abstract GenericStatementResult execute(String var1) throws Exception;

        @Override
        public abstract void close() throws Exception;
    }

    protected static abstract class GenericStatementResult {
        protected GenericStatementResult() {
        }

        protected abstract StatementResult.ResultType getResultType();

        protected abstract GenericResultSet getResultSet();

        protected abstract long getUpdateCount();
    }

    protected static enum ExpectedResultType {
        RESULT_SET,
        UPDATE_COUNT,
        NO_RESULT,
        EXCEPTION,
        EQUAL;


        StatementResult.ResultType getStatementResultType() {
            switch (this) {
                case NO_RESULT: {
                    return StatementResult.ResultType.NO_RESULT;
                }
                case RESULT_SET: {
                    return StatementResult.ResultType.RESULT_SET;
                }
                case UPDATE_COUNT: {
                    return StatementResult.ResultType.UPDATE_COUNT;
                }
            }
            throw new IllegalArgumentException("not supported");
        }
    }
}

