/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner;

import com.google.api.core.ApiClock;
import com.google.api.gax.retrying.RetrySettings;
import com.google.cloud.spanner.ErrorCode;
import com.google.cloud.spanner.SpannerException;
import com.google.cloud.spanner.SpannerExceptionFactory;
import com.google.cloud.spanner.SpannerRetryHelper;
import com.google.common.base.Stopwatch;
import com.google.common.truth.Truth;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.protobuf.Duration;
import com.google.protobuf.Message;
import com.google.rpc.RetryInfo;
import io.grpc.Context;
import io.grpc.Deadline;
import io.grpc.Metadata;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import io.grpc.protobuf.ProtoUtils;
import java.util.concurrent.Callable;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.junit.Assert;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class SpannerRetryHelperTest {
    @Test
    public void testRetryDoesNotTimeoutAfterTenMinutes() {
        FakeClock clock = new FakeClock();
        AtomicInteger attempts = new AtomicInteger();
        Callable<Integer> callable = () -> {
            if (attempts.getAndIncrement() == 0) {
                clock.currentTime = clock.currentTime + TimeUnit.MILLISECONDS.convert(10L, TimeUnit.MINUTES);
                throw SpannerExceptionFactory.newSpannerException((ErrorCode)ErrorCode.ABORTED, (String)"test");
            }
            return 2;
        };
        Assert.assertEquals((long)2L, (long)((Integer)SpannerRetryHelper.runTxWithRetriesOnAborted(callable, (RetrySettings)SpannerRetryHelper.txRetrySettings, (ApiClock)clock)).intValue());
    }

    @Test
    public void testRetryDoesFailAfterMoreThanOneDay() {
        FakeClock clock = new FakeClock();
        AtomicInteger attempts = new AtomicInteger();
        Callable<Integer> callable = () -> {
            if (attempts.getAndIncrement() == 0) {
                clock.currentTime = clock.currentTime + TimeUnit.MILLISECONDS.convert(25L, TimeUnit.HOURS);
                throw SpannerExceptionFactory.newSpannerException((ErrorCode)ErrorCode.ABORTED, (String)"test");
            }
            return 2;
        };
        SpannerException e = (SpannerException)Assert.assertThrows(SpannerException.class, () -> {
            Integer cfr_ignored_0 = (Integer)SpannerRetryHelper.runTxWithRetriesOnAborted((Callable)callable, (RetrySettings)SpannerRetryHelper.txRetrySettings, (ApiClock)clock);
        });
        Assert.assertEquals((Object)ErrorCode.ABORTED, (Object)e.getErrorCode());
        Assert.assertEquals((long)1L, (long)attempts.get());
    }

    @Test
    public void testCancelledContext() {
        Context.CancellableContext withCancellation = Context.current().withCancellation();
        CountDownLatch latch = new CountDownLatch(1);
        Callable<Integer> callable = () -> {
            latch.countDown();
            throw SpannerExceptionFactory.newSpannerException((ErrorCode)ErrorCode.ABORTED, (String)"test");
        };
        ScheduledExecutorService service = Executors.newScheduledThreadPool(1);
        service.submit(() -> {
            latch.await();
            withCancellation.cancel((Throwable)new InterruptedException());
            return null;
        });
        SpannerException e = (SpannerException)Assert.assertThrows(SpannerException.class, () -> withCancellation.run(() -> {
            Integer cfr_ignored_0 = (Integer)SpannerRetryHelper.runTxWithRetriesOnAborted((Callable)callable);
        }));
        Assert.assertEquals((Object)ErrorCode.CANCELLED, (Object)e.getErrorCode());
    }

    @Test
    public void testTimedOutContext() {
        ScheduledExecutorService service = Executors.newScheduledThreadPool(1);
        Callable<Integer> callable = () -> {
            throw SpannerExceptionFactory.newSpannerException((ErrorCode)ErrorCode.ABORTED, (String)"test");
        };
        Context.CancellableContext withDeadline = Context.current().withDeadline(Deadline.after((long)1L, (TimeUnit)TimeUnit.MILLISECONDS), service);
        SpannerException e = (SpannerException)Assert.assertThrows(SpannerException.class, () -> withDeadline.run(() -> {
            Integer cfr_ignored_0 = (Integer)SpannerRetryHelper.runTxWithRetriesOnAborted((Callable)callable);
        }));
        Assert.assertEquals((Object)ErrorCode.DEADLINE_EXCEEDED, (Object)e.getErrorCode());
    }

    @Test
    public void noException() {
        Callable<Integer> callable = () -> 2;
        Truth.assertThat((Integer)((Integer)SpannerRetryHelper.runTxWithRetriesOnAborted(callable))).isEqualTo((Object)2);
    }

    @Test(expected=IllegalStateException.class)
    public void propagateUncheckedException() {
        Callable<Integer> callable = () -> {
            throw new IllegalStateException("test");
        };
        SpannerRetryHelper.runTxWithRetriesOnAborted(callable);
    }

    @Test
    public void retryOnAborted() {
        AtomicInteger attempts = new AtomicInteger();
        Callable<Integer> callable = () -> {
            if (attempts.getAndIncrement() == 0) {
                throw this.abortedWithRetryInfo((int)TimeUnit.MILLISECONDS.toNanos(1L));
            }
            return 2;
        };
        Truth.assertThat((Integer)((Integer)SpannerRetryHelper.runTxWithRetriesOnAborted(callable))).isEqualTo((Object)2);
    }

    @Test
    public void retryMultipleTimesOnAborted() {
        AtomicInteger attempts = new AtomicInteger();
        Callable<Integer> callable = () -> {
            if (attempts.getAndIncrement() < 2) {
                throw this.abortedWithRetryInfo((int)TimeUnit.MILLISECONDS.toNanos(1L));
            }
            return 2;
        };
        Truth.assertThat((Integer)((Integer)SpannerRetryHelper.runTxWithRetriesOnAborted(callable))).isEqualTo((Object)2);
    }

    @Test(expected=IllegalStateException.class)
    public void retryOnAbortedAndThenPropagateUnchecked() {
        AtomicInteger attempts = new AtomicInteger();
        Callable<Integer> callable = () -> {
            if (attempts.getAndIncrement() == 0) {
                throw this.abortedWithRetryInfo((int)TimeUnit.MILLISECONDS.toNanos(1L));
            }
            throw new IllegalStateException("test");
        };
        SpannerRetryHelper.runTxWithRetriesOnAborted(callable);
    }

    @Test
    public void testExceptionWithRetryInfo() {
        new ThreadFactoryBuilder().setDaemon(true).build().newThread(() -> {
            while (true) {
                try {
                    while (true) {
                        Thread.sleep(Long.MAX_VALUE);
                    }
                }
                catch (InterruptedException interruptedException) {
                    continue;
                }
                break;
            }
        });
        int RETRY_DELAY_MILLIS = 100;
        Metadata.Key key = ProtoUtils.keyForProto((Message)RetryInfo.getDefaultInstance());
        Status status = Status.fromCodeValue((int)Status.Code.ABORTED.value());
        Metadata trailers = new Metadata();
        RetryInfo retryInfo = RetryInfo.newBuilder().setRetryDelay(Duration.newBuilder().setNanos((int)TimeUnit.NANOSECONDS.convert(100L, TimeUnit.MILLISECONDS)).build()).build();
        trailers.put(key, (Object)retryInfo);
        SpannerException e = SpannerExceptionFactory.newSpannerException((Throwable)new StatusRuntimeException(status, trailers));
        AtomicInteger attempts = new AtomicInteger();
        Callable<Integer> callable = () -> {
            if (attempts.getAndIncrement() == 0) {
                throw e;
            }
            return 2;
        };
        Stopwatch watch = Stopwatch.createStarted();
        Truth.assertThat((Integer)((Integer)SpannerRetryHelper.runTxWithRetriesOnAborted(callable))).isEqualTo((Object)2);
        long elapsed = watch.elapsed(TimeUnit.MILLISECONDS);
        Truth.assertThat((Long)elapsed).isAtLeast(99);
    }

    private SpannerException abortedWithRetryInfo(int nanos) {
        Metadata.Key key = ProtoUtils.keyForProto((Message)RetryInfo.getDefaultInstance());
        Status status = Status.fromCodeValue((int)Status.Code.ABORTED.value());
        Metadata trailers = new Metadata();
        RetryInfo retryInfo = RetryInfo.newBuilder().setRetryDelay(Duration.newBuilder().setNanos(nanos).setSeconds(0L)).build();
        trailers.put(key, (Object)retryInfo);
        return SpannerExceptionFactory.newSpannerException((ErrorCode)ErrorCode.ABORTED, (String)"test", (Throwable)new StatusRuntimeException(status, trailers));
    }

    private static class FakeClock
    implements ApiClock {
        private long currentTime;

        private FakeClock() {
        }

        public long nanoTime() {
            return TimeUnit.NANOSECONDS.convert(this.currentTime, TimeUnit.MILLISECONDS);
        }

        public long millisTime() {
            return this.currentTime;
        }
    }
}

