/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner.it;

import com.google.api.gax.longrunning.OperationFuture;
import com.google.api.gax.paging.Page;
import com.google.api.gax.rpc.FailedPreconditionException;
import com.google.cloud.Timestamp;
import com.google.cloud.spanner.Backup;
import com.google.cloud.spanner.BackupId;
import com.google.cloud.spanner.BackupInfo;
import com.google.cloud.spanner.Database;
import com.google.cloud.spanner.DatabaseAdminClient;
import com.google.cloud.spanner.DatabaseClient;
import com.google.cloud.spanner.DatabaseId;
import com.google.cloud.spanner.ErrorCode;
import com.google.cloud.spanner.Instance;
import com.google.cloud.spanner.InstanceAdminClient;
import com.google.cloud.spanner.InstanceId;
import com.google.cloud.spanner.IntegrationTestEnv;
import com.google.cloud.spanner.Mutation;
import com.google.cloud.spanner.Options;
import com.google.cloud.spanner.ParallelIntegrationTest;
import com.google.cloud.spanner.Restore;
import com.google.cloud.spanner.ResultSet;
import com.google.cloud.spanner.SpannerException;
import com.google.cloud.spanner.SpannerExceptionFactory;
import com.google.cloud.spanner.Statement;
import com.google.cloud.spanner.encryption.BackupEncryptionConfig;
import com.google.cloud.spanner.encryption.EncryptionConfigs;
import com.google.cloud.spanner.encryption.RestoreEncryptionConfig;
import com.google.cloud.spanner.testing.EmulatorSpannerHelper;
import com.google.cloud.spanner.testing.RemoteSpannerHelper;
import com.google.cloud.spanner.testing.TimestampHelper;
import com.google.common.base.Preconditions;
import com.google.common.base.Predicate;
import com.google.common.base.Stopwatch;
import com.google.common.collect.Iterables;
import com.google.common.truth.Truth;
import com.google.longrunning.Operation;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.spanner.admin.database.v1.CreateBackupMetadata;
import com.google.spanner.admin.database.v1.RestoreDatabaseMetadata;
import com.google.spanner.admin.database.v1.RestoreSourceType;
import io.grpc.Status;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.junit.After;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@Category(value={ParallelIntegrationTest.class})
@RunWith(value=JUnit4.class)
public class ITBackupTest {
    private static final Logger logger = Logger.getLogger(ITBackupTest.class.getName());
    private static final String EXPECTED_OP_NAME_FORMAT = "%s/backups/%s/operations/";
    private static final String KMS_KEY_NAME_PROPERTY = "spanner.testenv.kms_key.name";
    @ClassRule
    public static IntegrationTestEnv env = new IntegrationTestEnv();
    private static String keyName;
    private DatabaseAdminClient dbAdminClient;
    private InstanceAdminClient instanceAdminClient;
    private Instance instance;
    private RemoteSpannerHelper testHelper;
    private List<String> databases = new ArrayList<String>();
    private List<String> backups = new ArrayList<String>();
    private String projectId;
    private String instanceId;

    @BeforeClass
    public static void doNotRunOnEmulator() {
        Assume.assumeFalse((String)"backups are not supported on the emulator", (boolean)EmulatorSpannerHelper.isUsingEmulator());
        keyName = System.getProperty(KMS_KEY_NAME_PROPERTY);
        Preconditions.checkNotNull((Object)keyName, (Object)"Key name is null, please set a key to be used for this test. The necessary permissions should be grant to the spanner service account according to the CMEK user guide.");
    }

    @Before
    public void setUp() {
        logger.info("Setting up tests");
        this.testHelper = env.getTestHelper();
        this.dbAdminClient = this.testHelper.getClient().getDatabaseAdminClient();
        this.instanceAdminClient = this.testHelper.getClient().getInstanceAdminClient();
        this.instance = this.instanceAdminClient.getInstance(this.testHelper.getInstanceId().getInstance());
        this.projectId = this.testHelper.getInstanceId().getProject();
        this.instanceId = this.testHelper.getInstanceId().getInstance();
        logger.info("Finished setup");
        logger.info("Cancelling long-running test backup operations");
        Pattern pattern = Pattern.compile(".*/backups/testbck_\\d{6}_\\d{4}_bck\\d/operations/.*");
        try {
            for (Operation operation : this.dbAdminClient.listBackupOperations(this.instance.getId().getInstance(), new Options.ListOption[0]).iterateAll()) {
                Matcher matcher = pattern.matcher(operation.getName());
                if (!matcher.matches() || operation.getDone()) continue;
                Timestamp currentTime = Timestamp.now();
                Timestamp startTime = Timestamp.fromProto((com.google.protobuf.Timestamp)((CreateBackupMetadata)operation.getMetadata().unpack(CreateBackupMetadata.class)).getProgress().getStartTime());
                long diffSeconds = currentTime.getSeconds() - startTime.getSeconds();
                if (TimeUnit.HOURS.convert(diffSeconds, TimeUnit.SECONDS) < 6L) continue;
                logger.warning(String.format("Cancelling test backup operation %s that was started at %s", operation.getName(), startTime.toString()));
                this.dbAdminClient.cancelOperation(operation.getName());
            }
        }
        catch (InvalidProtocolBufferException e) {
            logger.log(Level.WARNING, "Could not list all existing backup operations.", e);
        }
        logger.info("Finished checking existing test backup operations");
    }

    @After
    public void tearDown() throws Exception {
        for (String backup : this.backups) {
            this.waitForDbOperations(backup);
            this.dbAdminClient.deleteBackup(this.testHelper.getInstanceId().getInstance(), backup);
        }
        this.backups.clear();
        for (String db : this.databases) {
            this.dbAdminClient.dropDatabase(this.testHelper.getInstanceId().getInstance(), db);
        }
    }

    private void waitForDbOperations(String backupId) throws InterruptedException {
        try {
            Backup backupMetadata = this.dbAdminClient.getBackup(this.testHelper.getInstanceId().getInstance(), backupId);
            boolean allDbOpsDone = false;
            while (!allDbOpsDone) {
                allDbOpsDone = true;
                block3: for (String referencingDb : backupMetadata.getProto().getReferencingDatabasesList()) {
                    String filter = String.format("name:%s/operations/ AND (metadata.@type:type.googleapis.com/google.spanner.admin.database.v1.OptimizeRestoredDatabaseMetadata)", referencingDb);
                    for (Operation op : this.dbAdminClient.listDatabaseOperations(this.testHelper.getInstanceId().getInstance(), new Options.ListOption[]{Options.filter((String)filter)}).iterateAll()) {
                        if (op.getDone()) continue;
                        Thread.sleep(5000L);
                        allDbOpsDone = false;
                        continue block3;
                    }
                }
            }
        }
        catch (SpannerException e) {
            if (e.getErrorCode() == ErrorCode.NOT_FOUND) {
                return;
            }
            throw e;
        }
    }

    @Test
    public void testBackups() throws InterruptedException, ExecutionException {
        String db1Id = this.testHelper.getUniqueDatabaseId() + "_db1";
        Database sourceDatabase1 = this.dbAdminClient.newDatabaseBuilder(DatabaseId.of((String)this.projectId, (String)this.instanceId, (String)db1Id)).setEncryptionConfig(EncryptionConfigs.customerManagedEncryption((String)keyName)).build();
        logger.info(String.format("Creating test database %s", db1Id));
        OperationFuture dbOp1 = this.dbAdminClient.createDatabase(sourceDatabase1, Collections.singletonList("CREATE TABLE FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)"));
        String db2Id = this.testHelper.getUniqueDatabaseId() + "_db2";
        logger.info(String.format("Creating test database %s", db2Id));
        OperationFuture dbOp2 = this.dbAdminClient.createDatabase(this.testHelper.getInstanceId().getInstance(), this.testHelper.getUniqueDatabaseId() + "_db2", Arrays.asList("CREATE TABLE BAR (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)"));
        Database db1 = (Database)dbOp1.get();
        Database db2 = (Database)dbOp2.get();
        this.databases.add(db1.getId().getDatabase());
        this.databases.add(db2.getId().getDatabase());
        DatabaseClient client = this.testHelper.getDatabaseClient(db2);
        client.writeAtLeastOnce(Arrays.asList(((Mutation.WriteBuilder)((Mutation.WriteBuilder)Mutation.newInsertOrUpdateBuilder((String)"BAR").set("ID").to(1L)).set("NAME").to("TEST")).build()));
        this.testDatabaseEncryption(db1, keyName);
        String backupId1 = this.testHelper.getUniqueBackupId() + "_bck1";
        String backupId2 = this.testHelper.getUniqueBackupId() + "_bck2";
        Timestamp expireTime = TimestampHelper.afterDays((int)7);
        Timestamp versionTime = this.getCurrentTimestamp(client);
        logger.info(String.format("Creating backups %s and %s in parallel", backupId1, backupId2));
        Backup backupToCreate1 = this.dbAdminClient.newBackupBuilder(BackupId.of((String)this.projectId, (String)this.instanceId, (String)backupId1)).setDatabase(db1.getId()).setExpireTime(expireTime).setVersionTime(versionTime).setEncryptionConfig((BackupEncryptionConfig)EncryptionConfigs.customerManagedEncryption((String)keyName)).build();
        Backup backupToCreate2 = this.dbAdminClient.newBackupBuilder(BackupId.of((String)this.projectId, (String)this.instanceId, (String)backupId2)).setDatabase(db2.getId()).setExpireTime(expireTime).build();
        OperationFuture op1 = this.dbAdminClient.createBackup(backupToCreate1);
        OperationFuture op2 = this.dbAdminClient.createBackup(backupToCreate2);
        this.backups.add(backupId1);
        this.backups.add(backupId2);
        this.testMetadata((OperationFuture<Backup, CreateBackupMetadata>)op1, (OperationFuture<Backup, CreateBackupMetadata>)op2, backupId1, backupId2, db1, db2);
        logger.info("Waiting for backup operations to finish");
        Backup backup1 = null;
        Backup backup2 = null;
        Stopwatch watch = Stopwatch.createStarted();
        try {
            backup1 = (Backup)op1.get(6L, TimeUnit.MINUTES);
            backup2 = (Backup)op2.get(Math.max(1L, 6L - watch.elapsed(TimeUnit.MINUTES)), TimeUnit.MINUTES);
        }
        catch (TimeoutException e) {
            logger.warning("Waiting for backup operations to finish timed out. Getting long-running operations.");
            while (!(watch.elapsed(TimeUnit.MINUTES) >= 12L || this.dbAdminClient.getOperation(op1.getName()).getDone() && this.dbAdminClient.getOperation(op2.getName()).getDone())) {
                Thread.sleep(10000L);
            }
            if (!this.dbAdminClient.getOperation(op1.getName()).getDone()) {
                logger.warning(String.format("Operation %s still not finished", op1.getName()));
                throw SpannerExceptionFactory.newSpannerException((ErrorCode)ErrorCode.DEADLINE_EXCEEDED, (String)"Backup1 still not finished. Test is giving up waiting for it.");
            }
            if (!this.dbAdminClient.getOperation(op2.getName()).getDone()) {
                logger.warning(String.format("Operation %s still not finished", op2.getName()));
                throw SpannerExceptionFactory.newSpannerException((ErrorCode)ErrorCode.DEADLINE_EXCEEDED, (String)"Backup2 still not finished. Test is giving up waiting for it.");
            }
            logger.info("Long-running operations finished. Getting backups by id.");
            backup1 = this.dbAdminClient.getBackup(this.instance.getId().getInstance(), backupId1);
            backup2 = this.dbAdminClient.getBackup(this.instance.getId().getInstance(), backupId2);
        }
        this.testBackupVersionTime(backup1, versionTime);
        this.testBackupEncryption(backup1, keyName);
        Timestamp commitTs = client.writeAtLeastOnce(Arrays.asList(((Mutation.WriteBuilder)((Mutation.WriteBuilder)Mutation.newInsertOrUpdateBuilder((String)"BAR").set("ID").to(2L)).set("NAME").to("TEST2")).build()));
        logger.info("Listing all backups");
        Truth.assertThat((Iterable)this.instance.listBackups(new Options.ListOption[0]).iterateAll()).containsAtLeast((Object)backup1, (Object)backup2, new Object[0]);
        logger.info("Listing backups with name bck1");
        Truth.assertThat((Iterable)this.dbAdminClient.listBackups(this.instanceId, new Options.ListOption[]{Options.filter((String)String.format("name:%s", backup1.getId().getName()))}).iterateAll()).containsExactly(new Object[]{backup1});
        logger.info("Listing ready backups");
        Iterable readyBackups = this.dbAdminClient.listBackups(this.instanceId, new Options.ListOption[]{Options.filter((String)"state:READY")}).iterateAll();
        Truth.assertThat((Iterable)readyBackups).containsAtLeast((Object)backup1, (Object)backup2, new Object[0]);
        logger.info("Listing backups for database db1");
        Truth.assertThat((Iterable)this.dbAdminClient.listBackups(this.instanceId, new Options.ListOption[]{Options.filter((String)String.format("database:%s", db1.getId().getName()))}).iterateAll()).containsExactly(new Object[]{backup1});
        Timestamp ts = Timestamp.ofTimeSecondsAndNanos((long)commitTs.getSeconds(), (int)0);
        logger.info(String.format("Listing backups created before %s", ts));
        Truth.assertThat((Iterable)this.dbAdminClient.listBackups(this.instanceId, new Options.ListOption[]{Options.filter((String)String.format("create_time<\"%s\"", ts))}).iterateAll()).containsAtLeast((Object)backup1, (Object)backup2, new Object[0]);
        logger.info("Listing backups with size>0");
        Truth.assertThat((Iterable)this.dbAdminClient.listBackups(this.instanceId, new Options.ListOption[]{Options.filter((String)"size_bytes>0")}).iterateAll()).contains((Object)backup2);
        Truth.assertThat((Iterable)this.dbAdminClient.listBackups(this.instanceId, new Options.ListOption[]{Options.filter((String)"size_bytes>0")}).iterateAll()).doesNotContain((Object)backup1);
        this.testPagination(2);
        logger.info("Finished listBackup tests");
        this.testGetBackup(db2, backupId2, expireTime);
        this.testUpdateBackup(backup1);
        this.testCreateInvalidExpirationDate(db1);
        this.testRestore(backup1, versionTime, keyName);
        this.testDelete(backupId2);
        this.testCancelBackupOperation(db1);
        logger.info("Finished all backup tests");
    }

    @Test(expected=SpannerException.class)
    public void backupCreationWithVersionTimeTooFarInThePastFails() throws Exception {
        Database testDatabase = this.testHelper.createTestDatabase(new String[0]);
        DatabaseId databaseId = testDatabase.getId();
        InstanceId instanceId = databaseId.getInstanceId();
        String backupId = this.testHelper.getUniqueBackupId();
        Timestamp expireTime = TimestampHelper.afterDays((int)7);
        Timestamp versionTime = TimestampHelper.daysAgo((int)30);
        Backup backupToCreate = this.dbAdminClient.newBackupBuilder(BackupId.of((InstanceId)instanceId, (String)backupId)).setDatabase(databaseId).setExpireTime(expireTime).setVersionTime(versionTime).build();
        this.getOrThrow(this.dbAdminClient.createBackup(backupToCreate));
    }

    @Test(expected=SpannerException.class)
    public void backupCreationWithVersionTimeInTheFutureFails() throws Exception {
        Database testDatabase = this.testHelper.createTestDatabase(new String[0]);
        DatabaseId databaseId = testDatabase.getId();
        InstanceId instanceId = databaseId.getInstanceId();
        String backupId = this.testHelper.getUniqueBackupId();
        Timestamp expireTime = TimestampHelper.afterDays((int)7);
        Timestamp versionTime = TimestampHelper.afterDays((int)1);
        Backup backupToCreate = this.dbAdminClient.newBackupBuilder(BackupId.of((InstanceId)instanceId, (String)backupId)).setDatabase(databaseId).setExpireTime(expireTime).setVersionTime(versionTime).build();
        this.getOrThrow(this.dbAdminClient.createBackup(backupToCreate));
    }

    private <T> T getOrThrow(OperationFuture<T, ?> operation) throws InterruptedException, ExecutionException {
        try {
            return (T)operation.get();
        }
        catch (ExecutionException e) {
            if (e.getCause() instanceof SpannerException) {
                throw (SpannerException)e.getCause();
            }
            throw e;
        }
    }

    private Timestamp getCurrentTimestamp(DatabaseClient client) {
        try (ResultSet resultSet = client.singleUse().executeQuery(Statement.of((String)"SELECT CURRENT_TIMESTAMP()"), new Options.QueryOption[0]);){
            resultSet.next();
            Timestamp timestamp = resultSet.getTimestamp(0);
            return timestamp;
        }
    }

    private void testBackupVersionTime(Backup backup, Timestamp versionTime) {
        logger.info("Verifying backup version time for " + backup.getId());
        Truth.assertThat((Comparable)backup.getVersionTime()).isEqualTo((Object)versionTime);
        logger.info("Done verifying backup version time for " + backup.getId());
    }

    private void testDatabaseEncryption(Database database, String expectedKey) {
        logger.info("Verifying database encryption for " + database.getId());
        Truth.assertThat((Object)database.getEncryptionConfig()).isNotNull();
        Truth.assertThat((String)database.getEncryptionConfig().getKmsKeyName()).isEqualTo((Object)expectedKey);
        logger.info("Done verifying database encryption for " + database.getId());
    }

    private void testBackupEncryption(Backup backup, String expectedKey) {
        logger.info("Verifying backup encryption for " + backup.getId());
        Truth.assertThat((Object)backup.getEncryptionInfo()).isNotNull();
        Truth.assertThat((String)backup.getEncryptionInfo().getKmsKeyVersion()).contains((CharSequence)expectedKey);
        logger.info("Done verifying backup encryption for " + backup.getId());
    }

    private void testMetadata(OperationFuture<Backup, CreateBackupMetadata> op1, OperationFuture<Backup, CreateBackupMetadata> op2, String backupId1, String backupId2, Database db1, Database db2) throws InterruptedException, ExecutionException {
        logger.info("Getting operation metadata 1");
        CreateBackupMetadata metadata1 = (CreateBackupMetadata)op1.getMetadata().get();
        logger.info("Getting operation metadata 2");
        CreateBackupMetadata metadata2 = (CreateBackupMetadata)op2.getMetadata().get();
        String expectedOperationName1 = String.format(EXPECTED_OP_NAME_FORMAT, this.testHelper.getInstanceId().getName(), backupId1);
        String expectedOperationName2 = String.format(EXPECTED_OP_NAME_FORMAT, this.testHelper.getInstanceId().getName(), backupId2);
        Truth.assertThat((String)op1.getName()).startsWith(expectedOperationName1);
        Truth.assertThat((String)op2.getName()).startsWith(expectedOperationName2);
        Truth.assertThat((String)metadata1.getDatabase()).isEqualTo((Object)db1.getId().getName());
        Truth.assertThat((String)metadata2.getDatabase()).isEqualTo((Object)db2.getId().getName());
        Truth.assertThat((String)metadata1.getName()).isEqualTo((Object)BackupId.of((InstanceId)this.testHelper.getInstanceId(), (String)backupId1).getName());
        Truth.assertThat((String)metadata2.getName()).isEqualTo((Object)BackupId.of((InstanceId)this.testHelper.getInstanceId(), (String)backupId2).getName());
        logger.info("Finished metadata tests");
    }

    private void testCreateInvalidExpirationDate(Database db) throws InterruptedException {
        Timestamp expireTime = TimestampHelper.daysAgo((int)1);
        String backupId = this.testHelper.getUniqueBackupId();
        logger.info(String.format("Creating backup %s with invalid expiration date", backupId));
        OperationFuture op = this.dbAdminClient.createBackup(this.instanceId, backupId, db.getId().getDatabase(), expireTime);
        this.backups.add(backupId);
        try {
            op.get();
            Assert.fail((String)"missing expected exception");
        }
        catch (ExecutionException e) {
            Throwable cause = e.getCause();
            Truth.assertThat((Throwable)cause).isInstanceOf(SpannerException.class);
            SpannerException se = (SpannerException)cause;
            Truth.assertThat((Comparable)se.getErrorCode()).isEqualTo((Object)ErrorCode.INVALID_ARGUMENT);
        }
    }

    private void testCancelBackupOperation(Database db) throws InterruptedException, ExecutionException {
        Timestamp expireTime = TimestampHelper.afterDays((int)7);
        String backupId = this.testHelper.getUniqueBackupId();
        logger.info(String.format("Starting to create backup %s", backupId));
        OperationFuture op = this.dbAdminClient.createBackup(this.instanceId, backupId, db.getId().getDatabase(), expireTime);
        this.backups.add(backupId);
        logger.info(String.format("Cancelling the creation of backup %s", backupId));
        this.dbAdminClient.cancelOperation(op.getName());
        logger.info("Fetching backup operations");
        boolean operationFound = false;
        for (Operation operation : this.dbAdminClient.listBackupOperations(this.instanceId, new Options.ListOption[]{Options.filter((String)String.format("name:%s", op.getName()))}).iterateAll()) {
            Truth.assertThat((Integer)operation.getError().getCode()).isEqualTo((Object)Status.Code.CANCELLED.value());
            operationFound = true;
        }
        Truth.assertThat((Boolean)operationFound).isTrue();
        logger.info("Finished cancel test");
    }

    private void testGetBackup(Database db, String backupId, Timestamp expireTime) {
        logger.info(String.format("Getting backup %s", backupId));
        Backup backup = this.instance.getBackup(backupId);
        Truth.assertThat((Comparable)backup.getState()).isEqualTo((Object)BackupInfo.State.READY);
        Truth.assertThat((Long)backup.getSize()).isGreaterThan((Comparable)Long.valueOf(0L));
        Truth.assertThat((Comparable)backup.getExpireTime()).isEqualTo((Object)expireTime);
        Truth.assertThat((Object)backup.getDatabase()).isEqualTo((Object)db.getId());
    }

    private void testUpdateBackup(Backup backup) {
        Timestamp tomorrow = TimestampHelper.afterDays((int)1);
        backup = backup.toBuilder().setExpireTime(tomorrow).build();
        logger.info(String.format("Updating expire time of backup %s to 1 week", backup.getId().getBackup()));
        backup.updateExpireTime();
        logger.info(String.format("Reloading backup %s", backup.getId().getBackup()));
        backup = backup.reload();
        Truth.assertThat((Comparable)backup.getExpireTime()).isEqualTo((Object)tomorrow);
        Timestamp in5Minutes = TimestampHelper.afterMinutes((int)5);
        backup = backup.toBuilder().setExpireTime(in5Minutes).build();
        try {
            logger.info(String.format("Updating expire time of backup %s to 5 minutes", backup.getId().getBackup()));
            backup.updateExpireTime();
            Assert.fail((String)"Missing expected exception");
        }
        catch (SpannerException e) {
            Truth.assertThat((Comparable)e.getErrorCode()).isEqualTo((Object)ErrorCode.INVALID_ARGUMENT);
        }
        backup = backup.reload();
        Truth.assertThat((Comparable)backup.getExpireTime()).isEqualTo((Object)tomorrow);
    }

    private void testPagination(int expectedMinimumTotalBackups) {
        logger.info("Listing backups using pagination");
        int numBackups = 0;
        logger.info("Fetching first page");
        Page page = this.dbAdminClient.listBackups(this.instanceId, new Options.ListOption[]{Options.pageSize((int)1)});
        Truth.assertThat((Iterable)page.getValues()).hasSize(1);
        ++numBackups;
        Truth.assertThat((Boolean)page.hasNextPage()).isTrue();
        HashSet<String> seenPageTokens = new HashSet<String>();
        seenPageTokens.add("");
        while (page.hasNextPage()) {
            logger.info(String.format("Fetching page %d with page token %s", numBackups + 1, page.getNextPageToken()));
            Truth.assertThat(seenPageTokens).doesNotContain((Object)page.getNextPageToken());
            seenPageTokens.add(page.getNextPageToken());
            page = this.dbAdminClient.listBackups(this.instanceId, new Options.ListOption[]{Options.pageToken((String)page.getNextPageToken()), Options.pageSize((int)1)});
            Truth.assertThat((Iterable)page.getValues()).hasSize(1);
            ++numBackups;
        }
        Truth.assertThat((Integer)numBackups).isAtLeast((Comparable)Integer.valueOf(expectedMinimumTotalBackups));
    }

    private void testDelete(String backupId) throws InterruptedException {
        this.waitForDbOperations(backupId);
        logger.info(String.format("Fetching backup %s", backupId));
        Backup backup = this.instance.getBackup(backupId);
        logger.info(String.format("Deleting backup %s", backupId));
        backup.delete();
        try {
            logger.info(String.format("Fetching non-existent backup %s", backupId));
            this.instance.getBackup(backupId);
            Assert.fail((String)"Missing expected exception");
        }
        catch (SpannerException e) {
            Truth.assertThat((Comparable)e.getErrorCode()).isEqualTo((Object)ErrorCode.NOT_FOUND);
        }
        logger.info(String.format("Deleting non-existent backup %s", backupId));
        backup.delete();
        logger.info("Finished delete tests");
    }

    private void testRestore(Backup backup, Timestamp versionTime, String expectedKey) throws InterruptedException, ExecutionException {
        String restoreOperationName;
        OperationFuture restoreOperation;
        String restoredDb = this.testHelper.getUniqueDatabaseId();
        int attempts = 0;
        while (true) {
            try {
                logger.info(String.format("Restoring backup %s to database %s", backup.getId().getBackup(), restoredDb));
                Restore restore = this.dbAdminClient.newRestoreBuilder(backup.getId(), DatabaseId.of((String)this.projectId, (String)this.instanceId, (String)restoredDb)).setEncryptionConfig((RestoreEncryptionConfig)EncryptionConfigs.customerManagedEncryption((String)expectedKey)).build();
                restoreOperation = this.dbAdminClient.restoreDatabase(restore);
                restoreOperationName = restoreOperation.getName();
            }
            catch (ExecutionException e) {
                if (e.getCause() instanceof FailedPreconditionException && e.getCause().getMessage().contains("Please retry the operation once the pending restores complete")) {
                    if (++attempts == 10) {
                        logger.info("Restore operation failed 10 times because of other pending restores. Skipping restore test.");
                        return;
                    }
                    logger.info(String.format("Restoring backup %s to database %s must wait because of other pending restore operation", backup.getId().getBackup(), restoredDb));
                    Thread.sleep(60000L);
                    continue;
                }
                throw e;
            }
            break;
        }
        this.databases.add(restoredDb);
        logger.info(String.format("Restore operation %s running", restoreOperationName));
        RestoreDatabaseMetadata metadata = (RestoreDatabaseMetadata)restoreOperation.getMetadata().get();
        Truth.assertThat((String)metadata.getBackupInfo().getBackup()).isEqualTo((Object)backup.getId().getName());
        Truth.assertThat((Comparable)metadata.getSourceType()).isEqualTo((Object)RestoreSourceType.BACKUP);
        Truth.assertThat((String)metadata.getName()).isEqualTo((Object)DatabaseId.of((InstanceId)this.testHelper.getInstanceId(), (String)restoredDb).getName());
        Truth.assertThat((Comparable)Timestamp.fromProto((com.google.protobuf.Timestamp)metadata.getBackupInfo().getVersionTime())).isEqualTo((Object)versionTime);
        Database database = (Database)restoreOperation.get();
        Truth.assertThat((String)database.getId().getDatabase()).isEqualTo((Object)restoredDb);
        Database reloadedDatabase = database.reload();
        Truth.assertThat((Comparable)Timestamp.fromProto((com.google.protobuf.Timestamp)reloadedDatabase.getProto().getRestoreInfo().getBackupInfo().getVersionTime())).isEqualTo((Object)versionTime);
        this.testDatabaseEncryption(reloadedDatabase, expectedKey);
        try {
            logger.info(String.format("Restoring backup %s to existing database %s", backup.getId().getBackup(), restoredDb));
            backup.restore(DatabaseId.of((InstanceId)this.testHelper.getInstanceId(), (String)restoredDb)).get();
            Assert.fail((String)"Missing expected exception");
        }
        catch (ExecutionException ee) {
            Truth.assertThat((Throwable)ee.getCause()).isInstanceOf(SpannerException.class);
            SpannerException e = (SpannerException)ee.getCause();
            Truth.assertThat((Comparable)e.getErrorCode()).isEqualTo((Object)ErrorCode.ALREADY_EXISTS);
        }
    }

    private void verifyRestoreOperations(final String backupOperationName, final String restoreOperationName) {
        Truth.assertThat((Boolean)Iterables.any((Iterable)this.instance.listBackupOperations(new Options.ListOption[0]).iterateAll(), (Predicate)new Predicate<Operation>(){

            public boolean apply(Operation input) {
                return input.getName().equals(backupOperationName);
            }
        })).isTrue();
        Truth.assertThat((Boolean)Iterables.any((Iterable)this.instance.listBackupOperations(new Options.ListOption[0]).iterateAll(), (Predicate)new Predicate<Operation>(){

            public boolean apply(Operation input) {
                return input.getName().equals(restoreOperationName);
            }
        })).isFalse();
        Truth.assertThat((Boolean)Iterables.any((Iterable)this.instance.listDatabaseOperations(new Options.ListOption[0]).iterateAll(), (Predicate)new Predicate<Operation>(){

            public boolean apply(Operation input) {
                return input.getName().equals(backupOperationName);
            }
        })).isFalse();
        Truth.assertThat((Boolean)Iterables.any((Iterable)this.instance.listDatabaseOperations(new Options.ListOption[0]).iterateAll(), (Predicate)new Predicate<Operation>(){

            public boolean apply(Operation input) {
                return input.getName().equals(restoreOperationName);
            }
        })).isTrue();
    }
}

