/*
 * Copyright 2019 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/*
 * Script that tests the different possible read-only staleness modes
 */

-- First test in autocommit mode. READONLY mode is not strictly necessary
SET AUTOCOMMIT = TRUE;
SET READONLY = FALSE;

--------------------- STRONG ----------------------------
@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='STRONG';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','STRONG'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='Strong';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','STRONG'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='strong';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','STRONG'
SHOW VARIABLE READ_ONLY_STALENESS;

-- Try to set STRONG with a timestamp value
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'STRONG 2018-11-15T13:09:25Z''
SET READ_ONLY_STALENESS='STRONG 2018-11-15T13:09:25Z';

-- Try to set STRONG with a duration value
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'STRONG 10s''
SET READ_ONLY_STALENESS='STRONG 10s';

--------------------- MIN_READ_TIMESTAMP ----------------------------
@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='MIN_READ_TIMESTAMP 2018-11-15T13:09:25Z';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','MIN_READ_TIMESTAMP 2018-11-15T13:09:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='Min_Read_Timestamp 2018-11-15T13:09:25-08:00';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','MIN_READ_TIMESTAMP 2018-11-15T21:09:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='min_read_timestamp		2018-11-15T13:09:25+07:45';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','MIN_READ_TIMESTAMP 2018-11-15T05:24:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

-- Try to set MIN_READ_TIMESTAMP without a timestamp
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'MIN_READ_TIMESTAMP''
SET READ_ONLY_STALENESS='MIN_READ_TIMESTAMP';

-- Try to set MIN_READ_TIMESTAMP with a duration
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'MIN_READ_TIMESTAMP 10s''
SET READ_ONLY_STALENESS='MIN_READ_TIMESTAMP 10s';

--------------------- READ_TIMESTAMP ----------------------------
@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='READ_TIMESTAMP 2018-11-15T13:09:25Z';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','READ_TIMESTAMP 2018-11-15T13:09:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='Read_Timestamp 2018-11-15T13:09:25-08:00';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','READ_TIMESTAMP 2018-11-15T21:09:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='read_timestamp		2018-11-15T13:09:25+07:45';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','READ_TIMESTAMP 2018-11-15T05:24:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

-- Try to set READ_TIMESTAMP without a timestamp
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'READ_TIMESTAMP''
SET READ_ONLY_STALENESS='READ_TIMESTAMP';

-- Try to set READ_TIMESTAMP with a duration
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'READ_TIMESTAMP 10s''
SET READ_ONLY_STALENESS='READ_TIMESTAMP 10s';

--------------------- MAX_STALENESS ----------------------------
@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='MAX_STALENESS 100ms';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','MAX_STALENESS 100ms'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='Max_Staleness 1000ms';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','MAX_STALENESS 1s'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='max_staleness		10001ns';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','MAX_STALENESS 10001ns'
SHOW VARIABLE READ_ONLY_STALENESS;

-- Try to set MAX_STALENESS without a duration
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'MAX_STALENESS''
SET READ_ONLY_STALENESS='MAX_STALENESS';

-- Try to set MAX_STALENESS with a timestamp
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'MAX_STALENESS 2018-11-15T13:09:25+07:45''
SET READ_ONLY_STALENESS='MAX_STALENESS 2018-11-15T13:09:25+07:45';

--------------------- EXACT_STALENESS ----------------------------
@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='EXACT_STALENESS 1000ms';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','EXACT_STALENESS 1s'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='Exact_Staleness 1001ms';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','EXACT_STALENESS 1001ms'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='exact_staleness		1000000000ns';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','EXACT_STALENESS 1s'
SHOW VARIABLE READ_ONLY_STALENESS;

-- Try to set EXACT_STALENESS without a duration
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'EXACT_STALENESS''
SET READ_ONLY_STALENESS='EXACT_STALENESS';

-- Try to set EXACT_STALENESS with a timestamp
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'EXACT_STALENESS 2018-11-15T13:09:25+07:45''
SET READ_ONLY_STALENESS='EXACT_STALENESS 2018-11-15T13:09:25+07:45';


------------------------------------------------------------------------------------------------------------------------------


-- Then test in transactional read-only mode.
SET AUTOCOMMIT = FALSE;
SET READONLY = TRUE;

--------------------- STRONG ----------------------------
@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='STRONG';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','STRONG'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='Strong';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','STRONG'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='strong';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','STRONG'
SHOW VARIABLE READ_ONLY_STALENESS;

-- Try to set STRONG with a timestamp value
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'STRONG 2018-11-15T13:09:25Z''
SET READ_ONLY_STALENESS='STRONG 2018-11-15T13:09:25Z';

-- Try to set STRONG with a duration value
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'STRONG 10s''
SET READ_ONLY_STALENESS='STRONG 10s';

--------------------- MIN_READ_TIMESTAMP ----------------------------
-- This is not allowed in transactional mode
@EXPECT EXCEPTION FAILED_PRECONDITION 'FAILED_PRECONDITION: MAX_STALENESS and MIN_READ_TIMESTAMP are only allowed in autocommit mode'
SET READ_ONLY_STALENESS='MIN_READ_TIMESTAMP 2018-11-15T13:09:25Z';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','STRONG'
SHOW VARIABLE READ_ONLY_STALENESS;

--------------------- READ_TIMESTAMP ----------------------------
@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='READ_TIMESTAMP 2018-11-15T13:09:25Z';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','READ_TIMESTAMP 2018-11-15T13:09:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='Read_Timestamp 2018-11-15T13:09:25-08:00';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','READ_TIMESTAMP 2018-11-15T21:09:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='read_timestamp		2018-11-15T13:09:25+07:45';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','READ_TIMESTAMP 2018-11-15T05:24:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

-- Try to set READ_TIMESTAMP without a timestamp
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'READ_TIMESTAMP''
SET READ_ONLY_STALENESS='READ_TIMESTAMP';

-- Try to set READ_TIMESTAMP with a duration
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'READ_TIMESTAMP 10s''
SET READ_ONLY_STALENESS='READ_TIMESTAMP 10s';

--------------------- MAX_STALENESS ----------------------------
-- only allowed in autocommit mode
@EXPECT EXCEPTION FAILED_PRECONDITION 'FAILED_PRECONDITION: MAX_STALENESS and MIN_READ_TIMESTAMP are only allowed in autocommit mode'
SET READ_ONLY_STALENESS='MAX_STALENESS 100ms';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','READ_TIMESTAMP 2018-11-15T05:24:25Z'
SHOW VARIABLE READ_ONLY_STALENESS;

--------------------- EXACT_STALENESS ----------------------------
@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='EXACT_STALENESS 1000ms';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','EXACT_STALENESS 1s'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='Exact_Staleness 1001ms';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','EXACT_STALENESS 1001ms'
SHOW VARIABLE READ_ONLY_STALENESS;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='exact_staleness		1000000000ns';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','EXACT_STALENESS 1s'
SHOW VARIABLE READ_ONLY_STALENESS;

-- Try to set EXACT_STALENESS without a duration
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'EXACT_STALENESS''
SET READ_ONLY_STALENESS='EXACT_STALENESS';

-- Try to set EXACT_STALENESS with a timestamp
@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown value for READ_ONLY_STALENESS: 'EXACT_STALENESS 2018-11-15T13:09:25+07:45''
SET READ_ONLY_STALENESS='EXACT_STALENESS 2018-11-15T13:09:25+07:45';


----------------------------------------------------------------------------------------------------------------

-- Then test in transactional read-write mode. This should also work, although it has no effect on the current transaction, unless the transaction mode is explicitly set to read only
SET AUTOCOMMIT = FALSE;
SET READONLY = FALSE;

@EXPECT NO_RESULT
SET READ_ONLY_STALENESS='STRONG';
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','STRONG'
SHOW VARIABLE READ_ONLY_STALENESS;

-- Then test while in an active transaction. This should not be allowed.
SET TRANSACTION READ ONLY;
SELECT *
FROM Singers;

@EXPECT EXCEPTION FAILED_PRECONDITION 'FAILED_PRECONDITION: Cannot set read-only staleness when a transaction has been started'
SET READ_ONLY_STALENESS='EXACT_STALENESS 1000ms';
-- Check that the staleness mode is still 'STRONG'
@EXPECT RESULT_SET 'READ_ONLY_STALENESS','STRONG'
SHOW VARIABLE READ_ONLY_STALENESS;

COMMIT;
