/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.spanner.connection;

import com.google.cloud.spanner.Dialect;
import com.google.cloud.spanner.ErrorCode;
import com.google.cloud.spanner.SpannerException;
import com.google.cloud.spanner.Statement;
import com.google.cloud.spanner.connection.AbstractStatementParser;
import com.google.cloud.spanner.connection.ClientSideStatementImpl;
import com.google.cloud.spanner.connection.ClientSideStatements;
import com.google.cloud.spanner.connection.PostgreSQLStatementParser;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.truth.Truth;
import java.io.File;
import java.io.FileNotFoundException;
import java.util.ArrayList;
import java.util.List;
import java.util.Scanner;
import java.util.Set;
import java.util.regex.Pattern;
import org.hamcrest.CoreMatchers;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.junit.Assert;
import org.junit.Assume;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
public class StatementParserTest {
    public static final String COPYRIGHT_PATTERN = "\\/\\*\n \\* Copyright \\d{4} Google LLC\n \\*\n \\* Licensed under the Apache License, Version 2.0 \\(the \"License\"\\);\n \\* you may not use this file except in compliance with the License.\n \\* You may obtain a copy of the License at\n \\*\n \\*       http://www.apache.org/licenses/LICENSE-2.0\n \\*\n \\* Unless required by applicable law or agreed to in writing, software\n \\* distributed under the License is distributed on an \"AS IS\" BASIS,\n \\* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.\n \\* See the License for the specific language governing permissions and\n \\* limitations under the License.\n \\*\\/\n";
    private static final Pattern EXPECT_PATTERN = Pattern.compile("(?is)\\s*(?:@EXPECT)\\s+'(.*)'");
    @Parameterized.Parameter
    public Dialect dialect;
    private AbstractStatementParser parser;
    private static final ImmutableMap<Dialect, String> COMMENTS_SCRIPTS = ImmutableMap.of((Object)Dialect.GOOGLE_STANDARD_SQL, (Object)"CommentsTest.sql", (Object)Dialect.POSTGRESQL, (Object)"postgresql/CommentsTest.sql");

    @Parameterized.Parameters(name="dialect = {0}")
    public static Object[] data() {
        return Dialect.values();
    }

    @Before
    public void setupParser() {
        this.parser = AbstractStatementParser.getInstance((Dialect)this.dialect);
    }

    @Test
    public void testRemoveCommentsInScript() {
        List<String> statements = this.readStatementsFromFile((String)COMMENTS_SCRIPTS.get((Object)this.dialect));
        String currentlyExpected = "";
        for (String statement : statements) {
            String sql = statement.trim();
            if (sql.startsWith("@EXPECT")) {
                java.util.regex.Matcher matcher = EXPECT_PATTERN.matcher(sql);
                if (matcher.matches()) {
                    currentlyExpected = matcher.group(1);
                    continue;
                }
                throw new IllegalArgumentException("Unknown @EXPECT statement: " + sql);
            }
            Truth.assertThat((String)this.parser.removeCommentsAndTrim(statement)).isEqualTo((Object)currentlyExpected);
        }
    }

    @Test
    public void testRemoveComments() {
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("")).isEqualTo((Object)"");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("SELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("-- This is a one line comment\nSELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("/* This is a simple multi line comment */\nSELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("/* This is a \nmulti line comment */\nSELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("/* This\nis\na\nmulti\nline\ncomment */\nSELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
        Assert.assertEquals((Object)"SELECT \"FOO\" FROM \"BAR\" WHERE name='test'", (Object)this.parser.removeCommentsAndTrim("-- Single line comment\nSELECT \"FOO\" FROM \"BAR\" WHERE name='test'"));
        Assert.assertEquals((Object)"SELECT \"FOO\" FROM \"BAR\" WHERE name='test' and id=1", (Object)this.parser.removeCommentsAndTrim("/* Multi\nline\ncomment\n*/SELECT \"FOO\" FROM \"BAR\" WHERE name='test' and id=1"));
        if (this.dialect == Dialect.POSTGRESQL) {
            Assert.assertEquals((Object)"SELECT \"FOO\nBAR\" FROM \"BAR\" WHERE name='test\ntest'", (Object)this.parser.removeCommentsAndTrim("-- Single line comment\nSELECT \"FOO\nBAR\" FROM \"BAR\" WHERE name='test\ntest'"));
            Assert.assertEquals((Object)"SELECT \"FOO\nBAR\" FROM \"BAR\" WHERE name='test\ntest' and id=1", (Object)this.parser.removeCommentsAndTrim("/* Multi\nline\ncomment\n*/SELECT \"FOO\nBAR\" FROM \"BAR\" WHERE name='test\ntest' and id=1"));
            Assert.assertEquals((Object)"SELECT 1", (Object)this.parser.removeCommentsAndTrim("/* This block comment surrounds a query which itself has a block comment...\nSELECT /* embedded single line */ 'embedded' AS x2;\n*/\nSELECT 1"));
        }
    }

    @Test
    public void testGoogleStandardSQLRemoveCommentsGsql() {
        Assume.assumeTrue((this.dialect == Dialect.GOOGLE_STANDARD_SQL ? 1 : 0) != 0);
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("/*GSQL*/")).isEqualTo((Object)"");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("/*GSQL*/SELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("/*GSQL*/-- This is a one line comment\nSELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("/*GSQL*//* This is a simple multi line comment */\nSELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("/*GSQL*//* This is a \nmulti line comment */\nSELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("/*GSQL*//* This\nis\na\nmulti\nline\ncomment */\nSELECT * FROM FOO")).isEqualTo((Object)"SELECT * FROM FOO");
    }

    @Test
    public void testStatementWithCommentContainingSlash() {
        String sql = "/*\n * Script for testing invalid/unrecognized statements\n */\n\n-- MERGE into test comment MERGE -- \n@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown statement'\nMERGE INTO Singers s\n/*** test ****/USING (VALUES (1, 'John', 'Doe')) v\nON v.column1 = s.SingerId\nWHEN NOT MATCHED \n  INSERT VALUES (v.column1, v.column2, v.column3)\nWHEN MATCHED\n  UPDATE SET FirstName = v.column2,\n             LastName = v.column3;";
        String sqlWithoutComments = "@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown statement'\nMERGE INTO Singers s\nUSING (VALUES (1, 'John', 'Doe')) v\nON v.column1 = s.SingerId\nWHEN NOT MATCHED \n  INSERT VALUES (v.column1, v.column2, v.column3)\nWHEN MATCHED\n  UPDATE SET FirstName = v.column2,\n             LastName = v.column3";
        AbstractStatementParser.ParsedStatement statement = this.parser.parse(Statement.of((String)sql));
        Truth.assertThat((String)statement.getSqlWithoutComments()).isEqualTo((Object)sqlWithoutComments);
    }

    @Test
    public void testStatementWithCommentContainingSlashAndNoAsteriskOnNewLine() {
        String sql = "/*\n * Script for testing invalid/unrecognized statements\n foo bar baz */\n\n-- MERGE INTO test comment MERGE\n@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown statement'\nMERGE INTO Singers s\nUSING (VALUES (1, 'John', 'Doe')) v\nON v.column1 = s.SingerId\n-- test again --\nWHEN NOT MATCHED \n  INSERT VALUES (v.column1, v.column2, v.column3)\nWHEN MATCHED\n  UPDATE SET FirstName = v.column2,\n             LastName = v.column3;";
        String sqlWithoutComments = "@EXPECT EXCEPTION INVALID_ARGUMENT 'INVALID_ARGUMENT: Unknown statement'\nMERGE INTO Singers s\nUSING (VALUES (1, 'John', 'Doe')) v\nON v.column1 = s.SingerId\n\nWHEN NOT MATCHED \n  INSERT VALUES (v.column1, v.column2, v.column3)\nWHEN MATCHED\n  UPDATE SET FirstName = v.column2,\n             LastName = v.column3";
        AbstractStatementParser.ParsedStatement statement = this.parser.parse(Statement.of((String)sql));
        Truth.assertThat((String)statement.getSqlWithoutComments()).isEqualTo((Object)sqlWithoutComments);
    }

    @Test
    public void testPostgresSQLDialectDollarQuoted() {
        Assume.assumeTrue((this.dialect == Dialect.POSTGRESQL ? 1 : 0) != 0);
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("$$foo$$")).isEqualTo((Object)"$$foo$$");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("$$--foo$$")).isEqualTo((Object)"$$--foo$$");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("$$\nline 1\n--line2$$")).isEqualTo((Object)"$$\nline 1\n--line2$$");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("$bar$--foo$bar$")).isEqualTo((Object)"$bar$--foo$bar$");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("$bar$\nThis is a valid string\n -- That could contain special characters$bar$")).isEqualTo((Object)"$bar$\nThis is a valid string\n -- That could contain special characters$bar$");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("SELECT FOO$BAR FROM SOME_TABLE")).isEqualTo((Object)"SELECT FOO$BAR FROM SOME_TABLE");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("SELECT FOO$BAR -- This is a comment\nFROM SOME_TABLE")).isEqualTo((Object)"SELECT FOO$BAR \nFROM SOME_TABLE");
        Truth.assertThat((String)this.parser.removeCommentsAndTrim("SELECT FOO, $BAR -- This is a comment\nFROM SOME_TABLE")).isEqualTo((Object)"SELECT FOO, $BAR \nFROM SOME_TABLE");
    }

    @Test
    public void testPostgreSQLDialectUnicodeEscapedIdentifiers() {
        Assume.assumeTrue((this.dialect == Dialect.POSTGRESQL ? 1 : 0) != 0);
        Assert.assertEquals((Object)"SELECT 'tricky' AS \"\\\"", (Object)this.parser.removeCommentsAndTrim("SELECT 'tricky' AS \"\\\""));
        Assert.assertEquals((Object)"SELECT 'tricky' AS U&\"\\\" UESCAPE '!'", (Object)this.parser.removeCommentsAndTrim("SELECT 'tricky' AS U&\"\\\" UESCAPE '!'"));
        Assert.assertEquals((Object)"SELECT '\\' AS \"tricky\"", (Object)this.parser.removeCommentsAndTrim("SELECT '\\' AS \"tricky\""));
        Assert.assertEquals((Object)"SELECT 'foo''bar'", (Object)this.parser.removeCommentsAndTrim("SELECT 'foo''bar'"));
        Assert.assertEquals((Object)"SELECT 'foo\"bar'", (Object)this.parser.removeCommentsAndTrim("SELECT 'foo\"bar'"));
        Assert.assertEquals((Object)"SELECT 'foo\"\"bar'", (Object)this.parser.removeCommentsAndTrim("SELECT 'foo\"\"bar'"));
        Assert.assertEquals((Object)"SELECT  'foo'", (Object)this.parser.removeCommentsAndTrim("SELECT /* This is a 'comment' */ 'foo'"));
        Assert.assertEquals((Object)"SELECT  'foo'", (Object)this.parser.removeCommentsAndTrim("SELECT /* This is a '''comment''' */ 'foo'"));
        Assert.assertEquals((Object)"SELECT  '''foo''' FROM bar", (Object)this.parser.removeCommentsAndTrim("SELECT /* This is a '''comment''' */ '''foo''' FROM bar"));
        Assert.assertEquals((Object)"SELECT  '''foo''' FROM \"\"\"\\bar\\\"\"\"", (Object)this.parser.removeCommentsAndTrim("SELECT /* This is a '''comment''' */ '''foo''' FROM \"\"\"\\bar\\\"\"\""));
    }

    @Test
    public void testPostgreSQLDialectSupportsEmbeddedComments() {
        Assume.assumeTrue((this.dialect == Dialect.POSTGRESQL ? 1 : 0) != 0);
        String sql = "/* This is a comment /* This is an embedded comment */ This is after the embedded comment */ SELECT 1";
        Assert.assertEquals((Object)"SELECT 1", (Object)this.parser.removeCommentsAndTrim("/* This is a comment /* This is an embedded comment */ This is after the embedded comment */ SELECT 1"));
    }

    @Test
    public void testGoogleStandardSQLDialectDoesNotSupportEmbeddedComments() {
        Assume.assumeTrue((this.dialect == Dialect.GOOGLE_STANDARD_SQL ? 1 : 0) != 0);
        String sql = "/* This is a comment /* This is an embedded comment */ This is after the embedded comment */ SELECT 1";
        Assert.assertEquals((Object)"This is after the embedded comment */ SELECT 1", (Object)this.parser.removeCommentsAndTrim("/* This is a comment /* This is an embedded comment */ This is after the embedded comment */ SELECT 1"));
    }

    @Test
    public void testPostgreSQLDialectUnterminatedComment() {
        Assume.assumeTrue((this.dialect == Dialect.POSTGRESQL ? 1 : 0) != 0);
        String sql = "/* This is a comment /* This is still a comment */ this is unterminated SELECT 1";
        try {
            this.parser.removeCommentsAndTrim("/* This is a comment /* This is still a comment */ this is unterminated SELECT 1");
            Assert.fail((String)"missing expected exception");
        }
        catch (SpannerException e) {
            Assert.assertEquals((Object)ErrorCode.INVALID_ARGUMENT, (Object)e.getErrorCode());
            Assert.assertTrue((String)"Message should contain 'unterminated block comment'", (boolean)e.getMessage().contains("unterminated block comment"));
        }
    }

    @Test
    public void testGoogleStandardSqlDialectDialectUnterminatedComment() {
        Assume.assumeTrue((this.dialect == Dialect.GOOGLE_STANDARD_SQL ? 1 : 0) != 0);
        String sql = "/* This is a comment /* This is still a comment */ this is unterminated SELECT 1";
        Assert.assertEquals((Object)"this is unterminated SELECT 1", (Object)this.parser.removeCommentsAndTrim("/* This is a comment /* This is still a comment */ this is unterminated SELECT 1"));
    }

    @Test
    public void testShowStatements() {
        AbstractStatementParser parser = AbstractStatementParser.getInstance((Dialect)this.dialect);
        Truth.assertThat((Comparable)parser.parse(Statement.of((String)"show variable autocommit bar")).getType()).isEqualTo((Object)AbstractStatementParser.StatementType.QUERY);
        Truth.assertThat((Comparable)parser.parse(Statement.of((String)"show variable autocommit")).getType()).isEqualTo((Object)AbstractStatementParser.StatementType.CLIENT_SIDE);
        if (this.dialect == Dialect.POSTGRESQL) {
            Truth.assertThat((Comparable)parser.parse(Statement.of((String)"show autocommit")).getType()).isEqualTo((Object)AbstractStatementParser.StatementType.CLIENT_SIDE);
            Truth.assertThat((Comparable)parser.parse(Statement.of((String)"show variable spanner.retry_aborts_internally")).getType()).isEqualTo((Object)AbstractStatementParser.StatementType.CLIENT_SIDE);
        } else {
            Truth.assertThat((Comparable)parser.parse(Statement.of((String)"show autocommit")).getType()).isEqualTo((Object)AbstractStatementParser.StatementType.QUERY);
            Truth.assertThat((Comparable)parser.parse(Statement.of((String)"show variable retry_aborts_internally")).getType()).isEqualTo((Object)AbstractStatementParser.StatementType.CLIENT_SIDE);
        }
        Truth.assertThat((Comparable)parser.parse(Statement.of((String)"show variable retry_aborts_internally bar")).getType()).isEqualTo((Object)AbstractStatementParser.StatementType.QUERY);
    }

    @Test
    public void testGoogleStandardSQLDialectStatementWithHashTagSingleLineComment() {
        Assume.assumeTrue((this.dialect == Dialect.GOOGLE_STANDARD_SQL ? 1 : 0) != 0);
        Truth.assertThat((String)this.parser.parse(Statement.of((String)"# this is a comment\nselect * from foo")).getSqlWithoutComments()).isEqualTo((Object)"select * from foo");
        Truth.assertThat((String)this.parser.parse(Statement.of((String)"select * from foo\n#this is a comment")).getSqlWithoutComments()).isEqualTo((Object)"select * from foo");
        Truth.assertThat((String)this.parser.parse(Statement.of((String)"select *\nfrom foo # this is a comment\nwhere bar=1")).getSqlWithoutComments()).isEqualTo((Object)"select *\nfrom foo \nwhere bar=1");
    }

    @Test
    public void testPostgreSQLDialectStatementWithHashTagSingleLineComment() {
        Assume.assumeTrue((this.dialect == Dialect.POSTGRESQL ? 1 : 0) != 0);
        Truth.assertThat((String)this.parser.parse(Statement.of((String)"# this is a comment\nselect * from foo")).getSqlWithoutComments()).isEqualTo((Object)"# this is a comment\nselect * from foo");
        Truth.assertThat((String)this.parser.parse(Statement.of((String)"select * from foo\n#this is a comment")).getSqlWithoutComments()).isEqualTo((Object)"select * from foo\n#this is a comment");
        Truth.assertThat((String)this.parser.parse(Statement.of((String)"select *\nfrom foo # this is a comment\nwhere bar=1")).getSqlWithoutComments()).isEqualTo((Object)"select *\nfrom foo # this is a comment\nwhere bar=1");
    }

    @Test
    public void testIsDdlStatement() {
        Truth.assertThat((Boolean)this.parser.isDdlStatement("")).isFalse();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("random text")).isFalse();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("CREATETABLE")).isFalse();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("CCREATE TABLE")).isFalse();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("SELECT 1")).isFalse();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("SELECT FOO FROM BAR")).isFalse();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("INSERT INTO FOO (ID, NAME) VALUES (1, 'NAME')")).isFalse();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("UPDATE FOO SET NAME='NAME' WHERE ID=1")).isFalse();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("DELETE FROM FOO")).isFalse();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("CREATE TABLE FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isTrue();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("alter table foo add Description string(100)")).isTrue();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("drop table foo")).isTrue();
        Truth.assertThat((Boolean)this.parser.isDdlStatement("Create index BAR on foo (name)")).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"\t\tCREATE\n\t   TABLE   FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isDdl()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"\n\n\nCREATE TABLE   FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isDdl()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"-- this is a comment\nCREATE TABLE   FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isDdl()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"/* multi line comment\n* with more information on the next line\n*/\nCREATE TABLE   FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isDdl()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"/** java doc comment\n* with more information on the next line\n*/\nCREATE TABLE   FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isDdl()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"-- SELECT in a single line comment \nCREATE TABLE   FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isDdl()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"/* SELECT in a multi line comment\n* with more information on the next line\n*/\nCREATE TABLE   FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isDdl()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"/** SELECT in a java doc comment\n* with more information on the next line\n*/\nCREATE TABLE   FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isDdl()).isTrue();
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"CREATE VIEW SingerNames\nSQL SECURITY INVOKER\nAS SELECT SingerId as SingerId,\n          CONCAT(Singers.FirstName, Singers.LastName) as Name\n   FROM Singers")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"create view SingerNames as select FullName from Singers")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/* this is a comment */ create view SingerNames as select FullName from Singers")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"create /* this is a comment */ view SingerNames as select FullName from Singers")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"create \n -- This is a comment \n view SingerNames as select FullName from Singers")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)" \t \n create   \n \t  view \n  \t SingerNames as select FullName from Singers")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"DROP VIEW SingerNames")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"ALTER VIEW SingerNames\nAS SELECT SingerId as SingerId,\n          CONCAT(Singers.FirstName, Singers.LastName) as Name\n   FROM Singers")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"analyze")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"Analyze")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"ANALYZE")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"\t ANALYZE\n ")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/* This is a comment */ ANALYZE ")).isDdl());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"-- comment\n ANALYZE ")).isDdl());
    }

    @Test
    public void testIsQuery() {
        Truth.assertThat((Boolean)this.parser.isQuery("")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("random text")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("SELECT1")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("SSELECT 1")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("SELECT 1")).isTrue();
        Truth.assertThat((Boolean)this.parser.isQuery("select 1")).isTrue();
        Truth.assertThat((Boolean)this.parser.isQuery("SELECT foo FROM bar WHERE id=@id")).isTrue();
        Truth.assertThat((Boolean)this.parser.isQuery("INSERT INTO FOO (ID, NAME) VALUES (1, 'NAME')")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("UPDATE FOO SET NAME='NAME' WHERE ID=1")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("DELETE FROM FOO")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("CREATE TABLE FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("alter table foo add Description string(100)")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("drop table foo")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("Create index BAR on foo (name)")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("select * from foo")).isTrue();
        Truth.assertThat((Boolean)this.parser.isQuery("INSERT INTO FOO (ID, NAME) SELECT ID+1, NAME FROM FOO")).isFalse();
        Truth.assertThat((Boolean)this.parser.isQuery("WITH subQ1 AS (SELECT SchoolID FROM Roster),\n     subQ2 AS (SELECT OpponentID FROM PlayerStats)\nSELECT * FROM subQ1\nUNION ALL\nSELECT * FROM subQ2")).isTrue();
        Truth.assertThat((Boolean)this.parser.isQuery("with subQ1 AS (SELECT SchoolID FROM Roster),\n     subQ2 AS (SELECT OpponentID FROM PlayerStats)\nselect * FROM subQ1\nUNION ALL\nSELECT * FROM subQ2")).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"-- this is a comment\nwith foo as (select * from bar)\nselect * from foo")).isQuery()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"-- this is a comment\nselect * from foo")).isQuery()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"/* multi line comment\n* with more information on the next line\n*/\nSELECT ID, NAME\nFROM\tTEST\n\tWHERE ID=1")).isQuery()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"/** java doc comment\n* with more information on the next line\n*/\nselect max(id) from test")).isQuery()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"-- INSERT in a single line comment \n    select 1")).isQuery()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"/* UPDATE in a multi line comment\n* with more information on the next line\n*/\nSELECT 1")).isQuery()).isTrue();
        Truth.assertThat((Boolean)this.parser.parse(Statement.of((String)"/** DELETE in a java doc comment\n* with more information on the next line\n*/\n\n\n\n -- UPDATE test\nSELECT 1")).isQuery()).isTrue();
    }

    @Test
    public void testGoogleStandardSQLDialectIsQuery_QueryHints() {
        Assume.assumeTrue((this.dialect == Dialect.GOOGLE_STANDARD_SQL ? 1 : 0) != 0);
        Assert.assertTrue((boolean)this.parser.isQuery("@{JOIN_METHOD=HASH_JOIN} SELECT * FROM PersonsTable"));
        Assert.assertTrue((boolean)this.parser.isQuery("@ {JOIN_METHOD=HASH_JOIN} SELECT * FROM PersonsTable"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{ JOIN_METHOD=HASH_JOIN} SELECT * FROM PersonsTable"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{JOIN_METHOD=HASH_JOIN } SELECT * FROM PersonsTable"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{JOIN_METHOD=HASH_JOIN}\nSELECT * FROM PersonsTable"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{\nJOIN_METHOD =  HASH_JOIN   \t}\n\t SELECT * FROM PersonsTable"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{JOIN_METHOD=HASH_JOIN}\n -- Single line comment\nSELECT * FROM PersonsTable"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{JOIN_METHOD=HASH_JOIN}\n /* Multi line comment\n with more comments\n */SELECT * FROM PersonsTable"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{JOIN_METHOD=HASH_JOIN} WITH subQ1 AS (SELECT SchoolID FROM Roster),\n     subQ2 AS (SELECT OpponentID FROM PlayerStats)\nSELECT * FROM subQ1\nUNION ALL\nSELECT * FROM subQ2"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{FORCE_INDEX=index_name} @{JOIN_METHOD=HASH_JOIN} SELECT * FROM tbl"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{FORCE_INDEX=index_name} @{JOIN_METHOD=HASH_JOIN} Select * FROM tbl"));
        Assert.assertTrue((boolean)this.parser.isQuery("@{FORCE_INDEX=index_name}\n@{JOIN_METHOD=HASH_JOIN}\nWITH subQ1 AS (SELECT SchoolID FROM Roster),\n     subQ2 AS (SELECT OpponentID FROM PlayerStats)\nSELECT * FROM subQ1\nUNION ALL\nSELECT * FROM subQ2"));
        Assert.assertFalse((boolean)this.parser.isQuery("@{JOIN_METHOD=HASH_JOIN SELECT * FROM PersonsTable"));
        Assert.assertFalse((boolean)this.parser.isQuery("@JOIN_METHOD=HASH_JOIN} SELECT * FROM PersonsTable"));
        Assert.assertFalse((boolean)this.parser.isQuery("@JOIN_METHOD=HASH_JOIN SELECT * FROM PersonsTable"));
        Assert.assertFalse((boolean)this.parser.isQuery("@{FORCE_INDEX=index_name} @{JOIN_METHOD=HASH_JOIN} UPDATE tbl set FOO=1 WHERE ID=2"));
    }

    @Test
    public void testIsUpdate_QueryHints() {
        Assume.assumeTrue((this.dialect == Dialect.GOOGLE_STANDARD_SQL ? 1 : 0) != 0);
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("@{LOCK_SCANNED_RANGES=exclusive} UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("@ {LOCK_SCANNED_RANGES=exclusive} UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("@{ LOCK_SCANNED_RANGES=exclusive} UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("@{LOCK_SCANNED_RANGES=exclusive } UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("@{LOCK_SCANNED_RANGES=exclusive}\nUPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("@{\nLOCK_SCANNED_RANGES =  exclusive   \t}\n\t UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("@{LOCK_SCANNED_RANGES=exclusive}\n -- Single line comment\nUPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("@{LOCK_SCANNED_RANGES=exclusive}\n /* Multi line comment\n with more comments\n */UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("@{LOCK_SCANNED_RANGES=exclusive} @{USE_ADDITIONAL_PARALLELISM=TRUE} UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("@{LOCK_SCANNED_RANGES=exclusive UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("@LOCK_SCANNED_RANGES=exclusive} UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("@LOCK_SCANNED_RANGES=exclusive UPDATE FOO SET NAME='foo' WHERE ID=1"));
    }

    @Test
    public void testIsUpdate_InsertStatements() {
        Assert.assertFalse((boolean)this.parser.isUpdateStatement(""));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("random text"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("INSERTINTO FOO (ID) VALUES (1)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("IINSERT INTO FOO (ID) VALUES (1)"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("INSERT INTO FOO (ID) VALUES (1)"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("insert into foo (id) values (1)"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("INSERT into Foo (id)\nSELECT id FROM bar WHERE id=@id"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("SELECT 1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("SELECT NAME FROM FOO WHERE ID=1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("CREATE TABLE FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("alter table foo add Description string(100)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("drop table foo"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("Create index BAR on foo (name)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("select * from foo"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("INSERT INTO FOO (ID, NAME) SELECT ID+1, NAME FROM FOO"));
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"-- this is a comment\ninsert into foo (id) values (1)")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/* multi line comment\n* with more information on the next line\n*/\nINSERT INTO FOO\n(ID)\tVALUES\n\t(1)")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/** java doc comment\n* with more information on the next line\n*/\nInsert intO foo (id) select 1")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"-- SELECT in a single line comment \n    insert into foo (id) values (1)")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/* CREATE in a multi line comment\n* with more information on the next line\n*/\nINSERT INTO FOO (ID) VALUES (1)")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/** DROP in a java doc comment\n* with more information on the next line\n*/\n\n\n\n -- SELECT test\ninsert into foo (id) values (1)")).isUpdate());
    }

    @Test
    public void testIsUpdate_UpdateStatements() {
        Assert.assertFalse((boolean)this.parser.isUpdateStatement(""));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("random text"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("UPDATEFOO SET NAME='foo' WHERE ID=1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("UUPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("UPDATE FOO SET NAME='foo' WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("update foo set name='foo' where id=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("update foo set name=\n(SELECT name FROM bar WHERE id=@id)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("SELECT 1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("SELECT NAME FROM FOO WHERE ID=1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("CREATE TABLE FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("alter table foo add Description string(100)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("drop table foo"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("Create index BAR on foo (name)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("select * from foo"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("UPDATE FOO SET NAME=(SELECT NAME FROM FOO) WHERE ID=(SELECT ID+1 FROM FOO)"));
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"-- this is a comment\nupdate foo set name='foo' where id=@id")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/* multi line comment\n* with more information on the next line\n*/\nUPDATE FOO\nSET NAME=\t'foo'\n\tWHERE ID=1")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/** java doc comment\n* with more information on the next line\n*/\nUPDATE FOO SET NAME=(select 'bar')")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"-- SELECT in a single line comment \n    update foo set name='bar'")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/* CREATE in a multi line comment\n* with more information on the next line\n*/\nUPDATE FOO SET NAME='BAR'")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/** DROP in a java doc comment\n* with more information on the next line\n*/\n\n\n\n -- SELECT test\nupdate foo set bar='foo'")).isUpdate());
    }

    @Test
    public void testIsUpdate_DeleteStatements() {
        Assert.assertFalse((boolean)this.parser.isUpdateStatement(""));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("random text"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("DELETEFROM FOO WHERE ID=1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("DDELETE FROM FOO WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("DELETE FROM FOO WHERE ID=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("delete from foo where id=1"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("delete from foo where name=\n(SELECT name FROM bar WHERE id=@id)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("SELECT 1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("SELECT NAME FROM FOO WHERE ID=1"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("CREATE TABLE FOO (ID INT64, NAME STRING(100)) PRIMARY KEY (ID)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("alter table foo add Description string(100)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("drop table foo"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("Create index BAR on foo (name)"));
        Assert.assertFalse((boolean)this.parser.isUpdateStatement("select * from foo"));
        Assert.assertTrue((boolean)this.parser.isUpdateStatement("UPDATE FOO SET NAME=(SELECT NAME FROM FOO) WHERE ID=(SELECT ID+1 FROM FOO)"));
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"-- this is a comment\ndelete from foo  where id=@id")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/* multi line comment\n* with more information on the next line\n*/\nDELETE FROM FOO\n\n\tWHERE ID=1")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/** java doc comment\n* with more information on the next line\n*/\nDELETE FROM FOO WHERE NAME=(select 'bar')")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"-- SELECT in a single line comment \n    delete from foo where name='bar'")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/* CREATE in a multi line comment\n* with more information on the next line\n*/\nDELETE FROM FOO WHERE NAME='BAR'")).isUpdate());
        Assert.assertTrue((boolean)this.parser.parse(Statement.of((String)"/** DROP in a java doc comment\n* with more information on the next line\n*/\n\n\n\n -- SELECT test\ndelete from foo where bar='foo'")).isUpdate());
    }

    @Test
    public void testParseStatementsWithNoParameters() throws ClientSideStatementImpl.CompileException {
        for (ClientSideStatementImpl statement : this.getAllStatements()) {
            if (statement.getSetStatement() != null) continue;
            for (String testStatement : statement.getExampleStatements()) {
                this.testParseStatement(testStatement, statement.getClass());
            }
        }
    }

    @Test
    public void testParseStatementsWithOneParameterAtTheEnd() throws ClientSideStatementImpl.CompileException {
        for (ClientSideStatementImpl statement : this.getAllStatements()) {
            if (statement.getSetStatement() == null) continue;
            for (String testStatement : statement.getExampleStatements()) {
                this.testParseStatementWithOneParameterAtTheEnd(testStatement, statement.getClass());
            }
        }
    }

    private Set<ClientSideStatementImpl> getAllStatements() throws ClientSideStatementImpl.CompileException {
        return ClientSideStatements.getInstance((Dialect)this.dialect).getCompiledStatements();
    }

    private <T extends ClientSideStatementImpl> void assertParsing(String value, Class<T> statementClass) {
        Truth.assertThat(this.parse(value)).isEqualTo(statementClass);
    }

    private <T extends ClientSideStatementImpl> void testParseStatement(String statement, Class<T> statementClass) {
        Truth.assertWithMessage((String)("\"" + statement + "\" should be " + statementClass.getName())).that(this.parse(statement)).isEqualTo(statementClass);
        this.assertParsing(this.upper(statement), statementClass);
        this.assertParsing(this.lower(statement), statementClass);
        this.assertParsing(this.withSpaces(statement), statementClass);
        this.assertParsing(this.withTabs(statement), statementClass);
        this.assertParsing(this.withLinefeeds(statement), statementClass);
        this.assertParsing(this.withLeadingSpaces(statement), statementClass);
        this.assertParsing(this.withLeadingTabs(statement), statementClass);
        this.assertParsing(this.withLeadingLinefeeds(statement), statementClass);
        this.assertParsing(this.withTrailingSpaces(statement), statementClass);
        this.assertParsing(this.withTrailingTabs(statement), statementClass);
        this.assertParsing(this.withTrailingLinefeeds(statement), statementClass);
        Truth.assertThat(this.parse(this.withInvalidPrefix(statement))).isNull();
        Truth.assertThat(this.parse(this.withInvalidSuffix(statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("%", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("_", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("&", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("$", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("@", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("!", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("*", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("(", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix(")", statement))).isNull();
        Truth.assertWithMessage((String)(this.withSuffix("%", statement) + " is not a valid statement")).that(this.parse(this.withSuffix("%", statement))).isNull();
        Truth.assertThat(this.parse(this.withSuffix("_", statement))).isNull();
        Truth.assertThat(this.parse(this.withSuffix("&", statement))).isNull();
        Truth.assertThat(this.parse(this.withSuffix("$", statement))).isNull();
        Truth.assertThat(this.parse(this.withSuffix("@", statement))).isNull();
        Truth.assertThat(this.parse(this.withSuffix("!", statement))).isNull();
        Truth.assertThat(this.parse(this.withSuffix("*", statement))).isNull();
        Truth.assertThat(this.parse(this.withSuffix("(", statement))).isNull();
        Truth.assertThat(this.parse(this.withSuffix(")", statement))).isNull();
    }

    private <T extends ClientSideStatementImpl> void testParseStatementWithOneParameterAtTheEnd(String statement, Class<T> statementClass) {
        Truth.assertWithMessage((String)("\"" + statement + "\" should be " + statementClass.getName())).that(this.parse(statement)).isEqualTo(statementClass);
        this.assertParsing(this.upper(statement), statementClass);
        this.assertParsing(this.lower(statement), statementClass);
        this.assertParsing(this.withSpaces(statement), statementClass);
        this.assertParsing(this.withTabs(statement), statementClass);
        this.assertParsing(this.withLinefeeds(statement), statementClass);
        this.assertParsing(this.withLeadingSpaces(statement), statementClass);
        this.assertParsing(this.withLeadingTabs(statement), statementClass);
        this.assertParsing(this.withLeadingLinefeeds(statement), statementClass);
        this.assertParsing(this.withTrailingSpaces(statement), statementClass);
        this.assertParsing(this.withTrailingTabs(statement), statementClass);
        this.assertParsing(this.withTrailingLinefeeds(statement), statementClass);
        Truth.assertThat(this.parse(this.withInvalidPrefix(statement))).isNull();
        this.assertParsing(this.withInvalidSuffix(statement), statementClass);
        Truth.assertThat(this.parse(this.withPrefix("%", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("_", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("&", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("$", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("@", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("!", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("*", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix("(", statement))).isNull();
        Truth.assertThat(this.parse(this.withPrefix(")", statement))).isNull();
        this.assertParsing(this.withSuffix("%", statement), statementClass);
        this.assertParsing(this.withSuffix("_", statement), statementClass);
        this.assertParsing(this.withSuffix("&", statement), statementClass);
        this.assertParsing(this.withSuffix("$", statement), statementClass);
        this.assertParsing(this.withSuffix("@", statement), statementClass);
        this.assertParsing(this.withSuffix("!", statement), statementClass);
        this.assertParsing(this.withSuffix("*", statement), statementClass);
        this.assertParsing(this.withSuffix("(", statement), statementClass);
        this.assertParsing(this.withSuffix(")", statement), statementClass);
    }

    @Test
    public void testGoogleStandardSQLDialectConvertPositionalParametersToNamedParameters() {
        Assume.assumeTrue((this.dialect == Dialect.GOOGLE_STANDARD_SQL ? 1 : 0) != 0);
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo where name=?").sqlWithNamedParameters).isEqualTo((Object)"select * from foo where name=@p1");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'?test?\"?test?\"?'?").sqlWithNamedParameters).isEqualTo((Object)"@p1'?test?\"?test?\"?'@p2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'?it\\'?s'?").sqlWithNamedParameters).isEqualTo((Object)"@p1'?it\\'?s'@p2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'?it\\\"?s'?").sqlWithNamedParameters).isEqualTo((Object)"@p1'?it\\\"?s'@p2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?\"?it\\\"?s\"?").sqlWithNamedParameters).isEqualTo((Object)"@p1\"?it\\\"?s\"@p2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'''?it\\'?s'''?").sqlWithNamedParameters).isEqualTo((Object)"@p1'''?it\\'?s'''@p2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?\"\"\"?it\\\"?s\"\"\"?").sqlWithNamedParameters).isEqualTo((Object)"@p1\"\"\"?it\\\"?s\"\"\"@p2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?`?it\\`?s`?").sqlWithNamedParameters).isEqualTo((Object)"@p1`?it\\`?s`@p2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?```?it\\`?s```?").sqlWithNamedParameters).isEqualTo((Object)"@p1```?it\\`?s```@p2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'''?it\\'?s \n ?it\\'?s'''?").sqlWithNamedParameters).isEqualTo((Object)"@p1'''?it\\'?s \n ?it\\'?s'''@p2");
        this.assertUnclosedLiteral("?'?it\\'?s \n ?it\\'?s'?");
        this.assertUnclosedLiteral("?'?it\\'?s \n ?it\\'?s?");
        this.assertUnclosedLiteral("?'''?it\\'?s \n ?it\\'?s'?");
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select 1, ?, 'test?test', \"test?test\", foo.* from `foo` where col1=? and col2='test' and col3=? and col4='?' and col5=\"?\" and col6='?''?''?'").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select 1, @p1, 'test?test', \"test?test\", foo.* from `foo` where col1=@p2 and col2='test' and col3=@p3 and col4='?' and col5=\"?\" and col6='?''?''?'")));
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo where name=? and col2 like ? and col3 > ?").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select * from foo where name=@p1 and col2 like @p2 and col3 > @p3")));
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo where id between ? and ?").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select * from foo where id between @p1 and @p2")));
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo limit ? offset ?").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select * from foo limit @p1 offset @p2")));
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo where col1=? and col2 like ? and col3 > ? and col4 < ? and col5 != ? and col6 not in (?, ?, ?) and col7 in (?, ?, ?) and col8 between ? and ?").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select * from foo where col1=@p1 and col2 like @p2 and col3 > @p3 and col4 < @p4 and col5 != @p5 and col6 not in (@p6, @p7, @p8) and col7 in (@p9, @p10, @p11) and col8 between @p12 and @p13")));
    }

    @Test
    public void testPostgreSQLDialectDialectConvertPositionalParametersToNamedParameters() {
        Assume.assumeTrue((this.dialect == Dialect.POSTGRESQL ? 1 : 0) != 0);
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo where name=?").sqlWithNamedParameters).isEqualTo((Object)"select * from foo where name=$1");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'?test?\"?test?\"?'?").sqlWithNamedParameters).isEqualTo((Object)"$1'?test?\"?test?\"?'$2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'?it\\''?s'?").sqlWithNamedParameters).isEqualTo((Object)"$1'?it\\''?s'$2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'?it\\\"?s'?").sqlWithNamedParameters).isEqualTo((Object)"$1'?it\\\"?s'$2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?\"?it\\\"\"?s\"?").sqlWithNamedParameters).isEqualTo((Object)"$1\"?it\\\"\"?s\"$2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'''?it\\''?s'''?").sqlWithNamedParameters).isEqualTo((Object)"$1'''?it\\''?s'''$2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?\"\"\"?it\\\"\"?s\"\"\"?").sqlWithNamedParameters).isEqualTo((Object)"$1\"\"\"?it\\\"\"?s\"\"\"$2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?$$?it$?s$$?").sqlWithNamedParameters).isEqualTo((Object)"$1$$?it$?s$$$2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?$tag$?it$$?s$tag$?").sqlWithNamedParameters).isEqualTo((Object)"$1$tag$?it$$?s$tag$$2");
        Truth.assertThat((String)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?$$?it\\'?s \n ?it\\'?s$$?").sqlWithNamedParameters).isEqualTo((Object)"$1$$?it\\'?s \n ?it\\'?s$$$2");
        Assert.assertEquals((Object)"$1'?it\\''?s \n ?it\\''?s'$2", (Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'?it\\''?s \n ?it\\''?s'?").sqlWithNamedParameters);
        this.assertUnclosedLiteral("?'?it\\''?s \n ?it\\''?s?");
        Assert.assertEquals((Object)"$1'''?it\\''?s \n ?it\\''?s'$2", (Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"?'''?it\\''?s \n ?it\\''?s'?").sqlWithNamedParameters);
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select 1, ?, 'test?test', \"test?test\", foo.* from `foo` where col1=? and col2='test' and col3=? and col4='?' and col5=\"?\" and col6='?''?''?'").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select 1, $1, 'test?test', \"test?test\", foo.* from `foo` where col1=$2 and col2='test' and col3=$3 and col4='?' and col5=\"?\" and col6='?''?''?'")));
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo where name=? and col2 like ? and col3 > ?").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select * from foo where name=$1 and col2 like $2 and col3 > $3")));
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo where id between ? and ?").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select * from foo where id between $1 and $2")));
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo limit ? offset ?").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select * from foo limit $1 offset $2")));
        MatcherAssert.assertThat((Object)this.parser.convertPositionalParametersToNamedParameters((char)'?', (String)"select * from foo where col1=? and col2 like ? and col3 > ? and col4 < ? and col5 != ? and col6 not in (?, ?, ?) and col7 in (?, ?, ?) and col8 between ? and ?").sqlWithNamedParameters, (Matcher)CoreMatchers.is((Matcher)CoreMatchers.equalTo((Object)"select * from foo where col1=$1 and col2 like $2 and col3 > $3 and col4 < $4 and col5 != $5 and col6 not in ($6, $7, $8) and col7 in ($9, $10, $11) and col8 between $12 and $13")));
    }

    @Test
    public void testPostgreSQLGetQueryParameters() {
        Assume.assumeTrue((this.dialect == Dialect.POSTGRESQL ? 1 : 0) != 0);
        PostgreSQLStatementParser parser = (PostgreSQLStatementParser)this.parser;
        Assert.assertEquals((Object)ImmutableSet.of(), (Object)parser.getQueryParameters("select * from foo"));
        Assert.assertEquals((Object)ImmutableSet.of((Object)"$1"), (Object)parser.getQueryParameters("select * from foo where bar=$1"));
        Assert.assertEquals((Object)ImmutableSet.of((Object)"$1", (Object)"$2", (Object)"$3"), (Object)parser.getQueryParameters("select $2 from foo where bar=$1 and baz=$3"));
        Assert.assertEquals((Object)ImmutableSet.of((Object)"$1", (Object)"$3"), (Object)parser.getQueryParameters("select '$2' from foo where bar=$1 and baz in ($1, $3)"));
        Assert.assertEquals((Object)ImmutableSet.of((Object)"$1"), (Object)parser.getQueryParameters("select '$2' from foo where bar=$1 and baz=$foo"));
    }

    private void assertUnclosedLiteral(String sql) {
        try {
            this.parser.convertPositionalParametersToNamedParameters('?', sql);
            Assert.fail((String)"missing expected exception");
        }
        catch (SpannerException e) {
            Truth.assertThat((Comparable)e.getErrorCode()).isSameInstanceAs((Object)ErrorCode.INVALID_ARGUMENT);
            Truth.assertThat((String)e.getMessage()).startsWith(ErrorCode.INVALID_ARGUMENT.name() + ": SQL statement contains an unclosed literal: " + sql);
        }
    }

    private <T extends ClientSideStatementImpl> Class<T> parse(String statement) {
        ClientSideStatementImpl optional = this.parser.parseClientSideStatement(statement);
        return optional != null ? optional.getClass() : null;
    }

    private String upper(String statement) {
        return statement.toUpperCase();
    }

    private String lower(String statement) {
        return statement.toLowerCase();
    }

    private String withLeadingSpaces(String statement) {
        return "   " + statement;
    }

    private String withLeadingTabs(String statement) {
        return "\t\t\t" + statement;
    }

    private String withLeadingLinefeeds(String statement) {
        return "\n\n\n" + statement;
    }

    private String withTrailingSpaces(String statement) {
        return statement + "  ";
    }

    private String withTrailingTabs(String statement) {
        return statement + "\t\t";
    }

    private String withTrailingLinefeeds(String statement) {
        return statement + "\n\n";
    }

    private String withSpaces(String statement) {
        return statement.replaceAll(" ", "   ");
    }

    private String withTabs(String statement) {
        return statement.replaceAll(" ", "\t");
    }

    private String withLinefeeds(String statement) {
        return statement.replaceAll(" ", "\n");
    }

    private String withInvalidPrefix(String statement) {
        return "foo " + statement;
    }

    private String withInvalidSuffix(String statement) {
        return statement + " bar";
    }

    private String withPrefix(String prefix, String statement) {
        return prefix + statement;
    }

    private String withSuffix(String suffix, String statement) {
        return statement + suffix;
    }

    private List<String> readStatementsFromFile(String filename) {
        File file = new File(this.getClass().getResource(filename).getFile());
        StringBuilder builder = new StringBuilder();
        try (Scanner scanner = new Scanner(file);){
            while (scanner.hasNextLine()) {
                String line = scanner.nextLine();
                builder.append(line).append("\n");
            }
        }
        catch (FileNotFoundException e) {
            throw new RuntimeException(e);
        }
        String script = builder.toString().replaceAll(COPYRIGHT_PATTERN, "");
        String[] array = script.split(";");
        ArrayList<String> res = new ArrayList<String>(array.length);
        for (String statement : array) {
            if (statement == null || statement.trim().length() <= 0) continue;
            res.add(statement);
        }
        return res;
    }
}

