package com.groupbyinc.api.model;

import com.groupbyinc.common.jackson.annotation.JsonProperty;

import java.util.List;
import java.util.Map;

/**
 * <code>
 * A Record represents a result from the search engine.
 * Each record has all the requested metadata as
 * specified in fields parameter.  Each record also has several special fields.
 *
 * - `id`: an MD5 hash of the unique URL of this record.
 * - `url`: the unique url of the record.
 * - `snippet`: If there is a keyword match, the snippet of text.  Matches have &lt;b&gt; tags surrounding them.
 * - `title`: The title of the record.
 * - `allMeta`: a map of key value pairs.  If a value is multi-assign, there will additionally be a &lt;key&gt;List
 * item, with all the values.
 * </code>
 *
 * @author will
 */
public class Record {

  private List<RefinementMatch> refinementMatches;
  @JsonProperty("_id") private String id;
  @JsonProperty("_u") private String url;
  @JsonProperty("_snippet") private String snippet;
  @JsonProperty("_t") private String title;
  private Map<String, Object> allMeta;

  /**
   * @internal
   * @return
   */
  public List<RefinementMatch> getRefinementMatches() {
    return refinementMatches;
  }

  /**
   * @internal
   * @param refinementMatches The list of refinement matches
   * @return
   */
  public Record setRefinementMatches(List<RefinementMatch> refinementMatches) {
    this.refinementMatches = refinementMatches;
    return this;
  }

  /**
   * @return The ID is generated from URL of this record which means it will
   * persist across updates.
   */
  public String getId() {
    return id;
  }

  /**
   * @param id
   *         Set the id
   *
   * @return
   */
  public Record setId(String id) {
    this.id = id;
    return this;
  }

  /**
   * <code>
   * This URL is generated by the data upload service and follows the format
   *
   *     http://<customerId>1<collection>.com/<id>.html
   * </code>
   *
   * @return The URL represents the Unique ID of the record.
   */
  public String getUrl() {
    return url;
  }

  /**
   * @param url
   *         Set the record url
   *
   * @return
   */
  public Record setUrl(String url) {
    this.url = url;
    return this;
  }

  /**
   * @return If a search was performed this result object may have a snippet
   * of matching text.
   */
  public String getSnippet() {
    return snippet;
  }

  /**
   * @param snippet
   *         Set the snippet
   *
   * @return
   */
  public Record setSnippet(String snippet) {
    this.snippet = snippet;
    return this;
  }

  /**
   * @param name
   *         Get a specific metadata value. Essentially this represents an
   *         attribute of the record data.
   *
   * @return
   */
  public Object getMetaValue(String name) {
    return allMeta.get(name);
  }

  /**
   * @return A map of all the metadata associated with this record.
   */
  public Map<String, Object> getAllMeta() {
    return allMeta;
  }

  /**
   * @param allMeta
   *         Set the metadata.
   *
   * @return
   */
  public Record setAllMeta(Map<String, Object> allMeta) {
    this.allMeta = allMeta;
    return this;
  }

  /**
   * <code>
   * The title of this record
   * </code>
   *
   * @return The title of this record.
   */
  public String getTitle() {
    return title;
  }

  /**
   * @param title
   *         Set the title
   */
  public Record setTitle(String title) {
    this.title = title;
    return this;
  }
}
