package com.gu.membership.salesforce

import com.github.nscala_time.time.Imports._

import org.json4s._
import org.json4s.DefaultReaders.{BooleanReader, StringReader}

trait Member {
  val salesforceContactId: String
  val salesforceAccountId: String
  val identityId: String
  val tier: Tier.Tier
  val regNumber: String
  val firstName: Option[String]
  val lastName: String
  val joinDate: DateTime
}

case class FreeMember(salesforceContactId: String,
                      salesforceAccountId: String,
                      identityId: String,
                      tier: Tier.Tier,
                      regNumber: String,
                      firstName: Option[String],
                      lastName: String,
                      joinDate: DateTime) extends Member

case class PaidMember(salesforceContactId: String,
                      salesforceAccountId: String,
                      identityId: String,
                      tier: Tier.Tier,
                      regNumber: String,
                      firstName: Option[String],
                      lastName: String,
                      joinDate: DateTime,
                      stripeCustomerId: String) extends Member

object Member {
  object Keys {
    val CONTACT_ID = "Id"
    val ACCOUNT_ID = "AccountId"
    val FIRST_NAME = "FirstName"
    val LAST_NAME = "LastName"
    val USER_ID = "IdentityID__c"
    val DEFAULT_CARD_ID = "Stripe_Default_Card_ID__c"
    val CUSTOMER_ID = "Stripe_Customer_ID__c"
    val TIER = "Membership_Tier__c"
    val REG_NUMBER = "Membership_Number__c"
    val CREATED = "CreatedDate"
    val EMAIL = "Email"
    val BIRTH_DATE = "Birthdate"
    val GENDER = "Gender__c"
    val MAILING_STREET = "MailingStreet"
    val MAILING_CITY = "MailingCity"
    val MAILING_STATE = "MailingState"
    val MAILING_POSTCODE = "MailingPostalCode"
    val MAILING_COUNTRY = "MailingCountry"
    val BILLING_STREET = "OtherStreet"
    val BILLING_CITY = "OtherCity"
    val BILLING_STATE = "OtherState"
    val BILLING_POSTCODE = "OtherPostalCode"
    val BILLING_COUNTRY = "OtherCountry"
    val ALLOW_THIRD_PARTY_EMAIL = "Allow_3rd_Party_Mail__c"
    val ALLOW_GU_RELATED_MAIL = "Allow_Guardian_Related_Mail__c"
    val ALLOW_MEMBERSHIP_MAIL = "Allow_Membership_Mail__c"
  }
}

object Tier extends Enumeration {
  type Tier = Value
  // ordering is important
  val None, Friend, Partner, Patron = Value

  val routeMap = Tier.values.map(t => t.toString.toLowerCase -> t).toMap
}

object MemberDeserializer {
  implicit object TierReader extends Reader[Tier.Tier] {
    def read(value: JValue): Tier.Tier = value match {
      case JNull => Tier.None
      case JString(s) => Tier.withName(s)
      case _ => throw new MappingException("Expected JString")
    }
  }

  implicit object DateTimeReader extends Reader[DateTime] {
    def read(value: JValue): DateTime = value match {
      case JString(s) => new DateTime(s)
      case _ => throw new MappingException("Expected JString")
    }
  }

  // The default OptionReader sucks, it doesn't convert JNull into None
  implicit def optionReader[A](implicit reader: Reader[A], mf: Manifest[A]) = new Reader[Option[A]] {
    def read(value: JValue): Option[A] = value match {
      case JNull | JNothing => None
      case _ => Some(value.as[A])
    }
  }

  implicit object MemberReader extends Reader[Member] {
    import Member.Keys._

    implicit class RichJObject(o: JObject) {
      def get[A](key: String)(implicit reader: Reader[A], mf: Manifest[A]): A =
        o.obj.find { case (k, v) => k == key }.map { case (k, v) => v }.getOrElse(JNothing).as[A]
    }

    def read(value: JValue): Member = value match {
      case o: JObject =>
        o.get[Option[String]](CUSTOMER_ID) match {
          case Some(customerId) => PaidMember(
            o.get[String](CONTACT_ID),
            o.get[String](ACCOUNT_ID),
            o.get[String](USER_ID),
            o.get[Tier.Tier](TIER),
            o.get[String](REG_NUMBER),
            o.get[Option[String]](FIRST_NAME),
            o.get[String](LAST_NAME),
            o.get[DateTime](CREATED),
            customerId
          )
          case None => FreeMember(
            o.get[String](CONTACT_ID),
            o.get[String](ACCOUNT_ID),
            o.get[String](USER_ID),
            o.get[Tier.Tier](TIER),
            o.get[String](REG_NUMBER),
            o.get[Option[String]](FIRST_NAME),
            o.get[String](LAST_NAME),
            o.get[DateTime](CREATED)
          )
        }
      case _ => throw new MappingException("Expected JObject")
    }
  }
}
