package com.gu.membership.salesforce

import com.gu.membership.util.Timing
import com.gu.monitoring.SalesforceMetrics

import scala.concurrent.Future

import dispatch._
import dispatch.Defaults._

import com.ning.http.client.Response

import com.typesafe.scalalogging.slf4j.LazyLogging

import org.json4s._
import org.json4s.native.JsonMethods._
import org.json4s.DefaultReaders._

case class Authentication(token: String, url: String)

case class ScalaforceError(s: String) extends Throwable {
  override def getMessage: String = s
}

trait Scalaforce extends LazyLogging {
  val consumerKey: String
  val consumerSecret: String

  val apiURL: String
  val apiUsername: String
  val apiPassword: String
  val apiToken: String

  val stage: String
  val application: String

  object Status {
    val OK = 200
    val NOT_FOUND = 404
  }

  lazy val metrics = new SalesforceMetrics(stage, application)

  def authentication: Authentication

  implicit val defaultFormats = DefaultFormats

  def toJson(value: Any): JValue = value match {
    case s: String => JString(s)
    case b: Boolean => JBool(b)
    case Some(x) => toJson(x)

    // type erasure so filter any non-string keys at runtime
    case m: Map[_, _] => JObject(m.collect { case (k: String, v: Any) => (k, toJson(v)) }.toList)

    case _ => JNothing
  }

  def httpClient(req: Req): Future[Response] = {
    val request = req.toRequest
    val requestLog = s"${request.getMethod} to ${request.getURI.getPath}"

    metrics.recordRequest()

    Http(req).map { response =>
      metrics.recordResponse(response.getStatusCode, request.getMethod)
      response
    }
  }

  private def urlAuth(endpoint: String) = {
    if (authentication.url.length == 0) {
      metrics.recordAuthenticationError()
      throw ScalaforceError(s"Can't build authenticated request for $endpoint, no Salesforce authentication")
    } else {
      url(s"${authentication.url}/$endpoint").setHeader("Authorization", s"Bearer ${authentication.token}")
    }
  }

  def get(endpoint: String): Future[Response] = httpClient(urlAuth(endpoint).GET)

  def post(endpoint: String, updateData: Map[String, Any]): Future[Response] = {
    val request = urlAuth(endpoint).POST
      .setContentType("application/json", "UTF-8")
      .setBody(compact(render(toJson(updateData))))

    httpClient(request)
  }

  /**
   * This uses the Salesforce Username-Password Flow to get an access token.
   *
   * https://help.salesforce.com/apex/HTViewHelpDoc?id=remoteaccess_oauth_username_password_flow.htm
   * https://www.salesforce.com/us/developer/docs/api_rest/Content/intro_understanding_username_password_oauth_flow.htm
   */
  def getAuthentication: Future[Authentication] = {
    val params = Map(
      "client_id" -> Seq(consumerKey),
      "client_secret" -> Seq(consumerSecret),
      "username" -> Seq(apiUsername),
      "password" -> Seq(apiPassword + apiToken),
      "grant_type" -> Seq("password")
    )
    val request = url(s"$apiURL/services/oauth2/token").POST.setParameters(params)

    Timing.record(metrics, "Authentication") {
      httpClient(request)
    }.map { response =>
      val json = as.json4s.Json(response)
      Authentication((json \ "access_token").as[String], (json \ "instance_url").as[String])
    }
  }

  object Contact {
    def read(key: String, id: String): Future[Option[JValue]] =
      Timing.record(metrics, "Read Contact") {
        get(s"services/data/v29.0/sobjects/Contact/$key/$id")
      }.map { response =>
        response.getStatusCode match {
          case Status.OK => Some(as.json4s.Json(response))
          case Status.NOT_FOUND => None

          case code => throw ScalaforceError(s"Salesforce returned code $code for Contact read $key $id")
        }
      }

    /**
     * We use a custom endpoint to upsert contacts because Salesforce doesn't return enough data
     * on its own. N.B: "newContact" is used both inserts and updates
     */
    def upsert(key: String, id: String, data: Map[String, Any]): Future[JValue] = {
      val updateData = Map("newContact" -> (data + (key -> id)))

      Timing.record(metrics, "Upsert Contact") {
        post("services/apexrest/RegisterCustomer/v1/", updateData)
      }.map { response =>
        val json = as.json4s.Json(response)
        if ((json \ "Success").as[Boolean]) {
          json \ "ContactRecord"
        } else {
          val err = (json \ "ErrorString").as[String]
          throw ScalaforceError(s"Request failed, got error $err")
        }
      }
    }
  }
}
