package com.gu.membership.salesforce

import scala.concurrent.{Promise, Future}
import scala.concurrent.duration._
import scala.concurrent.ExecutionContext.Implicits.global
import akka.actor.ActorSystem

import dispatch.url

import com.gu.membership.util.Timing

import job._
import job.Implicits._

// This is an implementation of the Salesforce Bulk API, its documentation can be found
// here: https://www.salesforce.com/us/developer/docs/api_asynch/
trait ScalaforceJob {
  self: Scalaforce =>

  val system: ActorSystem

  private def request[T <: Result](action: Action[T])(implicit reader: Reader[T]): Future[T] = {
    if (authentication.url.length == 0) {
      metrics.recordAuthenticationError()
      throw ScalaforceError(s"Can't build authenticated request for action ${action.name}")
    }

    val req = url(s"${authentication.url}/services/async/31.0/${action.url}")
      .setHeader("X-SFDC-Session", authentication.token)
      .setContentType("application/xml", "UTF-8")

    val request = action match {
      case read: ReadAction[_] => req.GET
      case create: WriteAction[_] => req.POST.setBody(create.body)
    }

    Timing.record(metrics, action.name) {
      httpClient(request).map { response =>
        reader.read(response) match {
          case Left(error) =>
            logger.debug(s"Salesforce action ${action.name} failed with response code ${response.getStatusCode}")
            logger.debug(response.getResponseBody)
            throw error

          case Right(obj) => obj
        }
      }
    }
  }

  private def getQueryResults(queries: Seq[BatchInfo]): Future[Seq[Map[String, String]]] = {
    val futureRows = queries.map { query =>
      for {
        result <- request(QueryGetResult(query))
        rows <- request(QueryGetRows(query, result))
      } yield rows.records
    }
    Future.reduce(futureRows)(_ ++ _)
  }

  private def completeJob(job: JobInfo): Future[Seq[BatchInfo]] = {
    val promise = Promise[Seq[BatchInfo]]()

    system.scheduler.scheduleOnce(10.seconds) {
      request(JobGetBatchList(job)).map {
        case FailedBatchList(batch) =>
          promise.failure(ScalaforceError(s" Batch ${batch.id} failed in job ${batch.jobId}, ${batch.stateMessage}"))

        case CompletedBatchList(batches) =>
          request(JobClose(job))

          promise.success(batches)

        case InProcessBatchList() => promise.completeWith(completeJob(job))
      }
    }

    promise.future
  }

  object Job {
    def query(objType: String, queries: Seq[String]): Future[Seq[Map[String, String]]] = {
      if (queries.isEmpty) {
        Future.successful(Nil)
      } else {
        Timing.record(metrics, "Query Job") {
          for {
            job <- request(JobCreate("query", objType))
            queries <- Future.sequence(queries.map { q => request(QueryCreate(job, q))})
            _ <- completeJob(job)

            results <- getQueryResults(queries)
          } yield results
        }
      }
    }

  }
}
