/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

import ClayForm, {ClayCheckbox} from '@clayui/form';
import ClayIcon from '@clayui/icon';
import ClayPanel from '@clayui/panel';
import {ClayTooltipProvider} from '@clayui/tooltip';
import {
	FormError,
	Input,
	MultiSelectItem,
} from '@liferay/object-js-components-web';
import {
	ILearnResourceContext,
	InputLocalized,
} from 'frontend-js-components-web';
import React from 'react';

import {NotificationTemplateError} from '../EditNotificationTemplate';
import {PrimaryRecipient} from './PrimaryRecipients';
import {SecondaryRecipient} from './SecondaryRecipients';
import {Sender} from './Sender';

import './EmailNotificationSettings.scss';

interface EmailNotificationSettingsProps {
	emailNotificationRoles: MultiSelectItem[];
	errors: FormError<NotificationTemplate & NotificationTemplateError>;
	learnResources: ILearnResourceContext;
	selectedLocale: Locale;
	setValues: (values: Partial<NotificationTemplate>) => void;
	values: NotificationTemplate;
}

const RECIPIENT_OPTIONS = [
	{
		label: Liferay.Language.get('user-email-address'),
		value: 'email',
	},
	{
		label: Liferay.Language.get('roles'),
		value: 'role',
	},
] as LabelValueObject[];

export function EmailNotificationSettings({
	emailNotificationRoles,
	errors,
	learnResources,
	selectedLocale,
	setValues,
	values,
}: EmailNotificationSettingsProps) {
	return (
		<>
			{Liferay.FeatureFlags['LPD-11165'] ? (
				<div className="lfr__notification-template-email-notification-settings">
					<ClayPanel
						displayTitle={Liferay.Language.get('sender')}
						displayType="unstyled"
					>
						<ClayPanel.Body>
							<Sender
								errors={errors}
								selectedLocale={selectedLocale}
								setValues={setValues}
								values={values}
							/>
						</ClayPanel.Body>
					</ClayPanel>

					<ClayPanel
						displayTitle={Liferay.Language.get(
							'primary-recipients'
						)}
						displayType="unstyled"
					>
						<ClayPanel.Body>
							<PrimaryRecipient
								emailNotificationRoles={emailNotificationRoles}
								errors={errors}
								learnResources={learnResources}
								recipientOptions={RECIPIENT_OPTIONS}
								selectedLocale={selectedLocale}
								setValues={setValues}
								values={values}
							/>
						</ClayPanel.Body>
					</ClayPanel>

					<ClayPanel
						displayTitle={Liferay.Language.get(
							'secondary-recipients'
						)}
						displayType="unstyled"
					>
						<ClayPanel.Body>
							<SecondaryRecipient
								emailNotificationRoles={emailNotificationRoles}
								learnResources={learnResources}
								recipientOptions={RECIPIENT_OPTIONS}
								setValues={setValues}
								values={values}
							/>
						</ClayPanel.Body>
					</ClayPanel>
				</div>
			) : (
				<>
					<InputLocalized
						disabled={values.system}
						error={errors.to}
						label={Liferay.Language.get('to')}
						name="to"
						onChange={(translation) => {
							setValues({
								...values,
								recipients: [
									{
										...values.recipients[0],
										to: translation,
									},
								],
							});
						}}
						placeholder=""
						required
						selectedLocale={selectedLocale}
						translations={
							(values.recipients[0] as EmailRecipients)
								.to as LocalizedValue<string>
						}
					/>

					<ClayForm.Group className="ml-1 row">
						<div className="mr-2">
							<ClayCheckbox
								checked={
									(values.recipients[0] as EmailRecipients)
										.singleRecipient
								}
								disabled={values.system}
								label={Liferay.Language.get(
									'send-emails-separately'
								)}
								onChange={({target: {checked}}) => {
									setValues({
										...values,
										recipients: [
											{
												...values.recipients[0],
												singleRecipient: checked,
											},
										],
									});
								}}
							/>
						</div>

						<ClayTooltipProvider>
							<span
								title={Liferay.Language.get(
									'each-to-recipient-will-receive-separate-emails'
								)}
							>
								<ClayIcon
									className="lfr__notification-template-email-notification-settings-tooltip-icon"
									symbol="question-circle-full"
								/>
							</span>
						</ClayTooltipProvider>
					</ClayForm.Group>

					<div className="row">
						<div className="col-lg-6">
							<Input
								disabled={values.system}
								label={Liferay.Language.get('cc')}
								name="cc"
								onChange={({target}) =>
									setValues({
										...values,
										recipients: [
											{
												...values.recipients[0],
												cc: target.value,
											},
										],
									})
								}
								value={
									(values.recipients[0] as EmailRecipients)
										.cc as string
								}
							/>
						</div>

						<div className="col-lg-6">
							<Input
								disabled={values.system}
								label={Liferay.Language.get('bcc')}
								name="bcc"
								onChange={({target}) =>
									setValues({
										...values,
										recipients: [
											{
												...values.recipients[0],
												bcc: target.value,
											},
										],
									})
								}
								value={
									(values.recipients[0] as EmailRecipients)
										.bcc as string
								}
							/>
						</div>
					</div>

					<div className="row">
						<div className="col-lg-6">
							<Input
								disabled={values.system}
								error={errors.from}
								label={Liferay.Language.get('from-address')}
								name="fromAddress"
								onChange={({target}) =>
									setValues({
										...values,
										recipients: [
											{
												...values.recipients[0],
												from: target.value,
											},
										],
									})
								}
								required
								value={
									(values.recipients[0] as EmailRecipients)
										.from
								}
							/>
						</div>

						<div className="col-lg-6">
							<InputLocalized
								disabled={values.system}
								error={errors.fromName}
								label={Liferay.Language.get('from-name')}
								name="fromName"
								onChange={(translation) => {
									setValues({
										...values,
										recipients: [
											{
												...values.recipients[0],
												fromName: translation,
											},
										],
									});
								}}
								placeholder=""
								required
								selectedLocale={selectedLocale}
								translations={
									(values.recipients[0] as EmailRecipients)
										.fromName
								}
							/>
						</div>
					</div>
				</>
			)}
		</>
	);
}
