/**
 * SPDX-FileCopyrightText: (c) 2000 Liferay, Inc. https://liferay.com
 * SPDX-License-Identifier: LGPL-2.1-or-later OR LicenseRef-Liferay-DXP-EULA-2.0.0-2023-06
 */

import {useMutation} from '@apollo/client';
import ClayButton from '@clayui/button';
import ClayForm, {ClayInput} from '@clayui/form';
import ClayModal, {useModal} from '@clayui/modal';
import React, {useContext, useRef} from 'react';

import {AppContext} from '../AppContext.es';
import {createSubTopicQuery, createTopicQuery} from '../utils/client.es';
import lang from '../utils/lang.es';
import {deleteCacheVariables} from '../utils/utils.es';

export default ({
	currentSectionId,
	onClose,
	onCreateNavigateTo,
	setError,
	visible,
}) => {
	const context = useContext(AppContext);
	const topicName = useRef(null);
	const topicDescription = useRef(null);

	const [createNewSubTopic] = useMutation(createSubTopicQuery, {
		onCompleted(data) {
			onCreateNavigateTo(
				context.useTopicNamesInURL
					? data.createMessageBoardSectionMessageBoardSection.title
					: data.createMessageBoardSectionMessageBoardSection.id
			);
		},
		update(proxy) {
			deleteCacheVariables(proxy, 'MessageBoardSection');
			proxy.gc();
		},
	});

	const [createNewTopic] = useMutation(createTopicQuery, {
		onCompleted(data) {
			onCreateNavigateTo(
				context.useTopicNamesInURL
					? data.createSiteMessageBoardSection.title
					: data.createSiteMessageBoardSection.id
			);
		},
		update(proxy) {
			deleteCacheVariables(proxy, 'MessageBoardSection');
			deleteCacheVariables(proxy, 'ROOT_QUERY');
			proxy.gc();
		},
	});

	const isValidTopic = (topic) => {
		const hyphens = /-+/g;
		if (hyphens.test(topic)) {
			const error = {
				message: lang.sub(
					Liferay.Language.get(
						'the-x-cannot-contain-the-following-invalid-characters-x'
					),
					[
						Liferay.Language.get('topic-name'),
						' - & \' @ \\\\ ] } : , = > / < \\n [ {  | + # ` ? \\" \\r ; / * ~',
					]
				),
			};
			setError(error);

			return false;
		}

		return true;
	};

	const createTopic = () => {
		if (isValidTopic(topicName.current.value)) {
			if (currentSectionId) {
				createNewSubTopic({
					variables: {
						description: topicDescription.current.value,
						parentMessageBoardSectionId: currentSectionId,
						title: topicName.current.value,
					},
				});
			}
			else {
				createNewTopic({
					variables: {
						description: topicDescription.current.value,
						siteKey: context.siteKey,
						title: topicName.current.value,
					},
				});
			}
		}
	};

	const {observer, onClose: close} = useModal({
		onClose,
	});

	return (
		<>
			{visible && (
				<ClayModal observer={observer} size="lg" status="info">
					<ClayModal.Header>
						{Liferay.Language.get('new-topic')}
					</ClayModal.Header>
					<ClayModal.Body>
						<ClayForm>
							<ClayForm.Group className="form-group-sm">
								<label htmlFor="basicInput">
									{Liferay.Language.get('topic-name')}
								</label>
								<ClayInput
									placeholder={Liferay.Language.get(
										'please-enter-a-valid-topic-name'
									)}
									ref={topicName}
									type="text"
								/>
							</ClayForm.Group>
							<ClayForm.Group className="form-group-sm">
								<label htmlFor="basicInput">
									{Liferay.Language.get('description')}
								</label>
								<ClayInput
									className="form-control"
									component="textarea"
									placeholder={Liferay.Language.get(
										'description'
									)}
									ref={topicDescription}
								/>
							</ClayForm.Group>
						</ClayForm>
					</ClayModal.Body>
					<ClayModal.Footer
						last={
							<ClayButton.Group spaced>
								<ClayButton
									displayType="secondary"
									onClick={close}
								>
									{Liferay.Language.get('cancel')}
								</ClayButton>
								<ClayButton
									displayType="primary"
									onClick={() => {
										createTopic();
										close();
									}}
								>
									{Liferay.Language.get('create')}
								</ClayButton>
							</ClayButton.Group>
						}
					/>
				</ClayModal>
			)}
		</>
	);
};
