/*
 * Decompiled with CFR 0.152.
 */
package org.apache.curator;

import java.io.Closeable;
import java.io.IOException;
import java.util.Queue;
import java.util.concurrent.Callable;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.curator.CuratorConnectionLossException;
import org.apache.curator.HandleHolder;
import org.apache.curator.SessionFailRetryLoop;
import org.apache.curator.connection.ConnectionHandlingPolicy;
import org.apache.curator.drivers.EventTrace;
import org.apache.curator.drivers.OperationTrace;
import org.apache.curator.drivers.TracerDriver;
import org.apache.curator.ensemble.EnsembleProvider;
import org.apache.curator.utils.CloseableUtils;
import org.apache.curator.utils.ThreadUtils;
import org.apache.curator.utils.ZookeeperFactory;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.ZooKeeper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ConnectionState
implements Watcher,
Closeable {
    private static final int MAX_BACKGROUND_EXCEPTIONS = 10;
    private static final boolean LOG_EVENTS = Boolean.getBoolean("curator-log-events");
    private static final Logger log = LoggerFactory.getLogger(ConnectionState.class);
    private final HandleHolder zooKeeper;
    private final AtomicBoolean isConnected = new AtomicBoolean(false);
    private final AtomicInteger lastNegotiatedSessionTimeoutMs = new AtomicInteger(0);
    private final EnsembleProvider ensembleProvider;
    private final int sessionTimeoutMs;
    private final int connectionTimeoutMs;
    private final AtomicReference<TracerDriver> tracer;
    private final ConnectionHandlingPolicy connectionHandlingPolicy;
    private final Queue<Exception> backgroundExceptions = new ConcurrentLinkedQueue<Exception>();
    private final Queue<Watcher> parentWatchers = new ConcurrentLinkedQueue<Watcher>();
    private final AtomicLong instanceIndex = new AtomicLong();
    private volatile long connectionStartMs = 0L;

    ConnectionState(ZookeeperFactory zookeeperFactory, EnsembleProvider ensembleProvider, int sessionTimeoutMs, int connectionTimeoutMs, Watcher parentWatcher, AtomicReference<TracerDriver> tracer, boolean canBeReadOnly, ConnectionHandlingPolicy connectionHandlingPolicy) {
        this.ensembleProvider = ensembleProvider;
        this.sessionTimeoutMs = sessionTimeoutMs;
        this.connectionTimeoutMs = connectionTimeoutMs;
        this.tracer = tracer;
        this.connectionHandlingPolicy = connectionHandlingPolicy;
        if (parentWatcher != null) {
            this.parentWatchers.offer(parentWatcher);
        }
        this.zooKeeper = new HandleHolder(zookeeperFactory, this, ensembleProvider, sessionTimeoutMs, canBeReadOnly);
    }

    ZooKeeper getZooKeeper() throws Exception {
        if (SessionFailRetryLoop.sessionForThreadHasFailed()) {
            throw new SessionFailRetryLoop.SessionFailedException();
        }
        Exception exception = this.backgroundExceptions.poll();
        if (exception != null) {
            new EventTrace("background-exceptions", this.tracer.get()).commit();
            throw exception;
        }
        boolean localIsConnected = this.isConnected.get();
        if (!localIsConnected) {
            this.checkTimeouts();
        }
        return this.zooKeeper.getZooKeeper();
    }

    boolean isConnected() {
        return this.isConnected.get();
    }

    void start() throws Exception {
        log.debug("Starting");
        this.ensembleProvider.start();
        this.reset();
    }

    @Override
    public void close() throws IOException {
        log.debug("Closing");
        CloseableUtils.closeQuietly(this.ensembleProvider);
        try {
            this.zooKeeper.closeAndClear();
        }
        catch (Exception e2) {
            ThreadUtils.checkInterrupted(e2);
            throw new IOException(e2);
        }
        finally {
            this.isConnected.set(false);
        }
    }

    void addParentWatcher(Watcher watcher) {
        this.parentWatchers.offer(watcher);
    }

    void removeParentWatcher(Watcher watcher) {
        this.parentWatchers.remove(watcher);
    }

    long getInstanceIndex() {
        return this.instanceIndex.get();
    }

    int getLastNegotiatedSessionTimeoutMs() {
        return this.lastNegotiatedSessionTimeoutMs.get();
    }

    @Override
    public void process(WatchedEvent event) {
        if (LOG_EVENTS) {
            log.debug("ConnectState watcher: " + event);
        }
        if (event.getType() == Watcher.Event.EventType.None) {
            boolean wasConnected = this.isConnected.get();
            boolean newIsConnected = this.checkState(event.getState(), wasConnected);
            if (newIsConnected != wasConnected) {
                this.isConnected.set(newIsConnected);
                this.connectionStartMs = System.currentTimeMillis();
                if (newIsConnected) {
                    this.lastNegotiatedSessionTimeoutMs.set(this.zooKeeper.getNegotiatedSessionTimeoutMs());
                    log.debug("Negotiated session timeout: " + this.lastNegotiatedSessionTimeoutMs.get());
                }
            }
        }
        for (Watcher parentWatcher : this.parentWatchers) {
            OperationTrace trace = new OperationTrace("connection-state-parent-process", this.tracer.get(), this.getSessionId());
            parentWatcher.process(event);
            trace.commit();
        }
    }

    EnsembleProvider getEnsembleProvider() {
        return this.ensembleProvider;
    }

    synchronized void reset() throws Exception {
        log.debug("reset");
        this.instanceIndex.incrementAndGet();
        this.isConnected.set(false);
        this.connectionStartMs = System.currentTimeMillis();
        this.zooKeeper.closeAndReset();
        this.zooKeeper.getZooKeeper();
    }

    private synchronized void checkTimeouts() throws Exception {
        final AtomicReference newConnectionString = new AtomicReference();
        Callable<String> hasNewConnectionString = new Callable<String>(){

            @Override
            public String call() {
                newConnectionString.set(ConnectionState.this.zooKeeper.getNewConnectionString());
                return (String)newConnectionString.get();
            }
        };
        int lastNegotiatedSessionTimeoutMs = this.getLastNegotiatedSessionTimeoutMs();
        int useSessionTimeoutMs = lastNegotiatedSessionTimeoutMs > 0 ? lastNegotiatedSessionTimeoutMs : this.sessionTimeoutMs;
        ConnectionHandlingPolicy.CheckTimeoutsResult result = this.connectionHandlingPolicy.checkTimeouts(hasNewConnectionString, this.connectionStartMs, useSessionTimeoutMs, this.connectionTimeoutMs);
        switch (result) {
            default: {
                break;
            }
            case NEW_CONNECTION_STRING: {
                this.handleNewConnectionString((String)newConnectionString.get());
                break;
            }
            case RESET_CONNECTION: {
                if (!Boolean.getBoolean("curator-dont-log-connection-problems")) {
                    long elapsed = System.currentTimeMillis() - this.connectionStartMs;
                    int maxTimeout = Math.max(useSessionTimeoutMs, this.connectionTimeoutMs);
                    log.warn(String.format("Connection attempt unsuccessful after %d (greater than max timeout of %d). Resetting connection and trying again with a new connection.", elapsed, maxTimeout));
                }
                this.reset();
                break;
            }
            case CONNECTION_TIMEOUT: {
                CuratorConnectionLossException connectionLossException = new CuratorConnectionLossException();
                if (!Boolean.getBoolean("curator-dont-log-connection-problems")) {
                    long elapsed = System.currentTimeMillis() - this.connectionStartMs;
                    log.error(String.format("Connection timed out for connection string (%s) and timeout (%d) / elapsed (%d)", this.zooKeeper.getConnectionString(), this.connectionTimeoutMs, elapsed), connectionLossException);
                }
                new EventTrace("connections-timed-out", this.tracer.get(), this.getSessionId()).commit();
                throw connectionLossException;
            }
            case SESSION_TIMEOUT: {
                this.handleExpiredSession();
            }
        }
    }

    public long getSessionId() {
        long sessionId = 0L;
        try {
            ZooKeeper zk = this.zooKeeper.getZooKeeper();
            if (zk != null) {
                sessionId = zk.getSessionId();
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return sessionId;
    }

    private boolean checkState(Watcher.Event.KeeperState state, boolean wasConnected) {
        String newConnectionString;
        boolean isConnected = wasConnected;
        boolean checkNewConnectionString = true;
        switch (state) {
            default: {
                isConnected = false;
                break;
            }
            case SyncConnected: 
            case ConnectedReadOnly: {
                isConnected = true;
                break;
            }
            case AuthFailed: {
                isConnected = false;
                log.error("Authentication failed");
                break;
            }
            case Expired: {
                isConnected = false;
                checkNewConnectionString = false;
                this.handleExpiredSession();
            }
            case SaslAuthenticated: 
        }
        if (state != Watcher.Event.KeeperState.Expired) {
            new EventTrace(state.toString(), this.tracer.get(), this.getSessionId()).commit();
        }
        if (checkNewConnectionString && (newConnectionString = this.zooKeeper.getNewConnectionString()) != null) {
            this.handleNewConnectionString(newConnectionString);
        }
        return isConnected;
    }

    private void handleNewConnectionString(String newConnectionString) {
        log.info("Connection string changed to: " + newConnectionString);
        new EventTrace("connection-string-changed", this.tracer.get(), this.getSessionId()).commit();
        try {
            ZooKeeper zooKeeper = this.zooKeeper.getZooKeeper();
            if (zooKeeper == null) {
                log.warn("Could not update the connection string because getZooKeeper() returned null.");
            } else if (this.ensembleProvider.updateServerListEnabled()) {
                zooKeeper.updateServerList(newConnectionString);
            } else {
                this.reset();
            }
        }
        catch (Exception e2) {
            ThreadUtils.checkInterrupted(e2);
            this.queueBackgroundException(e2);
        }
    }

    private void handleExpiredSession() {
        log.warn("Session expired event received");
        new EventTrace("session-expired", this.tracer.get(), this.getSessionId()).commit();
        try {
            this.reset();
        }
        catch (Exception e2) {
            ThreadUtils.checkInterrupted(e2);
            this.queueBackgroundException(e2);
        }
    }

    private void queueBackgroundException(Exception e2) {
        while (this.backgroundExceptions.size() >= 10) {
            this.backgroundExceptions.poll();
        }
        this.backgroundExceptions.offer(e2);
    }
}

