/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.producer.internals;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import org.apache.kafka.clients.ApiVersions;
import org.apache.kafka.clients.ClientRequest;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.KafkaClient;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.clients.NetworkClientUtils;
import org.apache.kafka.clients.RequestCompletionHandler;
import org.apache.kafka.clients.producer.internals.ProducerBatch;
import org.apache.kafka.clients.producer.internals.ProducerIdAndEpoch;
import org.apache.kafka.clients.producer.internals.RecordAccumulator;
import org.apache.kafka.clients.producer.internals.SenderMetricsRegistry;
import org.apache.kafka.clients.producer.internals.TransactionManager;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.MetricName;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.ApiException;
import org.apache.kafka.common.errors.AuthenticationException;
import org.apache.kafka.common.errors.ClusterAuthorizationException;
import org.apache.kafka.common.errors.InvalidMetadataException;
import org.apache.kafka.common.errors.OutOfOrderSequenceException;
import org.apache.kafka.common.errors.ProducerFencedException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.TopicAuthorizationException;
import org.apache.kafka.common.errors.TransactionalIdAuthorizationException;
import org.apache.kafka.common.errors.UnknownTopicOrPartitionException;
import org.apache.kafka.common.errors.UnsupportedVersionException;
import org.apache.kafka.common.metrics.Measurable;
import org.apache.kafka.common.metrics.MetricConfig;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Max;
import org.apache.kafka.common.metrics.stats.Meter;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.requests.AbstractRequest;
import org.apache.kafka.common.requests.InitProducerIdRequest;
import org.apache.kafka.common.requests.InitProducerIdResponse;
import org.apache.kafka.common.requests.ProduceRequest;
import org.apache.kafka.common.requests.ProduceResponse;
import org.apache.kafka.common.requests.RequestHeader;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;

public class Sender
implements Runnable {
    private final Logger log;
    private final KafkaClient client;
    private final RecordAccumulator accumulator;
    private final Metadata metadata;
    private final boolean guaranteeMessageOrder;
    private final int maxRequestSize;
    private final short acks;
    private final int retries;
    private final Time time;
    private volatile boolean running;
    private volatile boolean forceClose;
    private final SenderMetrics sensors;
    private final int requestTimeout;
    private final long retryBackoffMs;
    private final ApiVersions apiVersions;
    private final TransactionManager transactionManager;

    public Sender(LogContext logContext, KafkaClient client, Metadata metadata, RecordAccumulator accumulator, boolean guaranteeMessageOrder, int maxRequestSize, short acks, int retries, SenderMetricsRegistry metricsRegistry, Time time, int requestTimeout, long retryBackoffMs, TransactionManager transactionManager, ApiVersions apiVersions) {
        this.log = logContext.logger(Sender.class);
        this.client = client;
        this.accumulator = accumulator;
        this.metadata = metadata;
        this.guaranteeMessageOrder = guaranteeMessageOrder;
        this.maxRequestSize = maxRequestSize;
        this.running = true;
        this.acks = acks;
        this.retries = retries;
        this.time = time;
        this.sensors = new SenderMetrics(metricsRegistry);
        this.requestTimeout = requestTimeout;
        this.retryBackoffMs = retryBackoffMs;
        this.apiVersions = apiVersions;
        this.transactionManager = transactionManager;
    }

    @Override
    public void run() {
        this.log.debug("Starting Kafka producer I/O thread.");
        while (this.running) {
            try {
                this.run(this.time.milliseconds());
            }
            catch (Exception e2) {
                this.log.error("Uncaught error in kafka producer I/O thread: ", e2);
            }
        }
        this.log.debug("Beginning shutdown of Kafka producer I/O thread, sending remaining records.");
        while (!this.forceClose && (this.accumulator.hasUndrained() || this.client.inFlightRequestCount() > 0)) {
            try {
                this.run(this.time.milliseconds());
            }
            catch (Exception e3) {
                this.log.error("Uncaught error in kafka producer I/O thread: ", e3);
            }
        }
        if (this.forceClose) {
            this.log.debug("Aborting incomplete batches due to forced shutdown");
            this.accumulator.abortIncompleteBatches();
        }
        try {
            this.client.close();
        }
        catch (Exception e4) {
            this.log.error("Failed to close network client", e4);
        }
        this.log.debug("Shutdown of Kafka producer I/O thread has completed.");
    }

    void run(long now) {
        if (this.transactionManager != null) {
            try {
                if (this.transactionManager.shouldResetProducerStateAfterResolvingSequences()) {
                    this.transactionManager.resetProducerId();
                }
                if (!this.transactionManager.isTransactional()) {
                    this.maybeWaitForProducerId();
                } else if (this.transactionManager.hasUnresolvedSequences() && !this.transactionManager.hasFatalError()) {
                    this.transactionManager.transitionToFatalError(new KafkaException("The client hasn't received acknowledgment for some previously sent messages and can no longer retry them. It isn't safe to continue."));
                } else if (this.transactionManager.hasInFlightTransactionalRequest() || this.maybeSendTransactionalRequest(now)) {
                    this.client.poll(this.retryBackoffMs, now);
                    return;
                }
                if (this.transactionManager.hasFatalError() || !this.transactionManager.hasProducerId()) {
                    RuntimeException lastError = this.transactionManager.lastError();
                    if (lastError != null) {
                        this.maybeAbortBatches(lastError);
                    }
                    this.client.poll(this.retryBackoffMs, now);
                    return;
                }
                if (this.transactionManager.hasAbortableError()) {
                    this.accumulator.abortUndrainedBatches(this.transactionManager.lastError());
                }
            }
            catch (AuthenticationException e2) {
                this.log.trace("Authentication exception while processing transactional request: {}", e2);
                this.transactionManager.authenticationFailed(e2);
            }
        }
        long pollTimeout = this.sendProducerData(now);
        this.client.poll(pollTimeout, now);
    }

    private long sendProducerData(long now) {
        List<ProducerBatch> expiredBatches;
        Cluster cluster = this.metadata.fetch();
        RecordAccumulator.ReadyCheckResult result = this.accumulator.ready(cluster, now);
        if (!result.unknownLeaderTopics.isEmpty()) {
            for (String topic : result.unknownLeaderTopics) {
                this.metadata.add(topic);
            }
            this.metadata.requestUpdate();
        }
        Iterator<Node> iter = result.readyNodes.iterator();
        long notReadyTimeout = Long.MAX_VALUE;
        while (iter.hasNext()) {
            Node node = iter.next();
            if (this.client.ready(node, now)) continue;
            iter.remove();
            notReadyTimeout = Math.min(notReadyTimeout, this.client.connectionDelay(node, now));
        }
        Map<Integer, List<ProducerBatch>> batches = this.accumulator.drain(cluster, result.readyNodes, this.maxRequestSize, now);
        if (this.guaranteeMessageOrder) {
            for (List<ProducerBatch> batchList : batches.values()) {
                for (ProducerBatch batch : batchList) {
                    this.accumulator.mutePartition(batch.topicPartition);
                }
            }
        }
        if (!(expiredBatches = this.accumulator.expiredBatches(this.requestTimeout, now)).isEmpty()) {
            this.log.trace("Expired {} batches in accumulator", (Object)expiredBatches.size());
        }
        for (ProducerBatch expiredBatch : expiredBatches) {
            this.failBatch(expiredBatch, -1L, -1L, expiredBatch.timeoutException(), false);
            if (this.transactionManager == null || !expiredBatch.inRetry()) continue;
            this.transactionManager.markSequenceUnresolved(expiredBatch.topicPartition);
        }
        this.sensors.updateProduceRequestMetrics(batches);
        long pollTimeout = Math.min(result.nextReadyCheckDelayMs, notReadyTimeout);
        if (!result.readyNodes.isEmpty()) {
            this.log.trace("Nodes with data ready to send: {}", (Object)result.readyNodes);
            pollTimeout = 0L;
        }
        this.sendProduceRequests(batches, now);
        return pollTimeout;
    }

    private boolean maybeSendTransactionalRequest(long now) {
        TransactionManager.TxnRequestHandler nextRequestHandler;
        if (this.transactionManager.isCompleting() && this.accumulator.hasIncomplete()) {
            if (this.transactionManager.isAborting()) {
                this.accumulator.abortUndrainedBatches(new KafkaException("Failing batch since transaction was aborted"));
            }
            if (!this.accumulator.flushInProgress()) {
                this.accumulator.beginFlush();
            }
        }
        if ((nextRequestHandler = this.transactionManager.nextRequestHandler(this.accumulator.hasIncomplete())) == null) {
            return false;
        }
        AbstractRequest.Builder<?> requestBuilder = nextRequestHandler.requestBuilder();
        while (true) {
            block13: {
                Node targetNode = null;
                try {
                    if (nextRequestHandler.needsCoordinator()) {
                        targetNode = this.transactionManager.coordinator(nextRequestHandler.coordinatorType());
                        if (targetNode == null) {
                            this.transactionManager.lookupCoordinator(nextRequestHandler);
                            break;
                        }
                        if (!NetworkClientUtils.awaitReady(this.client, targetNode, this.time, this.requestTimeout)) {
                            this.transactionManager.lookupCoordinator(nextRequestHandler);
                            break;
                        }
                    } else {
                        targetNode = this.awaitLeastLoadedNodeReady(this.requestTimeout);
                    }
                    if (targetNode != null) {
                        if (nextRequestHandler.isRetry()) {
                            this.time.sleep(nextRequestHandler.retryBackoffMs());
                        }
                        ClientRequest clientRequest = this.client.newClientRequest(targetNode.idString(), requestBuilder, now, true, nextRequestHandler);
                        this.transactionManager.setInFlightTransactionalRequestCorrelationId(clientRequest.correlationId());
                        this.log.debug("Sending transactional request {} to node {}", (Object)requestBuilder, (Object)targetNode);
                        this.client.send(clientRequest, now);
                        return true;
                    }
                }
                catch (IOException e2) {
                    this.log.debug("Disconnect from {} while trying to send request {}. Going to back off and retry", (Object)targetNode, (Object)requestBuilder);
                    if (!nextRequestHandler.needsCoordinator()) break block13;
                    this.transactionManager.lookupCoordinator(nextRequestHandler);
                    break;
                }
            }
            this.time.sleep(this.retryBackoffMs);
            this.metadata.requestUpdate();
        }
        this.transactionManager.retry(nextRequestHandler);
        return true;
    }

    private void maybeAbortBatches(RuntimeException exception) {
        if (this.accumulator.hasIncomplete()) {
            this.log.error("Aborting producer batches due to fatal error", exception);
            this.accumulator.abortBatches(exception);
        }
    }

    public void initiateClose() {
        this.accumulator.close();
        this.running = false;
        this.wakeup();
    }

    public void forceClose() {
        this.forceClose = true;
        this.initiateClose();
    }

    private ClientResponse sendAndAwaitInitProducerIdRequest(Node node) throws IOException {
        String nodeId = node.idString();
        InitProducerIdRequest.Builder builder = new InitProducerIdRequest.Builder(null);
        ClientRequest request = this.client.newClientRequest(nodeId, builder, this.time.milliseconds(), true, null);
        return NetworkClientUtils.sendAndReceive(this.client, request, this.time);
    }

    private Node awaitLeastLoadedNodeReady(long remainingTimeMs) throws IOException {
        Node node = this.client.leastLoadedNode(this.time.milliseconds());
        if (node != null && NetworkClientUtils.awaitReady(this.client, node, this.time, remainingTimeMs)) {
            return node;
        }
        return null;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private void maybeWaitForProducerId() {
        while (!this.transactionManager.hasProducerId()) {
            if (this.transactionManager.hasError()) return;
            try {
                Node node = this.awaitLeastLoadedNodeReady(this.requestTimeout);
                if (node != null) {
                    ClientResponse response = this.sendAndAwaitInitProducerIdRequest(node);
                    InitProducerIdResponse initProducerIdResponse = (InitProducerIdResponse)response.responseBody();
                    Errors error = initProducerIdResponse.error();
                    if (error == Errors.NONE) {
                        ProducerIdAndEpoch producerIdAndEpoch = new ProducerIdAndEpoch(initProducerIdResponse.producerId(), initProducerIdResponse.epoch());
                        this.transactionManager.setProducerIdAndEpoch(producerIdAndEpoch);
                        return;
                    }
                    if (!(error.exception() instanceof RetriableException)) {
                        this.transactionManager.transitionToFatalError(error.exception());
                        return;
                    }
                    this.log.debug("Retriable error from InitProducerId response", (Object)error.message());
                } else {
                    this.log.debug("Could not find an available broker to send InitProducerIdRequest to. We will back off and try again.");
                }
            }
            catch (UnsupportedVersionException e2) {
                this.transactionManager.transitionToFatalError(e2);
                return;
            }
            catch (IOException e3) {
                this.log.debug("Broker {} disconnected while awaiting InitProducerId response", e3);
            }
            this.log.trace("Retry InitProducerIdRequest in {}ms.", (Object)this.retryBackoffMs);
            this.time.sleep(this.retryBackoffMs);
            this.metadata.requestUpdate();
        }
    }

    private void handleProduceResponse(ClientResponse response, Map<TopicPartition, ProducerBatch> batches, long now) {
        RequestHeader requestHeader = response.requestHeader();
        int correlationId = requestHeader.correlationId();
        if (response.wasDisconnected()) {
            this.log.trace("Cancelled request with header {} due to node {} being disconnected", (Object)requestHeader, (Object)response.destination());
            for (ProducerBatch batch : batches.values()) {
                this.completeBatch(batch, new ProduceResponse.PartitionResponse(Errors.NETWORK_EXCEPTION), correlationId, now);
            }
        } else if (response.versionMismatch() != null) {
            this.log.warn("Cancelled request {} due to a version mismatch with node {}", response, response.destination(), response.versionMismatch());
            for (ProducerBatch batch : batches.values()) {
                this.completeBatch(batch, new ProduceResponse.PartitionResponse(Errors.UNSUPPORTED_VERSION), correlationId, now);
            }
        } else {
            this.log.trace("Received produce response from node {} with correlation id {}", (Object)response.destination(), (Object)correlationId);
            if (response.hasResponse()) {
                ProduceResponse produceResponse = (ProduceResponse)response.responseBody();
                for (Map.Entry<TopicPartition, ProduceResponse.PartitionResponse> entry : produceResponse.responses().entrySet()) {
                    TopicPartition tp = entry.getKey();
                    ProduceResponse.PartitionResponse partResp = entry.getValue();
                    ProducerBatch batch = batches.get(tp);
                    this.completeBatch(batch, partResp, correlationId, now);
                }
                this.sensors.recordLatency(response.destination(), response.requestLatencyMs());
            } else {
                for (ProducerBatch batch : batches.values()) {
                    this.completeBatch(batch, new ProduceResponse.PartitionResponse(Errors.NONE), correlationId, now);
                }
            }
        }
    }

    private void completeBatch(ProducerBatch batch, ProduceResponse.PartitionResponse response, long correlationId, long now) {
        Errors error = response.error;
        if (error == Errors.MESSAGE_TOO_LARGE && batch.recordCount > 1 && (batch.magic() >= 2 || batch.isCompressed())) {
            this.log.warn("Got error produce response in correlation id {} on topic-partition {}, splitting and retrying ({} attempts left). Error: {}", new Object[]{correlationId, batch.topicPartition, this.retries - batch.attempts(), error});
            if (this.transactionManager != null) {
                this.transactionManager.removeInFlightBatch(batch);
            }
            this.accumulator.splitAndReenqueue(batch);
            this.accumulator.deallocate(batch);
            this.sensors.recordBatchSplit();
        } else if (error != Errors.NONE) {
            if (this.canRetry(batch, response)) {
                this.log.warn("Got error produce response with correlation id {} on topic-partition {}, retrying ({} attempts left). Error: {}", new Object[]{correlationId, batch.topicPartition, this.retries - batch.attempts() - 1, error});
                if (this.transactionManager == null) {
                    this.reenqueueBatch(batch, now);
                } else if (this.transactionManager.hasProducerIdAndEpoch(batch.producerId(), batch.producerEpoch())) {
                    this.log.debug("Retrying batch to topic-partition {}. ProducerId: {}; Sequence number : {}", batch.topicPartition, batch.producerId(), batch.baseSequence());
                    this.reenqueueBatch(batch, now);
                } else {
                    this.failBatch(batch, response, new OutOfOrderSequenceException("Attempted to retry sending a batch but the producer id changed from " + batch.producerId() + " to " + this.transactionManager.producerIdAndEpoch().producerId + " in the mean time. This batch will be dropped."), false);
                }
            } else if (error == Errors.DUPLICATE_SEQUENCE_NUMBER) {
                this.completeBatch(batch, response);
            } else {
                ApiException exception = error == Errors.TOPIC_AUTHORIZATION_FAILED ? new TopicAuthorizationException(batch.topicPartition.topic()) : (error == Errors.CLUSTER_AUTHORIZATION_FAILED ? new ClusterAuthorizationException("The producer is not authorized to do idempotent sends") : error.exception());
                this.failBatch(batch, response, exception, batch.attempts() < this.retries);
            }
            if (error.exception() instanceof InvalidMetadataException) {
                if (error.exception() instanceof UnknownTopicOrPartitionException) {
                    this.log.warn("Received unknown topic or partition error in produce request on partition {}. The topic/partition may not exist or the user may not have Describe access to it", (Object)batch.topicPartition);
                }
                this.metadata.requestUpdate();
            }
        } else {
            this.completeBatch(batch, response);
        }
        if (this.guaranteeMessageOrder) {
            this.accumulator.unmutePartition(batch.topicPartition);
        }
    }

    private void reenqueueBatch(ProducerBatch batch, long currentTimeMs) {
        this.accumulator.reenqueue(batch, currentTimeMs);
        this.sensors.recordRetries(batch.topicPartition.topic(), batch.recordCount);
    }

    private void completeBatch(ProducerBatch batch, ProduceResponse.PartitionResponse response) {
        if (this.transactionManager != null) {
            if (this.transactionManager.hasProducerIdAndEpoch(batch.producerId(), batch.producerEpoch())) {
                this.transactionManager.maybeUpdateLastAckedSequence(batch.topicPartition, batch.baseSequence() + batch.recordCount - 1);
                this.log.debug("ProducerId: {}; Set last ack'd sequence number for topic-partition {} to {}", batch.producerId(), batch.topicPartition, this.transactionManager.lastAckedSequence(batch.topicPartition));
            }
            this.transactionManager.updateLastAckedOffset(response, batch);
            this.transactionManager.removeInFlightBatch(batch);
        }
        if (batch.done(response.baseOffset, response.logAppendTime, null)) {
            this.accumulator.deallocate(batch);
        }
    }

    private void failBatch(ProducerBatch batch, ProduceResponse.PartitionResponse response, RuntimeException exception, boolean adjustSequenceNumbers) {
        this.failBatch(batch, response.baseOffset, response.logAppendTime, exception, adjustSequenceNumbers);
    }

    private void failBatch(ProducerBatch batch, long baseOffset, long logAppendTime, RuntimeException exception, boolean adjustSequenceNumbers) {
        if (this.transactionManager != null) {
            if (exception instanceof OutOfOrderSequenceException && !this.transactionManager.isTransactional() && this.transactionManager.hasProducerId(batch.producerId())) {
                this.log.error("The broker returned {} for topic-partition {} at offset {}. This indicates data loss on the broker, and should be investigated.", exception, batch.topicPartition, baseOffset);
                this.transactionManager.resetProducerId();
            } else if (exception instanceof ClusterAuthorizationException || exception instanceof TransactionalIdAuthorizationException || exception instanceof ProducerFencedException || exception instanceof UnsupportedVersionException) {
                this.transactionManager.transitionToFatalError(exception);
            } else if (this.transactionManager.isTransactional()) {
                this.transactionManager.transitionToAbortableError(exception);
            }
            this.transactionManager.removeInFlightBatch(batch);
            if (adjustSequenceNumbers) {
                this.transactionManager.adjustSequencesDueToFailedBatch(batch);
            }
        }
        this.sensors.recordErrors(batch.topicPartition.topic(), batch.recordCount);
        if (batch.done(baseOffset, logAppendTime, exception)) {
            this.accumulator.deallocate(batch);
        }
    }

    private boolean canRetry(ProducerBatch batch, ProduceResponse.PartitionResponse response) {
        return batch.attempts() < this.retries && (response.error.exception() instanceof RetriableException || this.transactionManager != null && this.transactionManager.canRetry(response, batch));
    }

    private void sendProduceRequests(Map<Integer, List<ProducerBatch>> collated, long now) {
        for (Map.Entry<Integer, List<ProducerBatch>> entry : collated.entrySet()) {
            this.sendProduceRequest(now, entry.getKey(), this.acks, this.requestTimeout, entry.getValue());
        }
    }

    private void sendProduceRequest(long now, int destination, short acks, int timeout, List<ProducerBatch> batches) {
        if (batches.isEmpty()) {
            return;
        }
        HashMap<TopicPartition, MemoryRecords> produceRecordsByPartition = new HashMap<TopicPartition, MemoryRecords>(batches.size());
        final HashMap<TopicPartition, ProducerBatch> recordsByPartition = new HashMap<TopicPartition, ProducerBatch>(batches.size());
        byte minUsedMagic = this.apiVersions.maxUsableProduceMagic();
        for (ProducerBatch batch : batches) {
            if (batch.magic() >= minUsedMagic) continue;
            minUsedMagic = batch.magic();
        }
        for (ProducerBatch batch : batches) {
            TopicPartition tp = batch.topicPartition;
            MemoryRecords records = batch.records();
            if (!records.hasMatchingMagic(minUsedMagic)) {
                records = batch.records().downConvert(minUsedMagic, 0L, this.time).records();
            }
            produceRecordsByPartition.put(tp, records);
            recordsByPartition.put(tp, batch);
        }
        String transactionalId = null;
        if (this.transactionManager != null && this.transactionManager.isTransactional()) {
            transactionalId = this.transactionManager.transactionalId();
        }
        ProduceRequest.Builder requestBuilder = ProduceRequest.Builder.forMagic(minUsedMagic, acks, timeout, produceRecordsByPartition, transactionalId);
        RequestCompletionHandler callback = new RequestCompletionHandler(){

            @Override
            public void onComplete(ClientResponse response) {
                Sender.this.handleProduceResponse(response, recordsByPartition, Sender.this.time.milliseconds());
            }
        };
        String nodeId = Integer.toString(destination);
        ClientRequest clientRequest = this.client.newClientRequest(nodeId, requestBuilder, now, acks != 0, callback);
        this.client.send(clientRequest, now);
        this.log.trace("Sent produce request to {}: {}", (Object)nodeId, (Object)requestBuilder);
    }

    public void wakeup() {
        this.client.wakeup();
    }

    public static Sensor throttleTimeSensor(SenderMetricsRegistry metrics) {
        Sensor produceThrottleTimeSensor = metrics.sensor("produce-throttle-time");
        produceThrottleTimeSensor.add(metrics.produceThrottleTimeAvg, new Avg());
        produceThrottleTimeSensor.add(metrics.produceThrottleTimeMax, new Max());
        return produceThrottleTimeSensor;
    }

    private class SenderMetrics {
        public final Sensor retrySensor;
        public final Sensor errorSensor;
        public final Sensor queueTimeSensor;
        public final Sensor requestTimeSensor;
        public final Sensor recordsPerRequestSensor;
        public final Sensor batchSizeSensor;
        public final Sensor compressionRateSensor;
        public final Sensor maxRecordSizeSensor;
        public final Sensor batchSplitSensor;
        private final SenderMetricsRegistry metrics;

        public SenderMetrics(SenderMetricsRegistry metrics) {
            this.metrics = metrics;
            this.batchSizeSensor = metrics.sensor("batch-size");
            this.batchSizeSensor.add(metrics.batchSizeAvg, new Avg());
            this.batchSizeSensor.add(metrics.batchSizeMax, new Max());
            this.compressionRateSensor = metrics.sensor("compression-rate");
            this.compressionRateSensor.add(metrics.compressionRateAvg, new Avg());
            this.queueTimeSensor = metrics.sensor("queue-time");
            this.queueTimeSensor.add(metrics.recordQueueTimeAvg, new Avg());
            this.queueTimeSensor.add(metrics.recordQueueTimeMax, new Max());
            this.requestTimeSensor = metrics.sensor("request-time");
            this.requestTimeSensor.add(metrics.requestLatencyAvg, new Avg());
            this.requestTimeSensor.add(metrics.requestLatencyMax, new Max());
            this.recordsPerRequestSensor = metrics.sensor("records-per-request");
            this.recordsPerRequestSensor.add(new Meter(metrics.recordSendRate, metrics.recordSendTotal));
            this.recordsPerRequestSensor.add(metrics.recordsPerRequestAvg, new Avg());
            this.retrySensor = metrics.sensor("record-retries");
            this.retrySensor.add(new Meter(metrics.recordRetryRate, metrics.recordRetryTotal));
            this.errorSensor = metrics.sensor("errors");
            this.errorSensor.add(new Meter(metrics.recordErrorRate, metrics.recordErrorTotal));
            this.maxRecordSizeSensor = metrics.sensor("record-size");
            this.maxRecordSizeSensor.add(metrics.recordSizeMax, new Max());
            this.maxRecordSizeSensor.add(metrics.recordSizeAvg, new Avg());
            this.metrics.addMetric(metrics.requestsInFlight, new Measurable(){

                @Override
                public double measure(MetricConfig config, long now) {
                    return Sender.this.client.inFlightRequestCount();
                }
            });
            metrics.addMetric(metrics.metadataAge, new Measurable(){

                @Override
                public double measure(MetricConfig config, long now) {
                    return (double)(now - Sender.this.metadata.lastSuccessfulUpdate()) / 1000.0;
                }
            });
            this.batchSplitSensor = metrics.sensor("batch-split-rate");
            this.batchSplitSensor.add(new Meter(metrics.batchSplitRate, metrics.batchSplitTotal));
        }

        private void maybeRegisterTopicMetrics(String topic) {
            String topicRecordsCountName = "topic." + topic + ".records-per-batch";
            Sensor topicRecordCount = this.metrics.getSensor(topicRecordsCountName);
            if (topicRecordCount == null) {
                Map<String, String> metricTags = Collections.singletonMap("topic", topic);
                topicRecordCount = this.metrics.sensor(topicRecordsCountName);
                MetricName rateMetricName = this.metrics.topicRecordSendRate(metricTags);
                MetricName totalMetricName = this.metrics.topicRecordSendTotal(metricTags);
                topicRecordCount.add(new Meter(rateMetricName, totalMetricName));
                String topicByteRateName = "topic." + topic + ".bytes";
                Sensor topicByteRate = this.metrics.sensor(topicByteRateName);
                rateMetricName = this.metrics.topicByteRate(metricTags);
                totalMetricName = this.metrics.topicByteTotal(metricTags);
                topicByteRate.add(new Meter(rateMetricName, totalMetricName));
                String topicCompressionRateName = "topic." + topic + ".compression-rate";
                Sensor topicCompressionRate = this.metrics.sensor(topicCompressionRateName);
                MetricName m2 = this.metrics.topicCompressionRate(metricTags);
                topicCompressionRate.add(m2, new Avg());
                String topicRetryName = "topic." + topic + ".record-retries";
                Sensor topicRetrySensor = this.metrics.sensor(topicRetryName);
                rateMetricName = this.metrics.topicRecordRetryRate(metricTags);
                totalMetricName = this.metrics.topicRecordRetryTotal(metricTags);
                topicRetrySensor.add(new Meter(rateMetricName, totalMetricName));
                String topicErrorName = "topic." + topic + ".record-errors";
                Sensor topicErrorSensor = this.metrics.sensor(topicErrorName);
                rateMetricName = this.metrics.topicRecordErrorRate(metricTags);
                totalMetricName = this.metrics.topicRecordErrorTotal(metricTags);
                topicErrorSensor.add(new Meter(rateMetricName, totalMetricName));
            }
        }

        public void updateProduceRequestMetrics(Map<Integer, List<ProducerBatch>> batches) {
            long now = Sender.this.time.milliseconds();
            for (List<ProducerBatch> nodeBatch : batches.values()) {
                int records = 0;
                for (ProducerBatch batch : nodeBatch) {
                    String topic = batch.topicPartition.topic();
                    this.maybeRegisterTopicMetrics(topic);
                    String topicRecordsCountName = "topic." + topic + ".records-per-batch";
                    Sensor topicRecordCount = Utils.notNull(this.metrics.getSensor(topicRecordsCountName));
                    topicRecordCount.record(batch.recordCount);
                    String topicByteRateName = "topic." + topic + ".bytes";
                    Sensor topicByteRate = Utils.notNull(this.metrics.getSensor(topicByteRateName));
                    topicByteRate.record(batch.estimatedSizeInBytes());
                    String topicCompressionRateName = "topic." + topic + ".compression-rate";
                    Sensor topicCompressionRate = Utils.notNull(this.metrics.getSensor(topicCompressionRateName));
                    topicCompressionRate.record(batch.compressionRatio());
                    this.batchSizeSensor.record(batch.estimatedSizeInBytes(), now);
                    this.queueTimeSensor.record(batch.queueTimeMs(), now);
                    this.compressionRateSensor.record(batch.compressionRatio());
                    this.maxRecordSizeSensor.record(batch.maxRecordSize, now);
                    records += batch.recordCount;
                }
                this.recordsPerRequestSensor.record(records, now);
            }
        }

        public void recordRetries(String topic, int count) {
            long now = Sender.this.time.milliseconds();
            this.retrySensor.record(count, now);
            String topicRetryName = "topic." + topic + ".record-retries";
            Sensor topicRetrySensor = this.metrics.getSensor(topicRetryName);
            if (topicRetrySensor != null) {
                topicRetrySensor.record(count, now);
            }
        }

        public void recordErrors(String topic, int count) {
            long now = Sender.this.time.milliseconds();
            this.errorSensor.record(count, now);
            String topicErrorName = "topic." + topic + ".record-errors";
            Sensor topicErrorSensor = this.metrics.getSensor(topicErrorName);
            if (topicErrorSensor != null) {
                topicErrorSensor.record(count, now);
            }
        }

        public void recordLatency(String node, long latency) {
            String nodeTimeName;
            Sensor nodeRequestTime;
            long now = Sender.this.time.milliseconds();
            this.requestTimeSensor.record(latency, now);
            if (!node.isEmpty() && (nodeRequestTime = this.metrics.getSensor(nodeTimeName = "node-" + node + ".latency")) != null) {
                nodeRequestTime.record(latency, now);
            }
        }

        void recordBatchSplit() {
            this.batchSplitSensor.record();
        }
    }
}

