/*
 * Decompiled with CFR 0.152.
 */
package com.networknt.consul;

import com.networknt.consul.ConsulConstants;
import com.networknt.consul.ConsulHeartbeatManager;
import com.networknt.consul.ConsulResponse;
import com.networknt.consul.ConsulService;
import com.networknt.consul.ConsulUtils;
import com.networknt.consul.client.ConsulClient;
import com.networknt.registry.URL;
import com.networknt.registry.URLParamType;
import com.networknt.registry.support.command.CommandFailbackRegistry;
import com.networknt.registry.support.command.CommandListener;
import com.networknt.registry.support.command.ServiceListener;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ConsulRegistry
extends CommandFailbackRegistry {
    private static final Logger logger = LoggerFactory.getLogger(ConsulRegistry.class);
    private ConsulClient client;
    private ConsulHeartbeatManager heartbeatManager;
    private int lookupInterval;
    private ConcurrentHashMap<String, ConcurrentHashMap<String, List<URL>>> serviceCache = new ConcurrentHashMap();
    private ConcurrentHashMap<String, String> commandCache = new ConcurrentHashMap();
    private ConcurrentHashMap<String, Long> lookupServices = new ConcurrentHashMap();
    private ConcurrentHashMap<String, String> lookupCommands = new ConcurrentHashMap();
    private ConcurrentHashMap<String, ConcurrentHashMap<URL, ServiceListener>> serviceListeners = new ConcurrentHashMap();
    private ConcurrentHashMap<String, ConcurrentHashMap<URL, CommandListener>> commandListeners = new ConcurrentHashMap();
    private ThreadPoolExecutor notifyExecutor;

    public ConsulRegistry(URL url, ConsulClient client) {
        super(url);
        this.client = client;
        this.heartbeatManager = new ConsulHeartbeatManager(client);
        this.heartbeatManager.start();
        this.lookupInterval = this.getUrl().getIntParameter(URLParamType.registrySessionTimeout.getName(), ConsulConstants.DEFAULT_LOOKUP_INTERVAL);
        ArrayBlockingQueue<Runnable> workQueue = new ArrayBlockingQueue<Runnable>(20000);
        this.notifyExecutor = new ThreadPoolExecutor(10, 30, 30000L, TimeUnit.MILLISECONDS, workQueue);
        logger.info("ConsulRegistry init finish.");
    }

    public ConcurrentHashMap<String, ConcurrentHashMap<URL, ServiceListener>> getServiceListeners() {
        return this.serviceListeners;
    }

    public ConcurrentHashMap<String, ConcurrentHashMap<URL, CommandListener>> getCommandListeners() {
        return this.commandListeners;
    }

    @Override
    protected void doRegister(URL url) {
        ConsulService service = ConsulUtils.buildService(url);
        this.client.registerService(service);
        this.heartbeatManager.addHeartbeatServcieId(service.getId());
    }

    @Override
    protected void doUnregister(URL url) {
        ConsulService service = ConsulUtils.buildService(url);
        this.client.unregisterService(service.getId());
        this.heartbeatManager.removeHeartbeatServiceId(service.getId());
    }

    @Override
    protected void doAvailable(URL url) {
        if (url != null) {
            throw new UnsupportedOperationException("Command consul registry not support available by urls yet");
        }
        this.heartbeatManager.setHeartbeatOpen(true);
    }

    @Override
    protected void doUnavailable(URL url) {
        if (url != null) {
            throw new UnsupportedOperationException("Command consul registry not support unavailable by urls yet");
        }
        this.heartbeatManager.setHeartbeatOpen(false);
    }

    @Override
    protected void subscribeService(URL url, ServiceListener serviceListener) {
        this.addServiceListener(url, serviceListener);
        this.startListenerThreadIfNewService(url);
    }

    private void startListenerThreadIfNewService(URL url) {
        Long value;
        String serviceName = url.getPath();
        if (!this.lookupServices.containsKey(serviceName) && (value = this.lookupServices.putIfAbsent(serviceName, 0L)) == null) {
            ServiceLookupThread lookupThread = new ServiceLookupThread(serviceName);
            lookupThread.setDaemon(true);
            lookupThread.start();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addServiceListener(URL url, ServiceListener serviceListener) {
        String service = ConsulUtils.getUrlClusterInfo(url);
        ConcurrentHashMap<URL, ServiceListener> map = this.serviceListeners.get(service);
        if (map == null) {
            this.serviceListeners.putIfAbsent(service, new ConcurrentHashMap());
            map = this.serviceListeners.get(service);
        }
        ConcurrentHashMap<URL, ServiceListener> concurrentHashMap = map;
        synchronized (concurrentHashMap) {
            map.put(url, serviceListener);
        }
    }

    @Override
    protected void subscribeCommand(URL url, CommandListener commandListener) {
        this.addCommandListener(url, commandListener);
        this.startListenerThreadIfNewCommand(url);
    }

    private void startListenerThreadIfNewCommand(URL url) {
        String command;
        String serviceName = url.getPath();
        if (!this.lookupCommands.containsKey(serviceName) && (command = this.lookupCommands.putIfAbsent(serviceName, "")) == null) {
            CommandLookupThread lookupThread = new CommandLookupThread(serviceName);
            lookupThread.setDaemon(true);
            lookupThread.start();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void addCommandListener(URL url, CommandListener commandListener) {
        String serviceName = url.getPath();
        ConcurrentHashMap<URL, CommandListener> map = this.commandListeners.get(serviceName);
        if (map == null) {
            this.commandListeners.putIfAbsent(serviceName, new ConcurrentHashMap());
            map = this.commandListeners.get(serviceName);
        }
        ConcurrentHashMap<URL, CommandListener> concurrentHashMap = map;
        synchronized (concurrentHashMap) {
            map.put(url, commandListener);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void unsubscribeService(URL url, ServiceListener listener) {
        ConcurrentHashMap<URL, ServiceListener> listeners = this.serviceListeners.get(ConsulUtils.getUrlClusterInfo(url));
        if (listeners != null) {
            ConcurrentHashMap<URL, ServiceListener> concurrentHashMap = listeners;
            synchronized (concurrentHashMap) {
                listeners.remove(url);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected void unsubscribeCommand(URL url, CommandListener listener) {
        ConcurrentHashMap<URL, CommandListener> listeners = this.commandListeners.get(url.getPath());
        if (listeners != null) {
            ConcurrentHashMap<URL, CommandListener> concurrentHashMap = listeners;
            synchronized (concurrentHashMap) {
                listeners.remove(url);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    protected List<URL> discoverService(URL url) {
        String serviceName = url.getPath();
        List<URL> serviceUrls = new ArrayList<URL>();
        ConcurrentHashMap<String, List<URL>> serviceMap = this.serviceCache.get(serviceName);
        if (serviceMap == null) {
            String string = serviceName.intern();
            synchronized (string) {
                serviceMap = this.serviceCache.get(serviceName);
                if (serviceMap == null) {
                    ConcurrentHashMap<String, List<URL>> urls = this.lookupServiceUpdate(serviceName);
                    this.updateServiceCache(serviceName, urls, false);
                    serviceMap = this.serviceCache.get(serviceName);
                }
            }
        }
        if (serviceMap != null) {
            serviceUrls = serviceMap.get(serviceName);
        }
        return serviceUrls;
    }

    @Override
    protected String discoverCommand(URL url) {
        String serviceName = url.getPath();
        String command = this.lookupCommandUpdate(serviceName);
        this.updateCommandCache(serviceName, command, false);
        return command;
    }

    private ConcurrentHashMap<String, List<URL>> lookupServiceUpdate(String serviceName) {
        Long lastConsulIndexId = this.lookupServices.get(serviceName) == null ? 0L : this.lookupServices.get(serviceName);
        ConsulResponse<List<ConsulService>> response = this.lookupConsulService(serviceName, lastConsulIndexId);
        if (response != null) {
            List<ConsulService> services = response.getValue();
            if (services != null && !services.isEmpty() && response.getConsulIndex() > lastConsulIndexId) {
                ConcurrentHashMap<String, List<URL>> serviceUrls = new ConcurrentHashMap<String, List<URL>>();
                for (ConsulService service : services) {
                    try {
                        URL url = ConsulUtils.buildUrl(service);
                        String cluster = ConsulUtils.getUrlClusterInfo(url);
                        List<URL> urlList = serviceUrls.get(cluster);
                        if (urlList == null) {
                            urlList = new ArrayList<URL>();
                            serviceUrls.put(cluster, urlList);
                        }
                        urlList.add(url);
                    }
                    catch (Exception e) {
                        logger.error("convert consul service to url fail! service:" + service, e);
                    }
                }
                this.lookupServices.put(serviceName, response.getConsulIndex());
                return serviceUrls;
            }
            logger.info(serviceName + " no need update, lastIndex:" + lastConsulIndexId);
        }
        return null;
    }

    private String lookupCommandUpdate(String serviceName) {
        String command = this.client.lookupCommand(serviceName);
        this.lookupCommands.put(serviceName, command);
        return command;
    }

    private ConsulResponse<List<ConsulService>> lookupConsulService(String serviceName, Long lastConsulIndexId) {
        ConsulResponse<List<ConsulService>> response = this.client.lookupHealthService(serviceName, lastConsulIndexId);
        return response;
    }

    private void updateServiceCache(String serviceName, ConcurrentHashMap<String, List<URL>> serviceUrls, boolean needNotify) {
        if (serviceUrls != null && !serviceUrls.isEmpty()) {
            ConcurrentHashMap<String, List<URL>> serviceMap = this.serviceCache.get(serviceName);
            if (serviceMap == null) {
                this.serviceCache.put(serviceName, serviceUrls);
            }
            for (Map.Entry<String, List<URL>> entry : serviceUrls.entrySet()) {
                boolean change = true;
                if (serviceMap != null) {
                    List<URL> oldUrls = serviceMap.get(entry.getKey());
                    List<URL> newUrls = entry.getValue();
                    if (newUrls == null || newUrls.isEmpty() || ConsulUtils.isSame(entry.getValue(), oldUrls)) {
                        change = false;
                    } else {
                        serviceMap.put(entry.getKey(), newUrls);
                    }
                }
                if (!change || !needNotify) continue;
                this.notifyExecutor.execute(new NotifyService(entry.getKey(), entry.getValue()));
                logger.info("light service notify-service: " + entry.getKey());
                StringBuilder sb = new StringBuilder();
                for (URL url : entry.getValue()) {
                    sb.append(url.getUri()).append(";");
                }
                logger.info("consul notify urls:" + sb.toString());
            }
        }
    }

    private void updateCommandCache(String serviceName, String command, boolean needNotify) {
        String oldCommand = this.commandCache.get(serviceName);
        if (!command.equals(oldCommand)) {
            this.commandCache.put(serviceName, command);
            if (needNotify) {
                this.notifyExecutor.execute(new NotifyCommand(serviceName, command));
                logger.info(String.format("command data change: serviceName=%s, command=%s: ", serviceName, command));
            }
        } else {
            logger.info(String.format("command data not change: serviceName=%s, command=%s: ", serviceName, command));
        }
    }

    private class NotifyCommand
    implements Runnable {
        private String serviceName;
        private String command;

        public NotifyCommand(String serviceName, String command) {
            this.serviceName = serviceName;
            this.command = command;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            ConcurrentHashMap listeners;
            ConcurrentHashMap concurrentHashMap = listeners = (ConcurrentHashMap)ConsulRegistry.this.commandListeners.get(this.serviceName);
            synchronized (concurrentHashMap) {
                for (Map.Entry entry : listeners.entrySet()) {
                    CommandListener commandListener = (CommandListener)entry.getValue();
                    commandListener.notifyCommand((URL)entry.getKey(), this.command);
                }
            }
        }
    }

    private class NotifyService
    implements Runnable {
        private String service;
        private List<URL> urls;

        public NotifyService(String service, List<URL> urls) {
            this.service = service;
            this.urls = urls;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            ConcurrentHashMap listeners = (ConcurrentHashMap)ConsulRegistry.this.serviceListeners.get(this.service);
            if (listeners != null) {
                ConcurrentHashMap concurrentHashMap = listeners;
                synchronized (concurrentHashMap) {
                    for (Map.Entry entry : listeners.entrySet()) {
                        ServiceListener serviceListener = (ServiceListener)entry.getValue();
                        serviceListener.notifyService((URL)entry.getKey(), ConsulRegistry.this.getUrl(), this.urls);
                    }
                }
            } else {
                logger.debug("need not notify service:" + this.service);
            }
        }
    }

    private class CommandLookupThread
    extends Thread {
        private String serviceName;

        public CommandLookupThread(String serviceName) {
            this.serviceName = serviceName;
        }

        @Override
        public void run() {
            logger.info("start command lookup thread. lookup interval: " + ConsulRegistry.this.lookupInterval + "ms, serviceName: " + this.serviceName);
            while (true) {
                try {
                    while (true) {
                        CommandLookupThread.sleep(ConsulRegistry.this.lookupInterval);
                        String command = ConsulRegistry.this.lookupCommandUpdate(this.serviceName);
                        ConsulRegistry.this.updateCommandCache(this.serviceName, command, true);
                    }
                }
                catch (Throwable e) {
                    logger.error("serviceName lookup thread fail!", e);
                    try {
                        Thread.sleep(2000L);
                    }
                    catch (InterruptedException interruptedException) {
                    }
                    continue;
                }
                break;
            }
        }
    }

    private class ServiceLookupThread
    extends Thread {
        private String serviceName;

        public ServiceLookupThread(String serviceName) {
            this.serviceName = serviceName;
        }

        @Override
        public void run() {
            logger.info("start service lookup thread. lookup interval: " + ConsulRegistry.this.lookupInterval + "ms, service: " + this.serviceName);
            while (true) {
                try {
                    while (true) {
                        ServiceLookupThread.sleep(ConsulRegistry.this.lookupInterval);
                        ConcurrentHashMap serviceUrls = ConsulRegistry.this.lookupServiceUpdate(this.serviceName);
                        ConsulRegistry.this.updateServiceCache(this.serviceName, serviceUrls, true);
                    }
                }
                catch (Throwable e) {
                    logger.error("service lookup thread fail!", e);
                    try {
                        Thread.sleep(2000L);
                    }
                    catch (InterruptedException interruptedException) {
                    }
                    continue;
                }
                break;
            }
        }
    }
}

