/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.clients.consumer.internals;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.kafka.clients.ClientResponse;
import org.apache.kafka.clients.Metadata;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.NoOffsetForPartitionException;
import org.apache.kafka.clients.consumer.OffsetAndTimestamp;
import org.apache.kafka.clients.consumer.OffsetOutOfRangeException;
import org.apache.kafka.clients.consumer.OffsetResetStrategy;
import org.apache.kafka.clients.consumer.internals.ConsumerNetworkClient;
import org.apache.kafka.clients.consumer.internals.RequestFuture;
import org.apache.kafka.clients.consumer.internals.RequestFutureAdapter;
import org.apache.kafka.clients.consumer.internals.RequestFutureListener;
import org.apache.kafka.clients.consumer.internals.StaleMetadataException;
import org.apache.kafka.clients.consumer.internals.SubscriptionState;
import org.apache.kafka.common.Cluster;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Node;
import org.apache.kafka.common.PartitionInfo;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.InvalidMetadataException;
import org.apache.kafka.common.errors.InvalidTopicException;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.SerializationException;
import org.apache.kafka.common.errors.TimeoutException;
import org.apache.kafka.common.errors.TopicAuthorizationException;
import org.apache.kafka.common.metrics.Metrics;
import org.apache.kafka.common.metrics.Sensor;
import org.apache.kafka.common.metrics.stats.Avg;
import org.apache.kafka.common.metrics.stats.Count;
import org.apache.kafka.common.metrics.stats.Max;
import org.apache.kafka.common.metrics.stats.Rate;
import org.apache.kafka.common.protocol.ApiKeys;
import org.apache.kafka.common.protocol.Errors;
import org.apache.kafka.common.record.InvalidRecordException;
import org.apache.kafka.common.record.LogEntry;
import org.apache.kafka.common.record.MemoryRecords;
import org.apache.kafka.common.record.Record;
import org.apache.kafka.common.record.TimestampType;
import org.apache.kafka.common.requests.FetchRequest;
import org.apache.kafka.common.requests.FetchResponse;
import org.apache.kafka.common.requests.ListOffsetRequest;
import org.apache.kafka.common.requests.ListOffsetResponse;
import org.apache.kafka.common.requests.MetadataRequest;
import org.apache.kafka.common.requests.MetadataResponse;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Fetcher<K, V> {
    private static final Logger log = LoggerFactory.getLogger(Fetcher.class);
    private final ConsumerNetworkClient client;
    private final Time time;
    private final int minBytes;
    private final int maxBytes;
    private final int maxWaitMs;
    private final int fetchSize;
    private final long retryBackoffMs;
    private final int maxPollRecords;
    private final boolean checkCrcs;
    private final Metadata metadata;
    private final FetchManagerMetrics sensors;
    private final SubscriptionState subscriptions;
    private final ConcurrentLinkedQueue<CompletedFetch> completedFetches;
    private final Deserializer<K> keyDeserializer;
    private final Deserializer<V> valueDeserializer;
    private PartitionRecords<K, V> nextInLineRecords = null;

    public Fetcher(ConsumerNetworkClient client, int minBytes, int maxBytes, int maxWaitMs, int fetchSize, int maxPollRecords, boolean checkCrcs, Deserializer<K> keyDeserializer, Deserializer<V> valueDeserializer, Metadata metadata, SubscriptionState subscriptions, Metrics metrics, String metricGrpPrefix, Time time, long retryBackoffMs) {
        this.time = time;
        this.client = client;
        this.metadata = metadata;
        this.subscriptions = subscriptions;
        this.minBytes = minBytes;
        this.maxBytes = maxBytes;
        this.maxWaitMs = maxWaitMs;
        this.fetchSize = fetchSize;
        this.maxPollRecords = maxPollRecords;
        this.checkCrcs = checkCrcs;
        this.keyDeserializer = keyDeserializer;
        this.valueDeserializer = valueDeserializer;
        this.completedFetches = new ConcurrentLinkedQueue();
        this.sensors = new FetchManagerMetrics(metrics, metricGrpPrefix);
        this.retryBackoffMs = retryBackoffMs;
    }

    public boolean hasCompletedFetches() {
        return !this.completedFetches.isEmpty();
    }

    private boolean matchesRequestedPartitions(FetchRequest request, FetchResponse response) {
        Set<TopicPartition> requestedPartitions = request.fetchData().keySet();
        Set<TopicPartition> fetchedPartitions = response.responseData().keySet();
        return fetchedPartitions.equals(requestedPartitions);
    }

    public void sendFetches() {
        for (Map.Entry<Node, FetchRequest> fetchEntry : this.createFetchRequests().entrySet()) {
            final FetchRequest request = fetchEntry.getValue();
            final Node fetchTarget = fetchEntry.getKey();
            this.client.send(fetchTarget, ApiKeys.FETCH, request).addListener(new RequestFutureListener<ClientResponse>(){

                @Override
                public void onSuccess(ClientResponse resp) {
                    FetchResponse response = new FetchResponse(resp.responseBody());
                    if (!Fetcher.this.matchesRequestedPartitions(request, response)) {
                        log.warn("Ignoring fetch response containing partitions {} since it does not match the requested partitions {}", (Object)response.responseData().keySet(), (Object)request.fetchData().keySet());
                        return;
                    }
                    HashSet<TopicPartition> partitions = new HashSet<TopicPartition>(response.responseData().keySet());
                    FetchResponseMetricAggregator metricAggregator = new FetchResponseMetricAggregator(Fetcher.this.sensors, partitions);
                    for (Map.Entry<TopicPartition, FetchResponse.PartitionData> entry : response.responseData().entrySet()) {
                        TopicPartition partition = entry.getKey();
                        long fetchOffset = request.fetchData().get((Object)partition).offset;
                        FetchResponse.PartitionData fetchData = entry.getValue();
                        Fetcher.this.completedFetches.add(new CompletedFetch(partition, fetchOffset, fetchData, metricAggregator));
                    }
                    ((Fetcher)Fetcher.this).sensors.fetchLatency.record(resp.requestLatencyMs());
                    ((Fetcher)Fetcher.this).sensors.fetchThrottleTimeSensor.record(response.getThrottleTime());
                }

                @Override
                public void onFailure(RuntimeException e) {
                    log.debug("Fetch request to {} failed", (Object)fetchTarget, (Object)e);
                }
            });
        }
    }

    public void resetOffsetsIfNeeded(Set<TopicPartition> partitions) {
        for (TopicPartition tp : partitions) {
            if (!this.subscriptions.isAssigned(tp) || !this.subscriptions.isOffsetResetNeeded(tp)) continue;
            this.resetOffset(tp);
        }
    }

    public void updateFetchPositions(Set<TopicPartition> partitions) {
        for (TopicPartition tp : partitions) {
            if (!this.subscriptions.isAssigned(tp) || this.subscriptions.isFetchable(tp)) continue;
            if (this.subscriptions.isOffsetResetNeeded(tp)) {
                this.resetOffset(tp);
                continue;
            }
            if (this.subscriptions.committed(tp) == null) {
                this.subscriptions.needOffsetReset(tp);
                this.resetOffset(tp);
                continue;
            }
            long committed = this.subscriptions.committed(tp).offset();
            log.debug("Resetting offset for partition {} to the committed offset {}", (Object)tp, (Object)committed);
            this.subscriptions.seek(tp, committed);
        }
    }

    public Map<String, List<PartitionInfo>> getAllTopicMetadata(long timeout) {
        return this.getTopicMetadata(MetadataRequest.allTopics(), timeout);
    }

    public Map<String, List<PartitionInfo>> getTopicMetadata(MetadataRequest request, long timeout) {
        if (!request.isAllTopics() && request.topics().isEmpty()) {
            return Collections.emptyMap();
        }
        long start = this.time.milliseconds();
        long remaining = timeout;
        do {
            long elapsed;
            RequestFuture<ClientResponse> future = this.sendMetadataRequest(request);
            this.client.poll(future, remaining);
            if (future.failed() && !future.isRetriable()) {
                throw future.exception();
            }
            if (future.succeeded()) {
                MetadataResponse response = new MetadataResponse(future.value().responseBody());
                Cluster cluster = response.cluster();
                Set<String> unauthorizedTopics = cluster.unauthorizedTopics();
                if (!unauthorizedTopics.isEmpty()) {
                    throw new TopicAuthorizationException(unauthorizedTopics);
                }
                boolean shouldRetry = false;
                Map<String, Errors> errors = response.errors();
                if (!errors.isEmpty()) {
                    log.debug("Topic metadata fetch included errors: {}", (Object)errors);
                    for (Map.Entry<String, Errors> errorEntry : errors.entrySet()) {
                        String topic = errorEntry.getKey();
                        Errors error = errorEntry.getValue();
                        if (error == Errors.INVALID_TOPIC_EXCEPTION) {
                            throw new InvalidTopicException("Topic '" + topic + "' is invalid");
                        }
                        if (error == Errors.UNKNOWN_TOPIC_OR_PARTITION) continue;
                        if (error.exception() instanceof RetriableException) {
                            shouldRetry = true;
                            continue;
                        }
                        throw new KafkaException("Unexpected error fetching metadata for topic " + topic, error.exception());
                    }
                }
                if (!shouldRetry) {
                    HashMap<String, List<PartitionInfo>> topicsPartitionInfos = new HashMap<String, List<PartitionInfo>>();
                    for (String topic : cluster.topics()) {
                        topicsPartitionInfos.put(topic, cluster.availablePartitionsForTopic(topic));
                    }
                    return topicsPartitionInfos;
                }
            }
            if ((remaining = timeout - (elapsed = this.time.milliseconds() - start)) <= 0L) continue;
            long backoff = Math.min(remaining, this.retryBackoffMs);
            this.time.sleep(backoff);
            remaining -= backoff;
        } while (remaining > 0L);
        throw new TimeoutException("Timeout expired while fetching topic metadata");
    }

    private RequestFuture<ClientResponse> sendMetadataRequest(MetadataRequest request) {
        Node node = this.client.leastLoadedNode();
        if (node == null) {
            return RequestFuture.noBrokersAvailable();
        }
        return this.client.send(node, ApiKeys.METADATA, request);
    }

    private void resetOffset(TopicPartition partition) {
        long timestamp;
        OffsetResetStrategy strategy = this.subscriptions.resetStrategy(partition);
        if (strategy == OffsetResetStrategy.EARLIEST) {
            timestamp = -2L;
        } else if (strategy == OffsetResetStrategy.LATEST) {
            timestamp = -1L;
        } else {
            throw new NoOffsetForPartitionException(partition);
        }
        log.debug("Resetting offset for partition {} to {} offset.", (Object)partition, (Object)strategy.name().toLowerCase(Locale.ROOT));
        long offset = this.getOffsetsByTimes(Collections.singletonMap(partition, timestamp), Long.MAX_VALUE).get(partition).offset();
        if (this.subscriptions.isAssigned(partition)) {
            this.subscriptions.seek(partition, offset);
        }
    }

    public Map<TopicPartition, OffsetAndTimestamp> getOffsetsByTimes(Map<TopicPartition, Long> timestampsToSearch, long timeout) {
        long elapsed;
        if (timestampsToSearch.isEmpty()) {
            return Collections.emptyMap();
        }
        long startMs = this.time.milliseconds();
        long remaining = timeout;
        do {
            RequestFuture<Map<TopicPartition, OffsetAndTimestamp>> future = this.sendListOffsetRequests(timestampsToSearch);
            this.client.poll(future, remaining);
            if (!future.isDone()) break;
            if (future.succeeded()) {
                return future.value();
            }
            if (!future.isRetriable()) {
                throw future.exception();
            }
            elapsed = this.time.milliseconds() - startMs;
            remaining = timeout - elapsed;
            if (remaining <= 0L) break;
            if (future.exception() instanceof InvalidMetadataException) {
                this.client.awaitMetadataUpdate(remaining);
                continue;
            }
            this.time.sleep(Math.min(remaining, this.retryBackoffMs));
        } while ((remaining = timeout - (elapsed = this.time.milliseconds() - startMs)) > 0L);
        throw new TimeoutException("Failed to get offsets by times in " + timeout + " ms");
    }

    public Map<TopicPartition, Long> beginningOffsets(Collection<TopicPartition> partitions, long timeout) {
        return this.beginningOrEndOffset(partitions, -2L, timeout);
    }

    public Map<TopicPartition, Long> endOffsets(Collection<TopicPartition> partitions, long timeout) {
        return this.beginningOrEndOffset(partitions, -1L, timeout);
    }

    private Map<TopicPartition, Long> beginningOrEndOffset(Collection<TopicPartition> partitions, long timestamp, long timeout) {
        HashMap<TopicPartition, Long> timestampsToSearch = new HashMap<TopicPartition, Long>();
        for (TopicPartition tp : partitions) {
            timestampsToSearch.put(tp, timestamp);
        }
        HashMap<TopicPartition, Long> result = new HashMap<TopicPartition, Long>();
        for (Map.Entry<TopicPartition, OffsetAndTimestamp> entry : this.getOffsetsByTimes(timestampsToSearch, timeout).entrySet()) {
            result.put(entry.getKey(), entry.getValue().offset());
        }
        return result;
    }

    public Map<TopicPartition, List<ConsumerRecord<K, V>>> fetchedRecords() {
        HashMap<TopicPartition, List<ConsumerRecord<TopicPartition, List<ConsumerRecord<K, V>>>>> drained = new HashMap<TopicPartition, List<ConsumerRecord<TopicPartition, List<ConsumerRecord<K, V>>>>>();
        int recordsRemaining = this.maxPollRecords;
        while (recordsRemaining > 0) {
            if (this.nextInLineRecords == null || ((PartitionRecords)this.nextInLineRecords).isDrained()) {
                CompletedFetch completedFetch = this.completedFetches.poll();
                if (completedFetch == null) break;
                this.nextInLineRecords = this.parseFetchedData(completedFetch);
                continue;
            }
            TopicPartition partition = ((PartitionRecords)this.nextInLineRecords).partition;
            List<ConsumerRecord<K, V>> records = this.drainRecords(this.nextInLineRecords, recordsRemaining);
            if (records.isEmpty()) continue;
            List currentRecords = (List)drained.get(partition);
            if (currentRecords == null) {
                drained.put(partition, records);
            } else {
                ArrayList<ConsumerRecord<K, V>> newRecords = new ArrayList<ConsumerRecord<K, V>>(records.size() + currentRecords.size());
                newRecords.addAll(currentRecords);
                newRecords.addAll(records);
                drained.put(partition, newRecords);
            }
            recordsRemaining -= records.size();
        }
        return drained;
    }

    private List<ConsumerRecord<K, V>> drainRecords(PartitionRecords<K, V> partitionRecords, int maxRecords) {
        if (((PartitionRecords)partitionRecords).isDrained()) {
            return Collections.emptyList();
        }
        if (!this.subscriptions.isAssigned(((PartitionRecords)partitionRecords).partition)) {
            log.debug("Not returning fetched records for partition {} since it is no longer assigned", (Object)((PartitionRecords)partitionRecords).partition);
        } else {
            long position = this.subscriptions.position(((PartitionRecords)partitionRecords).partition);
            if (!this.subscriptions.isFetchable(((PartitionRecords)partitionRecords).partition)) {
                log.debug("Not returning fetched records for assigned partition {} since it is no longer fetchable", (Object)((PartitionRecords)partitionRecords).partition);
            } else {
                if (((PartitionRecords)partitionRecords).fetchOffset == position) {
                    List partRecords = ((PartitionRecords)partitionRecords).drainRecords(maxRecords);
                    long nextOffset = ((ConsumerRecord)partRecords.get(partRecords.size() - 1)).offset() + 1L;
                    log.trace("Returning fetched records at offset {} for assigned partition {} and update position to {}", position, ((PartitionRecords)partitionRecords).partition, nextOffset);
                    this.subscriptions.position(((PartitionRecords)partitionRecords).partition, nextOffset);
                    return partRecords;
                }
                log.debug("Ignoring fetched records for {} at offset {} since the current position is {}", ((PartitionRecords)partitionRecords).partition, ((PartitionRecords)partitionRecords).fetchOffset, position);
            }
        }
        ((PartitionRecords)partitionRecords).drain();
        return Collections.emptyList();
    }

    private RequestFuture<Map<TopicPartition, OffsetAndTimestamp>> sendListOffsetRequests(Map<TopicPartition, Long> timestampsToSearch) {
        HashMap<Node, HashMap<TopicPartition, Long>> timestampsToSearchByNode = new HashMap<Node, HashMap<TopicPartition, Long>>();
        for (Map.Entry<TopicPartition, Long> entry : timestampsToSearch.entrySet()) {
            TopicPartition tp = entry.getKey();
            PartitionInfo info = this.metadata.fetch().partition(tp);
            if (info == null) {
                this.metadata.add(tp.topic());
                log.debug("Partition {} is unknown for fetching offset, wait for metadata refresh", (Object)tp);
                return RequestFuture.staleMetadata();
            }
            if (info.leader() == null) {
                log.debug("Leader for partition {} unavailable for fetching offset, wait for metadata refresh", (Object)tp);
                return RequestFuture.leaderNotAvailable();
            }
            Node node = info.leader();
            HashMap<TopicPartition, Long> topicData = (HashMap<TopicPartition, Long>)timestampsToSearchByNode.get(node);
            if (topicData == null) {
                topicData = new HashMap<TopicPartition, Long>();
                timestampsToSearchByNode.put(node, topicData);
            }
            topicData.put(entry.getKey(), entry.getValue());
        }
        final RequestFuture<Map<TopicPartition, OffsetAndTimestamp>> listOffsetRequestsFuture = new RequestFuture<Map<TopicPartition, OffsetAndTimestamp>>();
        final HashMap fetchedTimestampOffsets = new HashMap();
        final AtomicInteger remainingResponses = new AtomicInteger(timestampsToSearchByNode.size());
        for (Map.Entry entry : timestampsToSearchByNode.entrySet()) {
            this.sendListOffsetRequest((Node)entry.getKey(), (Map)entry.getValue()).addListener(new RequestFutureListener<Map<TopicPartition, OffsetAndTimestamp>>(){

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void onSuccess(Map<TopicPartition, OffsetAndTimestamp> value) {
                    RequestFuture requestFuture = listOffsetRequestsFuture;
                    synchronized (requestFuture) {
                        fetchedTimestampOffsets.putAll(value);
                        if (remainingResponses.decrementAndGet() == 0 && !listOffsetRequestsFuture.isDone()) {
                            listOffsetRequestsFuture.complete(fetchedTimestampOffsets);
                        }
                    }
                }

                /*
                 * WARNING - Removed try catching itself - possible behaviour change.
                 */
                @Override
                public void onFailure(RuntimeException e) {
                    RequestFuture requestFuture = listOffsetRequestsFuture;
                    synchronized (requestFuture) {
                        if (!listOffsetRequestsFuture.isDone()) {
                            listOffsetRequestsFuture.raise(e);
                        }
                    }
                }
            });
        }
        return listOffsetRequestsFuture;
    }

    private RequestFuture<Map<TopicPartition, OffsetAndTimestamp>> sendListOffsetRequest(final Node node, final Map<TopicPartition, Long> timestampsToSearch) {
        ListOffsetRequest request = new ListOffsetRequest(timestampsToSearch, -1);
        log.trace("Sending ListOffsetRequest {} to broker {}", (Object)request, (Object)node);
        return this.client.send(node, ApiKeys.LIST_OFFSETS, request).compose(new RequestFutureAdapter<ClientResponse, Map<TopicPartition, OffsetAndTimestamp>>(){

            @Override
            public void onSuccess(ClientResponse response, RequestFuture<Map<TopicPartition, OffsetAndTimestamp>> future) {
                ListOffsetResponse lor = new ListOffsetResponse(response.responseBody());
                log.trace("Received ListOffsetResponse {} from broker {}", (Object)lor, (Object)node);
                Fetcher.this.handleListOffsetResponse(timestampsToSearch, lor, future);
            }
        });
    }

    private void handleListOffsetResponse(Map<TopicPartition, Long> timestampsToSearch, ListOffsetResponse listOffsetResponse, RequestFuture<Map<TopicPartition, OffsetAndTimestamp>> future) {
        HashMap<TopicPartition, OffsetAndTimestamp> timestampOffsetMap = new HashMap<TopicPartition, OffsetAndTimestamp>();
        for (Map.Entry<TopicPartition, Long> entry : timestampsToSearch.entrySet()) {
            TopicPartition topicPartition = entry.getKey();
            ListOffsetResponse.PartitionData partitionData = listOffsetResponse.responseData().get(topicPartition);
            Errors error = Errors.forCode(partitionData.errorCode);
            if (error == Errors.NONE) {
                OffsetAndTimestamp offsetAndTimestamp = null;
                if (partitionData.offset != -1L) {
                    offsetAndTimestamp = new OffsetAndTimestamp(partitionData.offset, partitionData.timestamp);
                }
                log.debug("Fetched {} for partition {}", (Object)offsetAndTimestamp, (Object)topicPartition);
                timestampOffsetMap.put(topicPartition, offsetAndTimestamp);
                continue;
            }
            if (error == Errors.UNSUPPORTED_FOR_MESSAGE_FORMAT) {
                log.debug("Cannot search by timestamp for partition {} because the message format version is before 0.10.0", (Object)topicPartition);
                timestampOffsetMap.put(topicPartition, null);
                continue;
            }
            if (error == Errors.NOT_LEADER_FOR_PARTITION) {
                log.debug("Attempt to fetch offsets for partition {} failed due to obsolete leadership information, retrying.", (Object)topicPartition);
                future.raise(error);
                continue;
            }
            if (error == Errors.UNKNOWN_TOPIC_OR_PARTITION) {
                log.warn("Received unknown topic or partition error in ListOffset request for partition {}. The topic/partition may not exist or the user may not have Describe access to it", (Object)topicPartition);
                future.raise(error);
                continue;
            }
            log.warn("Attempt to fetch offsets for partition {} failed due to: {}", (Object)topicPartition, (Object)error.message());
            future.raise(new StaleMetadataException());
        }
        if (!future.isDone()) {
            future.complete(timestampOffsetMap);
        }
    }

    private List<TopicPartition> fetchablePartitions() {
        List<TopicPartition> fetchable = this.subscriptions.fetchablePartitions();
        if (this.nextInLineRecords != null && !((PartitionRecords)this.nextInLineRecords).isDrained()) {
            fetchable.remove(((PartitionRecords)this.nextInLineRecords).partition);
        }
        for (CompletedFetch completedFetch : this.completedFetches) {
            fetchable.remove(completedFetch.partition);
        }
        return fetchable;
    }

    private Map<Node, FetchRequest> createFetchRequests() {
        Cluster cluster = this.metadata.fetch();
        LinkedHashMap fetchable = new LinkedHashMap();
        for (TopicPartition partition : this.fetchablePartitions()) {
            Node node = cluster.leaderFor(partition);
            if (node == null) {
                this.metadata.requestUpdate();
                continue;
            }
            if (this.client.pendingRequestCount(node) == 0) {
                LinkedHashMap<TopicPartition, FetchRequest.PartitionData> fetch = (LinkedHashMap<TopicPartition, FetchRequest.PartitionData>)fetchable.get(node);
                if (fetch == null) {
                    fetch = new LinkedHashMap<TopicPartition, FetchRequest.PartitionData>();
                    fetchable.put(node, fetch);
                }
                long position = this.subscriptions.position(partition);
                fetch.put(partition, new FetchRequest.PartitionData(position, this.fetchSize));
                log.trace("Added fetch request for partition {} at offset {}", (Object)partition, (Object)position);
                continue;
            }
            log.trace("Skipping fetch for partition {} because there is an in-flight request to {}", (Object)partition, (Object)node);
        }
        HashMap<Node, FetchRequest> requests = new HashMap<Node, FetchRequest>();
        for (Map.Entry entry : fetchable.entrySet()) {
            Node node = (Node)entry.getKey();
            FetchRequest fetch = new FetchRequest(this.maxWaitMs, this.minBytes, this.maxBytes, (LinkedHashMap)entry.getValue());
            requests.put(node, fetch);
        }
        return requests;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private PartitionRecords<K, V> parseFetchedData(CompletedFetch completedFetch) {
        Errors error;
        PartitionRecords parsedRecords;
        int bytes;
        TopicPartition tp;
        block17: {
            tp = completedFetch.partition;
            FetchResponse.PartitionData partition = completedFetch.partitionData;
            long fetchOffset = completedFetch.fetchedOffset;
            bytes = 0;
            int recordsCount = 0;
            parsedRecords = null;
            error = Errors.forCode(partition.errorCode);
            try {
                if (!this.subscriptions.isFetchable(tp)) {
                    log.debug("Ignoring fetched records for partition {} since it is no longer fetchable", (Object)tp);
                    break block17;
                }
                if (error == Errors.NONE) {
                    Long position = this.subscriptions.position(tp);
                    if (position == null || position != fetchOffset) {
                        log.debug("Discarding stale fetch response for partition {} since its offset {} does not match the expected offset {}", tp, fetchOffset, position);
                        PartitionRecords<K, V> partitionRecords = null;
                        return partitionRecords;
                    }
                    ByteBuffer buffer = partition.recordSet;
                    MemoryRecords records = MemoryRecords.readableRecords(buffer);
                    ArrayList parsed = new ArrayList();
                    for (LogEntry logEntry : records) {
                        if (logEntry.offset() < position) continue;
                        parsed.add(this.parseRecord(tp, logEntry));
                        bytes += logEntry.size();
                    }
                    recordsCount = parsed.size();
                    this.sensors.recordTopicFetchMetrics(tp.topic(), bytes, recordsCount);
                    if (!parsed.isEmpty()) {
                        log.trace("Adding fetched record for partition {} with offset {} to buffered record list", (Object)tp, (Object)position);
                        parsedRecords = new PartitionRecords(fetchOffset, tp, parsed);
                        ConsumerRecord record = (ConsumerRecord)parsed.get(parsed.size() - 1);
                        this.sensors.recordsFetchLag.record(partition.highWatermark - record.offset());
                    }
                    break block17;
                }
                if (error == Errors.NOT_LEADER_FOR_PARTITION) {
                    log.debug("Error in fetch for partition {}: {}", (Object)tp, (Object)error.exceptionName());
                    this.metadata.requestUpdate();
                    break block17;
                }
                if (error == Errors.UNKNOWN_TOPIC_OR_PARTITION) {
                    log.warn("Received unknown topic or partition error in fetch for partition {}. The topic/partition may not exist or the user may not have Describe access to it", (Object)tp);
                    this.metadata.requestUpdate();
                    break block17;
                }
                if (error == Errors.OFFSET_OUT_OF_RANGE) {
                    if (fetchOffset != this.subscriptions.position(tp)) {
                        log.debug("Discarding stale fetch response for partition {} since the fetched offset {}does not match the current offset {}", tp, fetchOffset, this.subscriptions.position(tp));
                        break block17;
                    }
                    if (this.subscriptions.hasDefaultOffsetResetPolicy()) {
                        log.info("Fetch offset {} is out of range for partition {}, resetting offset", (Object)fetchOffset, (Object)tp);
                        this.subscriptions.needOffsetReset(tp);
                        break block17;
                    }
                    throw new OffsetOutOfRangeException(Collections.singletonMap(tp, fetchOffset));
                }
                if (error == Errors.TOPIC_AUTHORIZATION_FAILED) {
                    log.warn("Not authorized to read from topic {}.", (Object)tp.topic());
                    throw new TopicAuthorizationException(Collections.singleton(tp.topic()));
                }
                if (error == Errors.UNKNOWN) {
                    log.warn("Unknown error fetching data for topic-partition {}", (Object)tp);
                    break block17;
                }
                throw new IllegalStateException("Unexpected error code " + error.code() + " while fetching data");
            }
            finally {
                completedFetch.metricAggregator.record(tp, bytes, recordsCount);
            }
        }
        if (bytes > 0 || error != Errors.NONE) {
            this.subscriptions.movePartitionToEnd(tp);
        }
        return parsedRecords;
    }

    private ConsumerRecord<K, V> parseRecord(TopicPartition partition, LogEntry logEntry) {
        Record record = logEntry.record();
        if (this.checkCrcs) {
            try {
                record.ensureValid();
            }
            catch (InvalidRecordException e) {
                throw new KafkaException("Record for partition " + partition + " at offset " + logEntry.offset() + " is invalid, cause: " + e.getMessage());
            }
        }
        try {
            long offset = logEntry.offset();
            long timestamp = record.timestamp();
            TimestampType timestampType = record.timestampType();
            ByteBuffer keyBytes = record.key();
            byte[] keyByteArray = keyBytes == null ? null : Utils.toArray(keyBytes);
            Object key = keyBytes == null ? null : (Object)this.keyDeserializer.deserialize(partition.topic(), keyByteArray);
            ByteBuffer valueBytes = record.value();
            byte[] valueByteArray = valueBytes == null ? null : Utils.toArray(valueBytes);
            Object value = valueBytes == null ? null : (Object)this.valueDeserializer.deserialize(partition.topic(), valueByteArray);
            return new ConsumerRecord<Object, Object>(partition.topic(), partition.partition(), offset, timestamp, timestampType, record.checksum(), keyByteArray == null ? -1 : keyByteArray.length, valueByteArray == null ? -1 : valueByteArray.length, key, value);
        }
        catch (RuntimeException e) {
            throw new SerializationException("Error deserializing key/value for partition " + partition + " at offset " + logEntry.offset(), e);
        }
    }

    private static class FetchManagerMetrics {
        public final Metrics metrics;
        public final String metricGrpName;
        public final Sensor bytesFetched;
        public final Sensor recordsFetched;
        public final Sensor fetchLatency;
        public final Sensor recordsFetchLag;
        public final Sensor fetchThrottleTimeSensor;

        public FetchManagerMetrics(Metrics metrics, String metricGrpPrefix) {
            this.metrics = metrics;
            this.metricGrpName = metricGrpPrefix + "-fetch-manager-metrics";
            this.bytesFetched = metrics.sensor("bytes-fetched");
            this.bytesFetched.add(metrics.metricName("fetch-size-avg", this.metricGrpName, "The average number of bytes fetched per request"), new Avg());
            this.bytesFetched.add(metrics.metricName("fetch-size-max", this.metricGrpName, "The maximum number of bytes fetched per request"), new Max());
            this.bytesFetched.add(metrics.metricName("bytes-consumed-rate", this.metricGrpName, "The average number of bytes consumed per second"), new Rate());
            this.recordsFetched = metrics.sensor("records-fetched");
            this.recordsFetched.add(metrics.metricName("records-per-request-avg", this.metricGrpName, "The average number of records in each request"), new Avg());
            this.recordsFetched.add(metrics.metricName("records-consumed-rate", this.metricGrpName, "The average number of records consumed per second"), new Rate());
            this.fetchLatency = metrics.sensor("fetch-latency");
            this.fetchLatency.add(metrics.metricName("fetch-latency-avg", this.metricGrpName, "The average time taken for a fetch request."), new Avg());
            this.fetchLatency.add(metrics.metricName("fetch-latency-max", this.metricGrpName, "The max time taken for any fetch request."), new Max());
            this.fetchLatency.add(metrics.metricName("fetch-rate", this.metricGrpName, "The number of fetch requests per second."), new Rate(new Count()));
            this.recordsFetchLag = metrics.sensor("records-lag");
            this.recordsFetchLag.add(metrics.metricName("records-lag-max", this.metricGrpName, "The maximum lag in terms of number of records for any partition in this window"), new Max());
            this.fetchThrottleTimeSensor = metrics.sensor("fetch-throttle-time");
            this.fetchThrottleTimeSensor.add(metrics.metricName("fetch-throttle-time-avg", this.metricGrpName, "The average throttle time in ms"), new Avg());
            this.fetchThrottleTimeSensor.add(metrics.metricName("fetch-throttle-time-max", this.metricGrpName, "The maximum throttle time in ms"), new Max());
        }

        public void recordTopicFetchMetrics(String topic, int bytes, int records) {
            String name = "topic." + topic + ".bytes-fetched";
            Sensor bytesFetched = this.metrics.getSensor(name);
            if (bytesFetched == null) {
                Map<String, String> metricTags = Collections.singletonMap("topic", topic.replace('.', '_'));
                bytesFetched = this.metrics.sensor(name);
                bytesFetched.add(this.metrics.metricName("fetch-size-avg", this.metricGrpName, "The average number of bytes fetched per request for topic " + topic, metricTags), new Avg());
                bytesFetched.add(this.metrics.metricName("fetch-size-max", this.metricGrpName, "The maximum number of bytes fetched per request for topic " + topic, metricTags), new Max());
                bytesFetched.add(this.metrics.metricName("bytes-consumed-rate", this.metricGrpName, "The average number of bytes consumed per second for topic " + topic, metricTags), new Rate());
            }
            bytesFetched.record(bytes);
            name = "topic." + topic + ".records-fetched";
            Sensor recordsFetched = this.metrics.getSensor(name);
            if (recordsFetched == null) {
                HashMap<String, String> metricTags = new HashMap<String, String>(1);
                metricTags.put("topic", topic.replace('.', '_'));
                recordsFetched = this.metrics.sensor(name);
                recordsFetched.add(this.metrics.metricName("records-per-request-avg", this.metricGrpName, "The average number of records in each request for topic " + topic, metricTags), new Avg());
                recordsFetched.add(this.metrics.metricName("records-consumed-rate", this.metricGrpName, "The average number of records consumed per second for topic " + topic, metricTags), new Rate());
            }
            recordsFetched.record(records);
        }
    }

    private static class FetchResponseMetricAggregator {
        private final FetchManagerMetrics sensors;
        private final Set<TopicPartition> unrecordedPartitions;
        private int totalBytes;
        private int totalRecords;

        public FetchResponseMetricAggregator(FetchManagerMetrics sensors, Set<TopicPartition> partitions) {
            this.sensors = sensors;
            this.unrecordedPartitions = partitions;
        }

        public void record(TopicPartition partition, int bytes, int records) {
            this.unrecordedPartitions.remove(partition);
            this.totalBytes += bytes;
            this.totalRecords += records;
            if (this.unrecordedPartitions.isEmpty()) {
                this.sensors.bytesFetched.record(this.totalBytes);
                this.sensors.recordsFetched.record(this.totalRecords);
            }
        }
    }

    private static class CompletedFetch {
        private final TopicPartition partition;
        private final long fetchedOffset;
        private final FetchResponse.PartitionData partitionData;
        private final FetchResponseMetricAggregator metricAggregator;

        public CompletedFetch(TopicPartition partition, long fetchedOffset, FetchResponse.PartitionData partitionData, FetchResponseMetricAggregator metricAggregator) {
            this.partition = partition;
            this.fetchedOffset = fetchedOffset;
            this.partitionData = partitionData;
            this.metricAggregator = metricAggregator;
        }
    }

    private static class PartitionRecords<K, V> {
        private long fetchOffset;
        private TopicPartition partition;
        private List<ConsumerRecord<K, V>> records;
        private int position = 0;

        public PartitionRecords(long fetchOffset, TopicPartition partition, List<ConsumerRecord<K, V>> records) {
            this.fetchOffset = fetchOffset;
            this.partition = partition;
            this.records = records;
        }

        private boolean isDrained() {
            return this.records == null || this.position >= this.records.size();
        }

        private void drain() {
            this.records = null;
        }

        private List<ConsumerRecord<K, V>> drainRecords(int n) {
            if (this.isDrained()) {
                return Collections.emptyList();
            }
            int limit = Math.min(this.records.size(), this.position + n);
            List<ConsumerRecord<K, V>> res = Collections.unmodifiableList(this.records.subList(this.position, limit));
            this.position = limit;
            if (this.position < this.records.size()) {
                this.fetchOffset = this.records.get(this.position).offset();
            }
            return res;
        }
    }
}

