package com.nimbusds.openid.connect.provider.jwkset.validator;


import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jose.jwk.JWKMatcher;
import com.nimbusds.jose.jwk.JWKSelector;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.openid.connect.provider.jwkset.JWKSetSpec;


/**
 * Connect2id server OpenID Provider / OAuth 2.0 authorisation server JWK set
 * validator.
 */
public final class JWKSetValidator {
	
	
	/**
	 * Validates the minimum required JWK set for a Connect2id server
	 * OpenID Provider / OAuth 2.0 authorisation server.
	 *
	 * <ul>
	 *     <li>One signing RSA key pair
	 *     <li>One HMAC key
	 *     <li>One subject encryption AES key
	 *     <li>One refresh token encryption AES key
	 * </ul>
	 *
	 * @see JWKSetSpec
	 *
	 * @param jwkSet The JWK set to validate.
	 *
	 * @throws JOSEException If validation failed.
	 */
	public static void validateMinimumRequired(final JWKSet jwkSet)
		throws JOSEException {
		
		JWKMatcher rsaSigningKeyMatcher = JWKSetSpec.RotatedRSASigning.createKeyMatcher(JWSAlgorithm.RS256);
		
		if (new JWKSelector(rsaSigningKeyMatcher).select(jwkSet).isEmpty()) {
			throw new JOSEException("The JWK set must contain at least one signing RSA key pair with properties: " + rsaSigningKeyMatcher);
		}
		
		if (new JWKSelector(JWKSetSpec.HMAC.KEY_MATCHER).select(jwkSet).isEmpty()) {
			throw new JOSEException("The JWK set must contain one HMAC secret key with properties: " + JWKSetSpec.HMAC.KEY_MATCHER);
		}
		
		if (new JWKSelector(JWKSetSpec.SubjectEncryption.KEY_MATCHER).select(jwkSet).isEmpty()) {
			throw new JOSEException("The JWK set must contain one subject encryption AES key with properties: " + JWKSetSpec.SubjectEncryption.KEY_MATCHER);
		}
		
		if (new JWKSelector(JWKSetSpec.RefreshTokenEncryption.KEY_MATCHER).select(jwkSet).isEmpty()) {
			throw new JOSEException("The JWK set must contain one refresh token encryption AES key with properties: " + JWKSetSpec.RefreshTokenEncryption.KEY_MATCHER);
		}
	}
}
