package com.nimbusds.openid.connect.provider.spi.grants;


import java.util.*;

import net.jcip.annotations.Immutable;

import net.minidev.json.JSONObject;

import org.apache.commons.collections4.CollectionUtils;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.util.JSONObjectUtils;


/**
 * Basic OpenID Connect claims specification.
 */
@Immutable
public class BasicClaimsSpec {


	/**
	 * The names of the authorised OpenID Connect claims, {@code null} if
	 * none.
	 */
	private final Set<String> names;


	/**
	 * Additional or preset OpenID Connect claims to be included in the
	 * ID token and UserInfo response.
	 */
	private final PresetClaims presetClaims;


	/**
	 * Creates a new default basic OpenID Connect claims specification
	 * (empty).
	 */
	public BasicClaimsSpec() {

		this(null, null, null);
	}


	/**
	 * Creates a new basic OpenID Connect claims specification.
	 *
	 * @param names The names of the authorised OpenID Connect claims,
	 *              {@code null} if none.
	 */
	public BasicClaimsSpec(final Set<String> names) {

		this(names, null, null);
	}


	/**
	 * Creates a new basic OpenID Connect claims specification.
	 *
	 * @param names                The names of the authorised OpenID
	 *                             Connect claims, empty set or
	 *                             {@code null} if none.
	 * @param presetIDTokenClaims  Additional preset claims to be included
	 *                             in the ID token, {@code null} if none.
	 * @param presetUserInfoClaims Additional preset claims to be included
	 *                             in the UserInfo response, {@code null}
	 *                             if none.
	 */
	public BasicClaimsSpec(final Set<String> names,
			       final JSONObject presetIDTokenClaims,
			       final JSONObject presetUserInfoClaims) {

		this(names, new PresetClaims(presetIDTokenClaims, presetUserInfoClaims));
	}


	/**
	 * Creates a new basic OpenID Connect claims specification.
	 *
	 * @param names        The names of the authorised OpenID Connect
	 *                     claims, empty set or {@code null} if none.
	 * @param presetClaims The additional or preset claims to be included
	 *                     in the ID token and UserInfo response,
	 *                     {@code null} if none.
	 */
	public BasicClaimsSpec(final Set<String> names,
			       final PresetClaims presetClaims) {

		if (CollectionUtils.isNotEmpty(names)) {
			this.names = names;
		} else {
			this.names = Collections.emptySet();
		}

		if (presetClaims == null) {
			this.presetClaims = new PresetClaims();
		} else {
			this.presetClaims = presetClaims;
		}
	}


	/**
	 * Returns the authorised OpenID Connect claims.
	 *
	 * @return The names of the authorised OpenID Connect claims, empty set
	 *         if none.
	 */
	public Set<String> getNames() {

		return names;
	}


	/**
	 * The additional or preset claims to be included in the ID token and
	 * UserInfo response.
	 *
	 * @return The additional or preset claims.
	 */
	public PresetClaims getPresetClaims() {

		return presetClaims;
	}


	/**
	 * Returns the additional preset claims to be included in the ID token.
	 *
	 * @return The additional preset claims to be included in the ID token,
	 *         {@code null} if none.
	 */
	public JSONObject getPresetIDTokenClaims() {

		return presetClaims.getPresetIDTokenClaims();
	}


	/**
	 * Returns the additional preset claims to be included in the UserInfo
	 * response.
	 *
	 * @return The additional or preset claims to be included in the
	 *         UserInfo response, {@code null} if none.
	 */
	public JSONObject getPresetUserInfoClaims() {

		return presetClaims.getPresetUserInfoClaims();
	}


	/**
	 * Returns a JSON object representation of this basic claims
	 * specification.
	 *
	 * @return The JSON object.
	 */
	public JSONObject toJSONObject() {

		JSONObject o = new JSONObject();

		if (CollectionUtils.isNotEmpty(names)) {
			o.put("claims", new ArrayList<>(names));
		}

		if (! presetClaims.isEmpty()) {
			o.put("preset_claims", presetClaims.toJSONObject());
		}

		return o;
	}


	/**
	 * Parses a basic OpenID Connect claims specification from the
	 * specified JSON object.
	 *
	 * @param o The JSON object. Must not be {@code null}.
	 *
	 * @return The basic OpenID Connect claims specification.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static BasicClaimsSpec parse(final JSONObject o)
		throws ParseException {

		Set<String> claims = null;

		if (o.containsKey("claims")) {
			claims = new HashSet<>(Arrays.asList(JSONObjectUtils.getStringArray(o, "claims")));
		}

		PresetClaims presetClaims = null;

		if (o.containsKey("preset_claims")) {
			presetClaims = PresetClaims.parse(JSONObjectUtils.getJSONObject(o, "preset_claims"));
		}

		return new BasicClaimsSpec(claims, presetClaims);
	}
}
