package com.nimbusds.openid.connect.provider.spi.grants;


import java.util.*;

import net.jcip.annotations.Immutable;

import net.minidev.json.JSONObject;

import com.nimbusds.langtag.LangTag;
import com.nimbusds.langtag.LangTagException;
import com.nimbusds.langtag.LangTagUtils;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.util.JSONObjectUtils;
import com.nimbusds.openid.connect.sdk.claims.ClaimsTransport;


/**
 * OpenID Connect claims specification.
 */
@Immutable
public class ClaimsSpec extends BasicClaimsSpec {


	/**
	 * None (empty) claims specification.
	 */
	public static final ClaimsSpec NONE = new ClaimsSpec();


	/**
	 * The preferred claims locales, {@code null} if not specified.
	 */
	private final List<LangTag> locales;


	/**
	 * The preferred claims transport.
	 */
	private final ClaimsTransport transport;


	/**
	 * Creates a new default OpenID Connect claims specification (empty).
	 */
	public ClaimsSpec() {

		this(null, null, null, null, ClaimsTransport.getDefault());
	}


	/**
	 * Creates a new OpenID Connect claims specification.
	 *
	 * @param names The names of the authorised OpenID Connect claims,
	 *              {@code null} if none.
	 */
	public ClaimsSpec(final Set<String> names) {

		this(names, null, null, null, ClaimsTransport.getDefault());
	}


	/**
	 * Creates a new OpenID Connect claims specification.
	 * 
	 * @param names                The names of the authorised OpenID
	 *                             Connect claims, {@code null} if none.
	 * @param locales              The preferred claims locales,
	 *                             {@code null} if not specified.
	 * @param presetIDTokenClaims  Additional or preset claims to be
	 *                             included in the ID token, {@code null}
	 *                             if none.
	 * @param presetUserInfoClaims Additional or preset claims to be
	 *                             included in the UserInfo response,
	 *                             {@code null} if none.
	 * @param transport            The preferred claims transport. Must not
	 *                             be {@code null}.
	 */
	public ClaimsSpec(final Set<String> names,
			  final List<LangTag> locales,
			  final JSONObject presetIDTokenClaims, 
			  final JSONObject presetUserInfoClaims, 
			  final ClaimsTransport transport) {

		super(names, presetIDTokenClaims, presetUserInfoClaims);

		this.locales = locales;
		this.transport = transport;
	}


	/**
	 * Returns the preferred OpenID Connect claims locales.
	 *
	 * @return The preferred OpenID Connect claims locales, {@code null} if
	 *         not specified.
	 */
	public List<LangTag> getLocales() {

		return locales;
	}


	/**
	 * Returns the preferred claims transport.
	 *
	 * @return The preferred claims transport.
	 */
	public ClaimsTransport getTransport() {
		
		return transport;
	}


	/**
	 * Returns a JSON object representation of this claims specification.
	 *
	 * @return The JSON object.
	 */
	public JSONObject toJSONObject() {

		JSONObject o = super.toJSONObject();

		if (locales != null) {
			o.put("claims_locales", LangTagUtils.toStringList(locales));
		}

		o.put("claims_transport", transport.toString());

		return o;
	}


	/**
	 * Parses an OpenID Connect claims specification from the specified
	 * JSON object.
	 *
	 * @param o The JSON object. Must not be {@code null}.
	 *
	 * @return The OpenID Connect claims specification.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static ClaimsSpec parse(final JSONObject o)
		throws ParseException {

		BasicClaimsSpec basicSpec = BasicClaimsSpec.parse(o);

		List<LangTag> claimsLocales = null;

		if (JSONObjectUtils.containsKey(o, "claims_locales")) {

			try {
				claimsLocales = LangTagUtils.parseLangTagList(JSONObjectUtils.getStringArray(o, "claims_locales"));

			} catch (LangTagException e) {

				throw new ParseException("Invalid claims locales value: " + e.getMessage(), e);
			}
		}

		ClaimsTransport claimsTransport;

		if (o.containsKey("claims_transport")) {
			String c = JSONObjectUtils.getString(o, "claims_transport");

			try {
				claimsTransport = ClaimsTransport.valueOf(c.toUpperCase());
			} catch (IllegalArgumentException e) {
				throw new ParseException("Invalid claims transport");
			}
		} else {
			// Defaults to UserInfo if not specified
			claimsTransport = ClaimsTransport.getDefault();
		}

		return new ClaimsSpec(
			basicSpec.getNames(),
			claimsLocales,
			basicSpec.getPresetIDTokenClaims(),
			basicSpec.getPresetUserInfoClaims(),
			claimsTransport);
	}
}
