package com.nimbusds.openid.connect.provider.spi.grants;


import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.nimbusds.oauth2.sdk.ParseException;
import com.nimbusds.oauth2.sdk.id.Subject;
import com.nimbusds.oauth2.sdk.util.JSONObjectUtils;
import com.nimbusds.openid.connect.sdk.claims.ACR;
import com.nimbusds.openid.connect.sdk.claims.AMR;
import net.jcip.annotations.Immutable;
import net.minidev.json.JSONObject;
import org.apache.commons.collections4.CollectionUtils;


/**
 * Identity (ID) token specification.
 */
@Immutable
public class IDTokenSpec extends OptionalTokenSpec {


	/**
	 * None (no issue) ID token specification.
	 */
	public static final IDTokenSpec NONE = new IDTokenSpec();


	/**
	 * The time of the subject authentication. If {@code null} it will be
	 * set to now.
	 */
	private final Date authTime;


	/**
	 * The Authentication Context Class Reference (ACR), {@code null} if
	 * not specified.
	 */
	private final ACR acr;


	/**
	 * The Authentication Methods Reference (AMR) list, {@code null} if not
	 * specified.
	 */
	private final List<AMR> amrList;


	/**
	 * Creates a new default ID token specification (no issue).
	 */
	public IDTokenSpec() {

		this(false, 0L, null, null, null, null);
	}


	/**
	 * Creates a new ID token specification.
	 *
	 * @param issue               Controls the ID token issue. If
	 *                            {@code true} an ID token must be issued,
	 *                            {@code false} to prohibit issue.
	 * @param lifetime            The ID token lifetime, in seconds, zero
	 *                            if not specified (to let the Connect2id
	 *                            server apply the default configured
	 *                            lifetime for ID tokens).
	 * @param impersonatedSubject The subject in impersonation and
	 *                            delegation cases, {@code null} if not
	 *                            applicable.
	 */
	public IDTokenSpec(final boolean issue,
			   long lifetime,
			   final Subject impersonatedSubject) {

		this(issue, lifetime, null, null, null, impersonatedSubject);
	}


	/**
	 * Creates a new ID token specification.
	 *
	 * @param issue               Controls the ID token issue. If
	 *                            {@code true} an ID token must be issued,
	 *                            {@code false} to prohibit issue.
	 * @param lifetime            The ID token lifetime, in seconds, zero
	 *                            if not specified (to let the Connect2id
	 *                            server apply the default configured
	 *                            lifetime for ID tokens).
	 * @param authTime            The time of the subject authentication.
	 *                            If {@code null} it will be set to now.
	 *                            Applies only if an ID token is issued.
	 * @param acr                 The Authentication Context Class
	 *                            Reference (ACR), {@code null} if not
	 *                            specified. Applies only if an ID token is
	 *                            issued.
	 * @param amrList             The Authentication Methods Reference
	 *                            (AMR) list, {@code null} if not
	 *                            specified. Applies only if an ID token is
	 *                            issued.
	 * @param impersonatedSubject The subject in impersonation and
	 *                            delegation cases, {@code null} if not
	 *                            applicable.
	 */
	public IDTokenSpec(final boolean issue,
			   final long lifetime,
			   final Date authTime,
			   final ACR acr,
			   final List<AMR> amrList,
			   final Subject impersonatedSubject) {

		super(issue, lifetime, null, impersonatedSubject);

		if (issue) {
			// Applies only if a token is issued
			this.authTime = authTime;
			this.acr = acr;
			this.amrList = amrList;
		} else {
			this.authTime = null;
			this.acr = null;
			this.amrList = null;
		}
	}


	/**
	 * Returns the time of the subject authentication.
	 *
	 * @return The time of the subject authentication. If {@code null} it
	 *         will be set to now. Applies only if an ID token is issued.
	 */
	public Date getAuthTime() {

		return authTime;
	}


	/**
	 * Returns the Authentication Context Class Reference (ACR).
	 *
	 * @return The Authentication Context Class Reference (ACR),
	 *         {@code null} if not specified. Applies only if an ID token
	 *         is issued.
	 */
	public ACR getACR() {

		return acr;
	}


	/**
	 * Returns The Authentication Methods Reference (AMR) list.
	 *
	 * @return The Authentication Methods Reference (AMR) list,
	 *         {@code null} if not specified. Applies only if an ID token
	 *         is issued.
	 */
	public List<AMR> getAMRList() {

		return amrList;
	}


	@Override
	public JSONObject toJSONObject() {

		JSONObject o = super.toJSONObject();

		if (authTime != null) {
			o.put("auth_time", authTime.getTime() / 1000L);
		}

		if (acr != null) {
			o.put("acr", acr.getValue());
		}

		if (CollectionUtils.isNotEmpty(amrList)) {

			List<String> sl = new ArrayList<>(amrList.size());

			for (AMR amr: amrList) {
				sl.add(amr.getValue());
			}

			o.put("amr", sl);
		}

		return o;
	}


	/**
	 * Parses an ID token specification from the specified JSON object.
	 *
	 * @param jsonObject The JSON object. Must not be {@code null}.
	 *
	 * @return The ID token specification.
	 *
	 * @throws ParseException If parsing failed.
	 */
	public static IDTokenSpec parse(final JSONObject jsonObject)
		throws ParseException {

		OptionalTokenSpec optionalTokenSpec = OptionalTokenSpec.parse(jsonObject);

		Date authTime = null;

		if (jsonObject.containsKey("auth_time")) {
			authTime = new Date(JSONObjectUtils.getLong(jsonObject, "auth_time") * 1000L);
		}

		ACR acr = null;

		if (jsonObject.containsKey("acr")) {
			acr = new ACR(JSONObjectUtils.getString(jsonObject, "acr"));
		}

		List<AMR> amrList = null;

		if (jsonObject.containsKey("amr")) {
			String[] sa = JSONObjectUtils.getStringArray(jsonObject, "amr");
			amrList = new ArrayList<>(sa.length);
			for (String s: sa) {
				amrList.add(new AMR(s));
			}
		}

		return new IDTokenSpec(
			optionalTokenSpec.issue(),
			optionalTokenSpec.getLifetime(),
			authTime,
			acr,
			amrList,
			optionalTokenSpec.getImpersonatedSubject());
	}
}
