package com.nimbusds.openid.connect.provider.spi.par;


import net.jcip.annotations.ThreadSafe;

import com.nimbusds.oauth2.sdk.AuthorizationRequest;
import com.nimbusds.oauth2.sdk.GeneralException;


/**
 * Service Provider Interface (SPI) for performing additional validation of
 * Pushed Authorisation Requests (PAR).
 *
 * <p>The {@link #validate} method will be called after the Connect2id server
 * has performed standard validation of the OAuth 2.0 authorisation / OpenID
 * authentication request, such as such as checking the {@code client_id} and
 * ensuring the client is authorised the use the OAuth 2.0 grant. JWT-secured
 * authorisation requests (JAR) will be unwrapped / resolved before that.
 *
 * <p>The {@link #validate} method can reject the request by throwing a
 * {@link GeneralException} with an appropriate HTTP status code and error
 * code. The exception message will be logged and not output to the client.
 *
 * <p>Example:
 *
 * <pre>
 * throw new GeneralException("Scope not accepted scope", // will be logged
 * 	OAuth2Error.INVALID_SCOPE
 * 	.setHTTPStatusCode(400)
 * 	.setDescription("Scope not accepted: some_scope"));
 * </pre>
 *
 * The resulting HTTP response:
 *
 * <pre>
 * HTTP/1.1 400 Bad Request
 * Content-Type: application/json;charset=UTF-8
 * Cache-Control: no-store
 * Pragma: no-cache
 *
 * {
 *   "error"             : "invalid_scope",
 *   "error_description" : "Scope not accepted: some_scope"
 * }
 * </pre>
 *
 * <p>Implementations must be thread-safe.
 */
@ThreadSafe
public interface PARValidator {
	
	
	/**
	 * Validates the specified OAuth 2.0 authorisation / OpenID
	 * authentication request.
	 *
	 * @param authzRequest The request to perform additional validation on.
	 *                     Not {@code null}.
	 * @param validatorCtx The PAR validator context. Not {@code null}.
	 *
	 * @throws GeneralException If the request is rejected. Should include
	 *                          an appropriate HTTP status code and error
	 *                          code.
	 */
	void validate(final AuthorizationRequest authzRequest, final ValidatorContext validatorCtx)
		throws GeneralException;
}
