package com.nimbusds.openid.connect.provider.spi.tokens;


import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.nimbusds.langtag.LangTag;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.auth.X509CertificateConfirmation;
import com.nimbusds.oauth2.sdk.id.*;
import net.minidev.json.JSONObject;


/**
 * Access token authorisation.
 */
public interface AccessTokenAuthorization {
	
	
	/**
	 * Returns the token subject.
	 *
	 * @return The subject, {@code null} if not specified.
	 */
	Subject getSubject();
	
	
	/**
	 * Returns the token actor, in impersonation and delegation scenarios.
	 *
	 * @return The actor, {@code null} if not specified.
	 */
	Actor getActor();
	
	
	/**
	 * Returns the identifier of the client to which the token is issued.
	 *
	 * @return The client identifier, {@code null} if not specified.
	 */
	ClientID getClientID();
	
	
	/**
	 * Returns the scope of the token.
	 *
	 * @return The scope, {@code null} if not specified.
	 */
	Scope getScope();
	
	
	/**
	 * Returns the expiration time of the token.
	 *
	 * @return The expiration time, {@code null} if not specified.
	 */
	Instant getExpirationTime();
	
	
	/**
	 * Returns the issue time of the token.
	 *
	 * @return The issue time, {@code null} if not specified.
	 */
	Instant getIssueTime();
	
	
	/**
	 * Returns the issuer of the token.
	 *
	 * @return The issuer, {@code null} if not specified.
	 */
	Issuer getIssuer();
	
	
	/**
	 * Returns the audience list of the token, which may be the logical
	 * names of the intended resource servers.
	 *
	 * @return The audience list, {@code null} if not specified.
	 */
	List<Audience> getAudienceList();
	
	
	/**
	 * Returns the JSON Web Token (JWT) identifier of the token.
	 *
	 * @return The JWT ID, {@code null} if not specified or applicable.
	 */
	JWTID getJWTID();
	
	
	/**
	 * Returns the names of the consented OpenID claims to be accessed at
	 * the UserInfo endpoint.
	 *
	 * @return The claim names, {@code null} if not specified.
	 */
	Set<String> getClaimNames();
	
	
	/**
	 * Returns the preferred locales for the consented OpenID claims.
	 *
	 * @return The preferred claims locales, {@code null} if not specified.
	 */
	List<LangTag> getClaimsLocales();
	
	
	/**
	 * Returns the preset OpenID claims to be included in the UserInfo
	 * response.
	 *
	 * @return The preset OpenID claims, {@code null} if not specified.
	 */
	JSONObject getPresetClaims();
	
	
	/**
	 * Returns the optional data for the token.
	 *
	 * @return The optional data, represented as a JSON object,
	 *         {@code null} if not specified.
	 */
	JSONObject getData();
	
	
	/**
	 * Returns the client X.509 certificate confirmation (SHA-256
	 * thumbprint) for mutual TLS.
	 *
	 * @return The client X.509 certificate confirmation, {@code null} if
	 *         not specified.
	 */
	X509CertificateConfirmation getClientCertificateConfirmation();
	
	
	/**
	 * Returns a map of other top-level parameters.
	 *
	 * @return Other top-level parameters, the values should map to JSON
	 *         entities, {@code null} if none.
	 */
	default Map<String,Object> getOtherTopLevelParameters() {
		return null;
	}
}
