package com.nimbusds.openid.connect.provider.spi.tokens;


import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.nimbusds.langtag.LangTag;
import com.nimbusds.oauth2.sdk.Scope;
import com.nimbusds.oauth2.sdk.auth.X509CertificateConfirmation;
import com.nimbusds.oauth2.sdk.id.*;
import net.minidev.json.JSONObject;
import org.apache.commons.lang3.builder.ToStringBuilder;


/**
 * Mutable access token authorisation.
 */
public final class MutableAccessTokenAuthorization implements AccessTokenAuthorization {
	
	
	private Subject sub;
	
	
	private Actor act;
	
	
	private ClientID clientID;
	
	
	private Scope scope;
	
	
	private Instant exp;
	
	
	private Instant iat;
	
	
	private Issuer iss;
	
	
	private List<Audience> audList;
	
	
	private JWTID jti;
	
	
	private Set<String> claimNames;
	
	
	private List<LangTag> claimsLocales;
	
	
	private JSONObject presetClaims;
	
	
	private JSONObject data;
	
	
	private X509CertificateConfirmation cnfX5t;
	
	
	private Map<String, Object> otherTopLevelParams;
	
	
	/**
	 * Creates a new empty mutable access token authorisation.
	 */
	public MutableAccessTokenAuthorization() {
	}
	
	
	/**
	 * Creates a new mutable access token authorisation from the specified
	 * one.
	 *
	 * @param source The source access token authorisation. Must not be
	 *               {@code null}.
	 */
	public MutableAccessTokenAuthorization(final AccessTokenAuthorization source) {
		sub = source.getSubject();
		act = source.getActor();
		clientID = source.getClientID();
		scope = source.getScope();
		exp = source.getExpirationTime();
		iat = source.getIssueTime();
		iss = source.getIssuer();
		audList = source.getAudienceList();
		jti = source.getJWTID();
		claimNames = source.getClaimNames();
		claimsLocales = source.getClaimsLocales();
		presetClaims = source.getPresetClaims();
		data = source.getData();
		cnfX5t = source.getClientCertificateConfirmation();
		otherTopLevelParams = source.getOtherTopLevelParameters();
	}
	
	
	/**
	 * Sets the token subject.
	 *
	 * @param sub The subject, {@code null} if not specified.
	 *            
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withSubject(final Subject sub) {
		this.sub = sub;
		return this;
	}
	
	
	@Override
	public Subject getSubject() {
		return sub;
	}
	
	
	/**
	 * Sets the token actor, in impersonation and delegation scenarios.
	 *
	 * @param act The actor, {@code null} if not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withActor(final Actor act) {
		this.act = act;
		return this;
	}
	
	
	@Override
	public Actor getActor() {
		return act;
	}
	
	
	/**
	 * Sets the identifier of the client to which the token is issued.
	 *
	 * @param clientID The client identifier, {@code null} if not
	 *                 specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withClientID(final ClientID clientID) {
		this.clientID = clientID;
		return this;
	}
	
	
	@Override
	public ClientID getClientID() {
		return clientID;
	}
	
	
	/**
	 * Sets the scope of the token.
	 *
	 * @param scope The scope, {@code null} if not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withScope(final Scope scope) {
		this.scope = scope;
		return this;
	}
	
	
	@Override
	public Scope getScope() {
		return scope;
	}
	
	
	/**
	 * Sets the expiration time of the token.
	 *
	 * @param exp The expiration time, {@code null} if not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withExpirationTime(final Instant exp) {
		this.exp = exp;
		return this;
	}
	
	
	@Override
	public Instant getExpirationTime() {
		return exp;
	}
	
	
	/**
	 * Sets the issue time of the token.
	 *
	 * @param iat The issue time, {@code null} if not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withIssueTime(final Instant iat) {
		this.iat = iat;
		return this;
	}
	
	
	@Override
	public Instant getIssueTime() {
		return iat;
	}
	
	
	/**
	 * Sets the issuer of the token.
	 *
	 * @param iss The issuer, {@code null} if not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withIssuer(final Issuer iss) {
		this.iss = iss;
		return this;
	}
	
	
	@Override
	public Issuer getIssuer() {
		return iss;
	}
	
	
	/**
	 * Sets the audience list of the token, which may be the logical
	 * names of the intended resource servers.
	 *
	 * @param audList The audience list, {@code null} if not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withAudienceList(final List<Audience> audList) {
		this.audList = audList;
		return this;
	}
	
	
	@Override
	public List<Audience> getAudienceList() {
		return audList;
	}
	
	
	/**
	 * Sets the JSON Web Token (JWT) identifier of the token.
	 *
	 * @param jti The JWT ID, {@code null} if not specified or applicable.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withJWTID(final JWTID jti) {
		this.jti = jti;
		return this;
	}
	
	
	@Override
	public JWTID getJWTID() {
		return jti;
	}
	
	
	/**
	 * Sets the names of the consented OpenID claims to be accessed at
	 * the UserInfo endpoint.
	 *
	 * @param claimNames The claim names, {@code null} if not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withClaimNames(final Set<String> claimNames) {
		this.claimNames = claimNames;
		return this;
	}
	
	
	@Override
	public Set<String> getClaimNames() {
		return claimNames;
	}
	
	
	/**
	 * Sets the preferred locales for the consented OpenID claims.
	 *
	 * @param claimsLocales The preferred claims locales, {@code null} if
	 *                      not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withClaimsLocales(final List<LangTag> claimsLocales) {
		this.claimsLocales = claimsLocales;
		return this;
	}
	
	
	@Override
	public List<LangTag> getClaimsLocales() {
		return claimsLocales;
	}
	
	
	/**
	 * Sets the preset OpenID claims to be included in the UserInfo
	 * response.
	 *
	 * @param presetClaims The preset OpenID claims, {@code null} if not
	 *                     specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withPresetClaims(final JSONObject presetClaims) {
		this.presetClaims = presetClaims;
		return this;
	}
	
	
	@Override
	public JSONObject getPresetClaims() {
		return presetClaims;
	}
	
	
	/**
	 * Sets the optional data for the token.
	 *
	 * @param data The optional data, represented as a JSON object,
	 *             {@code null} if not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withData(final JSONObject data) {
		this.data = data;
		return this;
	}
	
	
	@Override
	public JSONObject getData() {
		return data;
	}
	
	
	/**
	 * Sets the client X.509 certificate confirmation (SHA-256 thumbprint)
	 * for mutual TLS.
	 *
	 * @param cnfX5t The client X.509 certificate confirmation,
	 *               {@code null} if not specified.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withClientCertificateConfirmation(final X509CertificateConfirmation cnfX5t) {
		this.cnfX5t = cnfX5t;
		return this;
	}
	
	
	@Override
	public X509CertificateConfirmation getClientCertificateConfirmation() {
		return cnfX5t;
	}
	
	
	/**
	 * Sets the other top-level parameters.
	 *
	 * @param params Other top-level parameters, the values should map to
	 *               JSON entities, {@code null} if none.
	 *
	 * @return This object.
	 */
	public MutableAccessTokenAuthorization withOtherTopLevelParameters(final Map<String, Object> params) {
		otherTopLevelParams = params;
		return this;
	}
	
	
	@Override
	public Map<String, Object> getOtherTopLevelParameters() {
		return otherTopLevelParams;
	}
	
	
	@Override
	public String toString() {
		return new ToStringBuilder(this)
			.append("sub", sub)
			.append("act", act)
			.append("client_id", clientID)
			.append("scope", scope)
			.append("exp", exp)
			.append("iat", iat)
			.append("iss", iss)
			.append("aud", audList)
			.append("jti", jti)
			.append("claim_names", claimNames)
			.append("claims_locales", claimsLocales)
			.append("preset_claims", presetClaims)
			.append("data", data)
			.append("cnf", cnfX5t)
			.append("other", otherTopLevelParams)
			.toString();
	}
}
