/*
 * Decompiled with CFR 0.152.
 */
package com.nimbusds.infinispan.persistence.ldap.backend;

import com.codahale.metrics.MetricSet;
import com.codahale.metrics.Timer;
import com.codahale.metrics.health.HealthCheck;
import com.nimbusds.common.appendable.Appendable;
import com.nimbusds.common.ldap.LDAPConnectionPoolFactory;
import com.nimbusds.common.ldap.LDAPConnectionPoolMetrics;
import com.nimbusds.common.ldap.LDAPHealthCheck;
import com.nimbusds.common.monitor.MonitorRegistries;
import com.nimbusds.infinispan.persistence.ldap.LDAPStoreConfiguration;
import com.nimbusds.infinispan.persistence.ldap.Loggers;
import com.nimbusds.infinispan.persistence.ldap.backend.LDAPModifyRequestFactory;
import com.nimbusds.infinispan.persistence.ldap.backend.LDAPTimers;
import com.unboundid.asn1.ASN1OctetString;
import com.unboundid.ldap.sdk.Control;
import com.unboundid.ldap.sdk.DN;
import com.unboundid.ldap.sdk.DeleteRequest;
import com.unboundid.ldap.sdk.Entry;
import com.unboundid.ldap.sdk.Filter;
import com.unboundid.ldap.sdk.LDAPConnection;
import com.unboundid.ldap.sdk.LDAPConnectionPool;
import com.unboundid.ldap.sdk.LDAPException;
import com.unboundid.ldap.sdk.LDAPInterface;
import com.unboundid.ldap.sdk.LDAPResult;
import com.unboundid.ldap.sdk.LDAPSearchException;
import com.unboundid.ldap.sdk.ModifyRequest;
import com.unboundid.ldap.sdk.ReadOnlyEntry;
import com.unboundid.ldap.sdk.ResultCode;
import com.unboundid.ldap.sdk.SearchRequest;
import com.unboundid.ldap.sdk.SearchResult;
import com.unboundid.ldap.sdk.SearchResultEntry;
import com.unboundid.ldap.sdk.SearchScope;
import com.unboundid.ldap.sdk.controls.SimplePagedResultsControl;
import java.util.LinkedList;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import net.jcip.annotations.ThreadSafe;
import org.infinispan.persistence.spi.PersistenceException;

@ThreadSafe
public class LDAPConnector {
    public static final Filter MATCH_ANY_FILTER = Filter.createPresenceFilter((String)"objectClass");
    private final LDAPStoreConfiguration config;
    private final LDAPConnectionPool ldapConnPool;
    private final LDAPModifyRequestFactory ldapModifyRequestFactory;
    private final boolean supportAttributeOptions;
    private final LDAPTimers ldapTimers;
    private final String cacheName;

    public LDAPConnector(LDAPStoreConfiguration config, String cacheName, Set<String> attributes, boolean supportAttributeOptions) {
        this.config = config;
        this.cacheName = cacheName;
        this.supportAttributeOptions = supportAttributeOptions;
        LDAPConnectionPoolFactory factory = new LDAPConnectionPoolFactory(config.ldapServer, config.customTrustStore, config.customKeyStore, config.ldapUser);
        try {
            this.ldapConnPool = factory.createLDAPConnectionPool();
        }
        catch (Exception e) {
            throw new PersistenceException("LDAP connection pool creation for " + cacheName + " cache failed: " + e.getMessage(), (Throwable)e);
        }
        this.ldapConnPool.setConnectionPoolName(cacheName);
        this.checkBaseDN();
        this.ldapModifyRequestFactory = new LDAPModifyRequestFactory(attributes);
        this.ldapTimers = new LDAPTimers(cacheName + ".");
        String prefix = cacheName + ".ldapStore";
        MonitorRegistries.register((MetricSet)new LDAPConnectionPoolMetrics(this.ldapConnPool, prefix));
        MonitorRegistries.register((String)prefix, (HealthCheck)new LDAPHealthCheck((LDAPInterface)this.ldapConnPool, config.ldapDirectory.baseDN, Loggers.LDAP_LOG));
        Loggers.MAIN_LOG.info("[IL0100] Created new LDAP store connector for " + cacheName + " cache");
    }

    private void checkBaseDN() {
        try {
            if (this.ldapConnPool.getEntry(this.config.ldapDirectory.baseDN.toString()) == null) {
                Loggers.MAIN_LOG.warn("[IL0101] The configured LDAP store base DN for {} cache doesn't exist: {}", new Object[]{this.cacheName, this.config.ldapDirectory.baseDN});
            }
        }
        catch (LDAPException e) {
            Loggers.MAIN_LOG.warn("[IL0102] Couldn't verify the LDAP store base DN for {} cache: {}", new Object[]{this.cacheName, e.getMessage()});
        }
    }

    public LDAPConnectionPool getPool() {
        return this.ldapConnPool;
    }

    public ReadOnlyEntry retrieveEntry(DN dn) {
        Timer.Context timerCtx = this.ldapTimers.getTimer.time();
        try {
            SearchResultEntry searchResultEntry = this.ldapConnPool.getEntry(dn.toString(), new String[]{"*"});
            return searchResultEntry;
        }
        catch (LDAPException e) {
            throw new PersistenceException("LDAP get of " + dn + " failed: " + e.getResultString(), (Throwable)e);
        }
        finally {
            timerCtx.stop();
        }
    }

    public void retrieveEntries(Appendable<ReadOnlyEntry> appendable) {
        SearchRequest searchRequest = new SearchRequest(this.config.ldapDirectory.baseDN.toString(), SearchScope.ONE, MATCH_ANY_FILTER, new String[]{"*"});
        this.doSearch(searchRequest, appendable);
    }

    public boolean entryExists(DN dn) {
        Timer.Context timerCtx = this.ldapTimers.getTimer.time();
        try {
            boolean bl = this.ldapConnPool.getEntry(dn.toString(), new String[]{"1.1"}) != null;
            return bl;
        }
        catch (LDAPException e) {
            throw new PersistenceException("LDAP get of " + dn + " failed: " + e.getResultString(), (Throwable)e);
        }
        finally {
            timerCtx.stop();
        }
    }

    public boolean addEntry(ReadOnlyEntry entry) {
        LDAPResult ldapResult;
        Timer.Context timerCtx = this.ldapTimers.addTimer.time();
        try {
            ldapResult = this.ldapConnPool.add((Entry)entry);
        }
        catch (LDAPException e) {
            if (e.getResultCode().equals((Object)ResultCode.ENTRY_ALREADY_EXISTS)) {
                boolean bl = false;
                return bl;
            }
            throw new PersistenceException("LDAP add for " + entry.getDN() + " failed: " + e.getResultString(), (Throwable)e);
        }
        finally {
            timerCtx.stop();
        }
        ResultCode resultCode = ldapResult.getResultCode();
        if (resultCode.equals((Object)ResultCode.SUCCESS)) {
            return true;
        }
        if (resultCode.equals((Object)ResultCode.ENTRY_ALREADY_EXISTS)) {
            return false;
        }
        throw new PersistenceException("LDAP add for " + entry.getDN() + " failed: " + resultCode.getName());
    }

    public boolean replaceEntry(ReadOnlyEntry entry) {
        LDAPResult ldapResult;
        ModifyRequest modifyRequest;
        if (this.supportAttributeOptions) {
            DN dn;
            try {
                dn = new DN(entry.getDN());
            }
            catch (LDAPException e) {
                throw new PersistenceException(e.getMessage(), (Throwable)e);
            }
            ReadOnlyEntry existingEntry = this.retrieveEntry(dn);
            if (existingEntry == null) {
                return false;
            }
            modifyRequest = this.ldapModifyRequestFactory.composeModifyRequest((Entry)entry, (Entry)existingEntry);
            if (modifyRequest == null) {
                return true;
            }
        } else {
            modifyRequest = this.ldapModifyRequestFactory.composeModifyRequest((Entry)entry);
        }
        Timer.Context timerCtx = this.ldapTimers.modifyTimer.time();
        try {
            ldapResult = this.ldapConnPool.modify(modifyRequest);
        }
        catch (LDAPException e) {
            if (e.getResultCode().equals((Object)ResultCode.NO_SUCH_OBJECT)) {
                boolean bl = false;
                return bl;
            }
            throw new PersistenceException("LDAP modify for " + modifyRequest.getDN() + " failed: " + e.getResultString(), (Throwable)e);
        }
        finally {
            timerCtx.stop();
        }
        ResultCode resultCode = ldapResult.getResultCode();
        if (resultCode.equals((Object)ResultCode.SUCCESS)) {
            return true;
        }
        if (resultCode.equals((Object)ResultCode.NO_SUCH_OBJECT)) {
            return false;
        }
        throw new PersistenceException("LDAP modify " + modifyRequest.getDN() + " failed: " + resultCode.getName());
    }

    public boolean deleteEntry(DN dn) {
        LDAPResult result;
        DeleteRequest deleteRequest = new DeleteRequest(dn);
        Timer.Context timerCtx = this.ldapTimers.deleteTimer.time();
        try {
            result = this.ldapConnPool.delete(deleteRequest);
        }
        catch (LDAPException e) {
            ResultCode resultCode = e.getResultCode();
            if (resultCode.equals((Object)ResultCode.NO_SUCH_OBJECT)) {
                boolean bl = false;
                return bl;
            }
            throw new PersistenceException("LDAP delete of " + dn + " failed: " + e.getResultString(), (Throwable)e);
        }
        finally {
            timerCtx.stop();
        }
        ResultCode resultCode = result.getResultCode();
        if (resultCode.equals((Object)ResultCode.SUCCESS)) {
            return true;
        }
        if (resultCode.equals((Object)ResultCode.NO_SUCH_OBJECT)) {
            return false;
        }
        throw new PersistenceException("LDAP delete of " + dn + " failed: " + resultCode.getName());
    }

    protected static boolean indicatesConnectionException(LDAPException e) {
        return LDAPConnector.indicatesConnectionException(e.getResultCode());
    }

    protected static boolean indicatesConnectionException(ResultCode code) {
        return code.equals((Object)ResultCode.CONNECT_ERROR) || code.equals((Object)ResultCode.SERVER_DOWN) || code.equals((Object)ResultCode.TIMEOUT) || code.equals((Object)ResultCode.UNAVAILABLE);
    }

    private static ASN1OctetString parsePageCookie(SearchResult sr) {
        Control control = sr.getResponseControl("1.2.840.113556.1.4.319");
        if (control instanceof SimplePagedResultsControl) {
            SimplePagedResultsControl spr = (SimplePagedResultsControl)control;
            return spr.getCookie();
        }
        return null;
    }

    private void doSearch(SearchRequest searchRequest, Appendable<ReadOnlyEntry> appendable) {
        LDAPConnection connection;
        try {
            connection = this.ldapConnPool.getConnection();
        }
        catch (LDAPException e) {
            throw new PersistenceException(e.getMessage(), (Throwable)e);
        }
        ASN1OctetString cookie = null;
        do {
            SearchResult searchResult;
            searchRequest.replaceControl((Control)new SimplePagedResultsControl(this.config.ldapDirectory.pageSize, cookie));
            Timer.Context timerCtx = this.ldapTimers.searchTimer.time();
            try {
                searchResult = connection.search(searchRequest);
            }
            catch (LDAPSearchException e) {
                String msg = "[AS0109] LDAP search " + searchRequest.getFilter() + " failed: " + e.getMessage();
                if (LDAPConnector.indicatesConnectionException((LDAPException)((Object)e))) {
                    this.ldapConnPool.releaseDefunctConnection(connection);
                } else {
                    this.ldapConnPool.releaseConnection(connection);
                }
                throw new PersistenceException(msg, (Throwable)e);
            }
            finally {
                timerCtx.stop();
            }
            cookie = LDAPConnector.parsePageCookie(searchResult);
            searchResult.getSearchEntries().forEach(arg_0 -> appendable.append(arg_0));
        } while (cookie != null && cookie.getValueLength() > 0);
        this.ldapConnPool.releaseConnection(connection);
    }

    public int countEntries() {
        SearchRequest request = new SearchRequest(this.config.ldapDirectory.baseDN.toString(), SearchScope.ONE, MATCH_ANY_FILTER, new String[]{"1.1"});
        AtomicInteger count = new AtomicInteger();
        this.doSearch(request, (Appendable<ReadOnlyEntry>)((Appendable)entry -> count.incrementAndGet()));
        return count.intValue();
    }

    public int deleteEntries() {
        SearchRequest request = new SearchRequest(this.config.ldapDirectory.baseDN.toString(), SearchScope.ONE, MATCH_ANY_FILTER, new String[]{"1.1"});
        LinkedList entryDNs = new LinkedList();
        this.doSearch(request, (Appendable<ReadOnlyEntry>)((Appendable)entry -> entryDNs.add(entry.getDN())));
        int count = 0;
        for (String dn : entryDNs) {
            try {
                if (!this.deleteEntry(new DN(dn))) continue;
                ++count;
            }
            catch (LDAPException e) {
                throw new PersistenceException(e.getMessage(), (Throwable)e);
            }
        }
        return count;
    }

    public void shutdown() {
        this.ldapConnPool.close();
        if (this.ldapConnPool.isClosed()) {
            Loggers.MAIN_LOG.info("[IL0107] Shut down LDAP connector for {} cache", new Object[]{this.cacheName});
        } else {
            Loggers.MAIN_LOG.error("[IL0108] Attempted to shut down LDAP connector for {} cache, detected unreleased LDAP connections", new Object[]{this.cacheName});
        }
    }
}

