/*
 * Decompiled with CFR 0.152.
 */
package com.salesforce.cantor.s3;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.ListVersionsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.S3VersionSummary;
import com.amazonaws.services.s3.model.VersionListing;
import com.salesforce.cantor.common.CommonPreconditions;
import com.salesforce.cantor.common.ObjectsPreconditions;
import com.salesforce.cantor.s3.StreamingObjects;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ObjectsOnS3
implements StreamingObjects {
    private static final Logger logger = LoggerFactory.getLogger(ObjectsOnS3.class);
    private static final String bucketNameFormat = "%s-cantor-namespace-%d";
    private static final int streamingChunkSize = 0x400000;
    private final AmazonS3 s3Client;
    private final String bucketPrefix;
    private final String bucketNameAllNamespaces;

    public ObjectsOnS3(AmazonS3 s3Client) throws IOException {
        this(s3Client, "default");
    }

    public ObjectsOnS3(AmazonS3 s3Client, String bucketPrefix) throws IOException {
        CommonPreconditions.checkArgument((s3Client != null ? 1 : 0) != 0, (String)"null s3 client");
        CommonPreconditions.checkString((String)bucketPrefix, (String)"null/empty bucket prefix");
        this.s3Client = s3Client;
        this.bucketPrefix = bucketPrefix;
        this.bucketNameAllNamespaces = String.format("%s-all-namespaces", bucketPrefix);
        try {
            this.s3Client.createBucket(this.bucketNameAllNamespaces);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception creating required buckets for objects on s3:", (Throwable)e);
            throw new IOException("exception creating required buckets for objects on s3:", e);
        }
    }

    public Collection<String> namespaces() throws IOException {
        try {
            return this.doGetNamespaces();
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception getting namespaces:", (Throwable)e);
            throw new IOException("exception getting namespaces:", e);
        }
    }

    public void create(String namespace) throws IOException {
        CommonPreconditions.checkCreate((String)namespace);
        try {
            this.doCreate(namespace);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception creating namespace:", (Throwable)e);
            throw new IOException("exception creating namespace: ", e);
        }
    }

    public void drop(String namespace) throws IOException {
        CommonPreconditions.checkDrop((String)namespace);
        try {
            this.doDrop(namespace);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception dropping namespace:", (Throwable)e);
            throw new IOException("exception dropping namespace: ", e);
        }
    }

    public void store(String namespace, String key, byte[] bytes) throws IOException {
        ObjectsPreconditions.checkStore((String)namespace, (String)key, (byte[])bytes);
        try {
            this.doStore(namespace, key, bytes);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception storing namespace:", (Throwable)e);
            throw new IOException("exception storing namespace: ", e);
        }
    }

    public byte[] get(String namespace, String key) throws IOException {
        ObjectsPreconditions.checkGet((String)namespace, (String)key);
        try {
            return this.doGet(namespace, key);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception getting namespace:", (Throwable)e);
            throw new IOException("exception getting namespace: ", e);
        }
    }

    public boolean delete(String namespace, String key) throws IOException {
        ObjectsPreconditions.checkDelete((String)namespace, (String)key);
        try {
            return this.doDelete(namespace, key);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception deleting namespace:", (Throwable)e);
            throw new IOException("exception deleting namespace: ", e);
        }
    }

    public Collection<String> keys(String namespace, int start, int count) throws IOException {
        ObjectsPreconditions.checkKeys((String)namespace, (int)start, (int)count);
        try {
            return this.doKeys(namespace, start, count);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception getting keys:", (Throwable)e);
            throw new IOException("exception getting keys: ", e);
        }
    }

    public int size(String namespace) throws IOException {
        ObjectsPreconditions.checkSize((String)namespace);
        try {
            return this.doSize(namespace);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception getting size:", (Throwable)e);
            throw new IOException("exception getting size: ", e);
        }
    }

    @Override
    public void store(String namespace, String key, InputStream stream, long length) throws IOException {
        CommonPreconditions.checkString((String)namespace);
        CommonPreconditions.checkString((String)key);
        CommonPreconditions.checkArgument((stream != null ? 1 : 0) != 0, (String)"null stream");
        CommonPreconditions.checkArgument((length > 0L ? 1 : 0) != 0, (String)"zero/negative length");
        try {
            this.doStore(namespace, key, stream, length);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception storing stream:", (Throwable)e);
        }
    }

    @Override
    public InputStream stream(String namespace, String key) throws IOException {
        CommonPreconditions.checkString((String)namespace);
        CommonPreconditions.checkString((String)key);
        try {
            return this.doStream(namespace, key);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception streaming:", (Throwable)e);
            return null;
        }
    }

    private void doCreate(String namespace) throws IOException {
        String bucket = this.toBucketName(namespace);
        if (this.s3Client.doesBucketExistV2(bucket)) {
            logger.info("bucket '{}' already exists; ignoring create", (Object)bucket);
            return;
        }
        logger.info("bucket '{}' for namespace '{}' doesn't exist; creating it", (Object)bucket, (Object)namespace);
        this.s3Client.createBucket(bucket);
        if (!this.s3Client.doesBucketExistV2(bucket)) {
            throw new IOException("failed to create namespace on s3 with bucket name: " + bucket);
        }
        this.s3Client.putObject(this.bucketNameAllNamespaces, namespace, bucket);
    }

    private void doDrop(String namespace) throws IOException {
        String bucket = this.toBucketName(namespace);
        if (!this.s3Client.doesBucketExistV2(bucket)) {
            logger.debug("bucket '{}' does not exist; ignoring drop", (Object)bucket);
            return;
        }
        logger.info("bucket '{}' exists; dropping it", (Object)bucket);
        ObjectListing objectListing = this.s3Client.listObjects(bucket);
        while (true) {
            for (S3ObjectSummary summary : objectListing.getObjectSummaries()) {
                this.s3Client.deleteObject(bucket, summary.getKey());
            }
            if (!objectListing.isTruncated()) break;
            objectListing = this.s3Client.listNextBatchOfObjects(objectListing);
        }
        VersionListing versionList = this.s3Client.listVersions(new ListVersionsRequest().withBucketName(bucket));
        while (true) {
            for (S3VersionSummary summary : versionList.getVersionSummaries()) {
                this.s3Client.deleteVersion(bucket, summary.getKey(), summary.getVersionId());
            }
            if (!versionList.isTruncated()) break;
            versionList = this.s3Client.listNextBatchOfVersions(versionList);
        }
        this.s3Client.deleteBucket(bucket);
        if (this.s3Client.doesBucketExistV2(bucket)) {
            throw new IOException("failed to drop namespace on s3 with bucket name: " + bucket);
        }
        this.deleteObject(this.bucketNameAllNamespaces, namespace);
    }

    private void doStore(String namespace, String key, byte[] bytes) {
        String bucket = this.toBucketName(namespace);
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentLength((long)bytes.length);
        logger.info("storing {} object bytes at '{}.{}'", new Object[]{bytes.length, bucket, key});
        this.s3Client.putObject(bucket, key, (InputStream)new ByteArrayInputStream(bytes), metadata);
    }

    private void doStore(String namespace, String key, InputStream stream, long length) {
        String bucket = this.toBucketName(namespace);
        ObjectMetadata metadata = new ObjectMetadata();
        metadata.setContentLength(length);
        logger.info("storing stream with length={} at '{}.{}'", new Object[]{length, bucket, key});
        this.s3Client.putObject(bucket, key, stream, metadata);
    }

    private byte[] doGet(String namespace, String key) throws IOException {
        String bucket = this.toBucketName(namespace);
        logger.debug("retrieving object at '{}.{}'", (Object)bucket, (Object)key);
        if (!this.s3Client.doesObjectExist(bucket, key)) {
            logger.debug("object '{}.{}' doesn't exist, returning null", (Object)bucket, (Object)key);
            return null;
        }
        return this.getObjectBytes(bucket, key);
    }

    private InputStream doStream(String namespace, String key) throws IOException {
        String bucket = this.toBucketName(namespace);
        if (!this.s3Client.doesBucketExistV2(bucket)) {
            throw new IOException(String.format("couldn't find bucket '%s' for namespace '%s'", bucket, namespace));
        }
        S3Object object = this.s3Client.getObject(bucket, key);
        if (object == null) {
            logger.warn("object '{}.{}' should exist, but got null, returning null", (Object)bucket, (Object)key);
            throw new IOException(String.format("couldn't find S3 object with key '%s' in bucket '%s' for namespace '%s'", key, bucket, namespace));
        }
        return object.getObjectContent();
    }

    private byte[] getObjectBytes(String bucket, String key) throws IOException {
        ByteArrayOutputStream buffer;
        S3Object s3Object = this.s3Client.getObject(bucket, key);
        if (s3Object == null) {
            return null;
        }
        try (S3ObjectInputStream inputStream = s3Object.getObjectContent();){
            int read;
            buffer = new ByteArrayOutputStream();
            byte[] data = new byte[0x400000];
            while ((read = inputStream.read(data, 0, data.length)) != -1) {
                buffer.write(data, 0, read);
            }
        }
        buffer.flush();
        return buffer.toByteArray();
    }

    private boolean doDelete(String namespace, String key) throws IOException {
        String bucket = this.toBucketName(namespace);
        if (!this.s3Client.doesObjectExist(bucket, key)) {
            return false;
        }
        return this.deleteObject(bucket, key);
    }

    private int doSize(String namespace) {
        String bucket = this.toBucketName(namespace);
        if (!this.s3Client.doesBucketExistV2(bucket)) {
            return -1;
        }
        int totalSize = 0;
        ObjectListing listing = this.s3Client.listObjects(bucket);
        do {
            totalSize += listing.getObjectSummaries().size();
            logger.debug("got {} keys from {}", (Object)listing.getObjectSummaries().size(), (Object)listing);
        } while ((listing = this.s3Client.listNextBatchOfObjects(listing)).isTruncated());
        return totalSize;
    }

    private Collection<String> doKeys(String namespace, int start, int count) throws IOException {
        String bucket = this.toBucketName(namespace);
        if (!this.s3Client.doesBucketExistV2(bucket)) {
            throw new IOException(String.format("couldn't find bucket '%s' for namespace '%s'", bucket, namespace));
        }
        return this.getKeys(bucket, start, count);
    }

    private Collection<String> getKeys(String bucket, int start, int count) {
        HashSet<String> keys = new HashSet<String>();
        int index = 0;
        ObjectListing listing = this.s3Client.listObjects(bucket);
        do {
            for (S3ObjectSummary summary : listing.getObjectSummaries()) {
                if (index < start) {
                    logger.debug("skipping {} at index={} start={}", new Object[]{summary.getKey(), index++, start});
                    continue;
                }
                keys.add(summary.getKey());
                if (keys.size() != count) continue;
                logger.debug("retrieved {}/{} keys, returning early", (Object)keys.size(), (Object)count);
                return keys;
            }
            logger.debug("got {} keys from {}", (Object)listing.getObjectSummaries().size(), (Object)listing);
        } while ((listing = this.s3Client.listNextBatchOfObjects(listing)).isTruncated());
        return keys;
    }

    private boolean deleteObject(String bucket, String key) {
        this.s3Client.deleteObject(bucket, key);
        VersionListing versionList = this.s3Client.listVersions(bucket, key);
        for (S3VersionSummary summary : versionList.getVersionSummaries()) {
            logger.debug("deleting version {}", (Object)summary.getKey());
            this.s3Client.deleteVersion(bucket, summary.getKey(), summary.getVersionId());
        }
        return true;
    }

    private Collection<String> doGetNamespaces() {
        return this.getKeys(this.bucketNameAllNamespaces, 0, -1);
    }

    private String toBucketName(String namespace) {
        return String.format(bucketNameFormat, this.bucketPrefix, Math.abs(namespace.hashCode()));
    }
}

