/*
 * Decompiled with CFR 0.152.
 */
package com.salesforce.cantor.s3;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.CSVInput;
import com.amazonaws.services.s3.model.CSVOutput;
import com.amazonaws.services.s3.model.CompressionType;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.ExpressionType;
import com.amazonaws.services.s3.model.FileHeaderInfo;
import com.amazonaws.services.s3.model.GetObjectRequest;
import com.amazonaws.services.s3.model.InputSerialization;
import com.amazonaws.services.s3.model.JSONInput;
import com.amazonaws.services.s3.model.JSONOutput;
import com.amazonaws.services.s3.model.JSONType;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.OutputSerialization;
import com.amazonaws.services.s3.model.S3Object;
import com.amazonaws.services.s3.model.S3ObjectInputStream;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.amazonaws.services.s3.model.SelectObjectContentEvent;
import com.amazonaws.services.s3.model.SelectObjectContentEventVisitor;
import com.amazonaws.services.s3.model.SelectObjectContentRequest;
import com.amazonaws.services.s3.model.SelectObjectContentResult;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3Utils {
    private static final Logger logger = LoggerFactory.getLogger(S3Utils.class);
    private static final int streamingChunkSize = 0x400000;

    public static Collection<String> getKeys(AmazonS3 s3Client, String bucketName, String prefix) throws IOException {
        return S3Utils.getKeys(s3Client, bucketName, prefix, 0, -1);
    }

    public static Collection<String> getKeys(AmazonS3 s3Client, String bucketName, String prefix, int start, int count) throws IOException {
        if (!s3Client.doesBucketExistV2(bucketName)) {
            throw new IOException(String.format("couldn't find bucket '%s'", bucketName));
        }
        HashSet<String> keys = new HashSet<String>();
        int index = 0;
        ObjectListing listing = null;
        do {
            List objectSummaries;
            if ((objectSummaries = (listing = listing == null ? s3Client.listObjects(bucketName, prefix) : s3Client.listNextBatchOfObjects(listing)).getObjectSummaries()).size() - 1 + index < start) {
                logger.debug("skipping {} objects to index={}", (Object)objectSummaries.size(), (Object)(index += objectSummaries.size()));
                listing = s3Client.listNextBatchOfObjects(listing);
                continue;
            }
            for (S3ObjectSummary summary : objectSummaries) {
                if (start > index++) continue;
                keys.add(summary.getKey());
                if (keys.size() != count) continue;
                logger.debug("retrieved {}/{} keys, returning early", (Object)keys.size(), (Object)count);
                return keys;
            }
            logger.debug("got {} keys from {}", (Object)listing.getObjectSummaries().size(), (Object)listing);
        } while (listing.isTruncated());
        return keys;
    }

    public static byte[] getObjectBytes(AmazonS3 s3Client, String bucketName, String key) throws IOException {
        return S3Utils.getObjectBytes(s3Client, bucketName, key, 0L, -1L);
    }

    public static byte[] getObjectBytes(AmazonS3 s3Client, String bucketName, String key, long start, long end) throws IOException {
        ByteArrayOutputStream buffer;
        if (!s3Client.doesObjectExist(bucketName, key)) {
            logger.debug("object '{}.{}' doesn't exist, returning null", (Object)bucketName, (Object)key);
            return null;
        }
        GetObjectRequest request = new GetObjectRequest(bucketName, key);
        if (start >= 0L && end > 0L) {
            request.setRange(start, end);
        } else if (start >= 0L && end < 0L) {
            request.setRange(start);
        }
        S3Object s3Object = s3Client.getObject(request);
        try (S3ObjectInputStream inputStream = s3Object.getObjectContent();){
            int read;
            buffer = new ByteArrayOutputStream();
            byte[] data = new byte[0x400000];
            while ((read = inputStream.read(data, 0, data.length)) != -1) {
                buffer.write(data, 0, read);
            }
        }
        buffer.flush();
        return buffer.toByteArray();
    }

    public static InputStream getObjectStream(AmazonS3 s3Client, String bucketName, String key) {
        if (!s3Client.doesObjectExist(bucketName, key)) {
            logger.warn(String.format("couldn't find S3 object with key '%s' in bucket '%s'", key, bucketName));
            return null;
        }
        return s3Client.getObject(bucketName, key).getObjectContent();
    }

    public static void putObject(AmazonS3 s3Client, String bucketName, String key, InputStream content, ObjectMetadata metadata) throws IOException {
        if (!s3Client.doesBucketExistV2(bucketName)) {
            throw new IOException(String.format("couldn't find bucket '%s'", bucketName));
        }
        s3Client.putObject(bucketName, key, content, metadata);
    }

    public static boolean deleteObject(AmazonS3 s3Client, String bucketName, String key) {
        if (!s3Client.doesObjectExist(bucketName, key)) {
            return false;
        }
        s3Client.deleteObject(bucketName, key);
        return true;
    }

    public static void deleteObjects(AmazonS3 s3Client, String bucketName, Collection<String> keys) {
        if (keys == null || keys.isEmpty()) {
            return;
        }
        DeleteObjectsRequest request = new DeleteObjectsRequest(bucketName);
        request.setKeys(keys.stream().map(DeleteObjectsRequest.KeyVersion::new).collect(Collectors.toList()));
        s3Client.deleteObjects(request);
    }

    public static void deleteObjects(AmazonS3 s3Client, String bucketName, String prefix) {
        if (!s3Client.doesBucketExistV2(bucketName)) {
            logger.debug("bucket '{}' does not exist; ignoring drop", (Object)bucketName);
            return;
        }
        logger.info("bucket '{}' exists; dropping it", (Object)bucketName);
        ObjectListing objectListing = s3Client.listObjects(bucketName, prefix);
        while (true) {
            for (S3ObjectSummary summary : objectListing.getObjectSummaries()) {
                s3Client.deleteObject(bucketName, summary.getKey());
            }
            if (!objectListing.isTruncated()) break;
            objectListing = s3Client.listNextBatchOfObjects(objectListing);
        }
    }

    public static int getSize(AmazonS3 s3Client, String bucket, String bucketPrefix) {
        if (!s3Client.doesBucketExistV2(bucket)) {
            return -1;
        }
        int totalSize = 0;
        ObjectListing listing = null;
        do {
            listing = listing == null ? s3Client.listObjects(bucket, bucketPrefix) : s3Client.listNextBatchOfObjects(listing);
            totalSize += listing.getObjectSummaries().size();
            logger.debug("got {} keys from {}", (Object)listing.getObjectSummaries().size(), (Object)listing);
        } while (listing.isTruncated());
        return totalSize;
    }

    public static String getCleanKeyForNamespace(String namespace) {
        String cleanName = namespace.replaceAll("[^A-Za-z0-9_\\-]", "").toLowerCase();
        return String.format("cantor-%s-%s", cleanName.substring(0, Math.min(32, cleanName.length())), Math.abs(namespace.hashCode()));
    }

    public static class S3Select {
        public static InputStream queryObjectJson(AmazonS3 s3Client, String bucket, String key, String query) {
            return S3Select.queryObject(s3Client, S3Select.generateJsonRequest(bucket, key, query));
        }

        public static InputStream queryObjectCsv(AmazonS3 s3Client, String bucket, String key, String query) {
            return S3Select.queryObject(s3Client, S3Select.generateCsvRequest(bucket, key, query));
        }

        public static InputStream queryObject(AmazonS3 s3Client, final SelectObjectContentRequest request) {
            SelectObjectContentResult result = s3Client.selectObjectContent(request);
            return result.getPayload().getRecordsInputStream(new SelectObjectContentEventVisitor(){

                public void visit(SelectObjectContentEvent.StatsEvent event) {
                    logger.info("s3 select query stats: bucket={} key={} query={} bytes-scanned={} bytes-processed={}", new Object[]{request.getBucketName(), request.getKey(), request.getExpression(), event.getDetails().getBytesProcessed(), event.getDetails().getBytesScanned()});
                }

                public void visit(SelectObjectContentEvent.EndEvent event) {
                    logger.info("s3 select query completed for bucket={} key={} query={}", new Object[]{request.getBucketName(), request.getKey(), request.getExpression()});
                }
            });
        }

        public static SelectObjectContentRequest generateJsonRequest(String bucket, String key, String query) {
            SelectObjectContentRequest request = new SelectObjectContentRequest();
            request.setBucketName(bucket);
            request.setKey(key);
            request.setExpression(query);
            request.setExpressionType(ExpressionType.SQL);
            InputSerialization inputSerialization = new InputSerialization();
            inputSerialization.setJson(new JSONInput().withType(JSONType.LINES));
            inputSerialization.setCompressionType(CompressionType.NONE);
            request.setInputSerialization(inputSerialization);
            OutputSerialization outputSerialization = new OutputSerialization();
            outputSerialization.setJson(new JSONOutput());
            request.setOutputSerialization(outputSerialization);
            return request;
        }

        public static SelectObjectContentRequest generateCsvRequest(String bucket, String key, String query) {
            SelectObjectContentRequest request = new SelectObjectContentRequest();
            request.setBucketName(bucket);
            request.setKey(key);
            request.setExpression(query);
            request.setExpressionType(ExpressionType.SQL);
            InputSerialization inputSerialization = new InputSerialization();
            inputSerialization.setCsv(new CSVInput().withFileHeaderInfo(FileHeaderInfo.USE).withFieldDelimiter(","));
            inputSerialization.setCompressionType(CompressionType.NONE);
            request.setInputSerialization(inputSerialization);
            OutputSerialization outputSerialization = new OutputSerialization();
            outputSerialization.setCsv(new CSVOutput());
            request.setOutputSerialization(outputSerialization);
            return request;
        }
    }
}

