/*
 * Decompiled with CFR 0.152.
 */
package com.salesforce.cantor.s3;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.util.StringInputStream;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.salesforce.cantor.Namespaceable;
import com.salesforce.cantor.common.CommonPreconditions;
import com.salesforce.cantor.s3.S3Utils;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractBaseS3Namespaceable
implements Namespaceable {
    private static final Logger logger = LoggerFactory.getLogger(AbstractBaseS3Namespaceable.class);
    protected final AmazonS3 s3Client;
    protected final String bucketName;
    private final String namespaceLookupKey;
    private final LoadingCache<String, Optional<String>> namespaceCache;

    public AbstractBaseS3Namespaceable(AmazonS3 s3Client, String bucketName, String type) throws IOException {
        CommonPreconditions.checkArgument((s3Client != null ? 1 : 0) != 0, (String)"null s3 client");
        CommonPreconditions.checkString((String)bucketName, (String)"null/empty bucket name");
        this.s3Client = s3Client;
        this.bucketName = bucketName;
        this.namespaceLookupKey = S3Utils.getCleanKeyForNamespace(String.format("all-namespaces-%s", type));
        try {
            if (!this.s3Client.doesBucketExistV2(this.bucketName)) {
                throw new IllegalStateException("bucket does not exist: " + this.bucketName);
            }
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception creating required buckets for objects on s3:", (Throwable)e);
            throw new IOException("exception creating required buckets for objects on s3:", e);
        }
        this.namespaceCache = CacheBuilder.newBuilder().build((CacheLoader)new CacheLoader<String, Optional<String>>(){
            final Map<String, String> cachedNamespaces = new HashMap<String, String>();

            public Optional<String> load(String namespace) throws IOException {
                if (this.cachedNamespaces.containsKey(namespace)) {
                    return Optional.of(this.cachedNamespaces.get(namespace));
                }
                AbstractBaseS3Namespaceable.this.refreshNamespaces(this.cachedNamespaces);
                return Optional.ofNullable(this.cachedNamespaces.get(namespace));
            }
        });
        ScheduledExecutorService executor = Executors.newScheduledThreadPool(1);
        executor.scheduleAtFixedRate(this::refreshCache, 0L, 30L, TimeUnit.SECONDS);
    }

    public Collection<String> namespaces() throws IOException {
        try {
            return this.doGetNamespaces();
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception getting namespaces", (Throwable)e);
            throw new IOException("exception getting namespaces", e);
        }
    }

    public void create(String namespace) throws IOException {
        CommonPreconditions.checkCreate((String)namespace);
        try {
            this.doCreate(namespace);
        }
        catch (AmazonS3Exception e) {
            this.namespaceCache.invalidate((Object)namespace);
            logger.warn("exception creating namespace: " + namespace, (Throwable)e);
            throw new IOException("exception creating namespace: " + namespace, e);
        }
    }

    public void drop(String namespace) throws IOException {
        CommonPreconditions.checkDrop((String)namespace);
        try {
            this.doDrop(namespace);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception dropping namespace: " + namespace, (Throwable)e);
            throw new IOException("exception dropping namespace: " + namespace, e);
        }
    }

    protected abstract String getObjectKeyPrefix(String var1);

    protected void checkNamespace(String namespace) throws IOException {
        CommonPreconditions.checkNamespace((String)namespace);
        Optional namespaceKey = (Optional)this.namespaceCache.getUnchecked((Object)namespace);
        if (!namespaceKey.isPresent()) {
            throw new IOException(String.format("namespace '%s' does not exist", namespace));
        }
    }

    private Collection<String> doGetNamespaces() throws IOException {
        InputStream namespacesCsv = S3Utils.getObjectStream(this.s3Client, this.bucketName, this.namespaceLookupKey);
        if (namespacesCsv == null) {
            return Collections.emptyList();
        }
        try (BufferedReader namespaceReader = new BufferedReader(new InputStreamReader(namespacesCsv));){
            Collection collection = namespaceReader.lines().skip(1L).map(namespaceCsv -> namespaceCsv.split(",")[0]).collect(Collectors.toList());
            return collection;
        }
    }

    private void doCreate(String namespace) throws IOException {
        logger.info("creating namespace '{}' and adding to '{}.{}'", new Object[]{namespace, this.bucketName, this.namespaceLookupKey});
        byte[] namespacesCsv = S3Utils.getObjectBytes(this.s3Client, this.bucketName, this.namespaceLookupKey);
        this.namespaceCache.put((Object)namespace, Optional.ofNullable(S3Utils.getCleanKeyForNamespace(namespace)));
        if (namespacesCsv == null || namespacesCsv.length == 0) {
            StringInputStream csvForNamespaces = new StringInputStream("namespace,key\n" + namespace + "," + S3Utils.getCleanKeyForNamespace(namespace));
            S3Utils.putObject(this.s3Client, this.bucketName, this.namespaceLookupKey, (InputStream)csvForNamespaces, new ObjectMetadata());
            return;
        }
        String namespaces = new String(namespacesCsv);
        String newNamespace = "\n" + namespace + "," + S3Utils.getCleanKeyForNamespace(namespace);
        if (namespaces.contains(newNamespace)) {
            return;
        }
        namespaces = namespaces + newNamespace;
        ByteArrayInputStream updatedNamespaceList = new ByteArrayInputStream(namespaces.getBytes(StandardCharsets.UTF_8));
        S3Utils.putObject(this.s3Client, this.bucketName, this.namespaceLookupKey, updatedNamespaceList, new ObjectMetadata());
    }

    private void doDrop(String namespace) throws IOException {
        logger.info("dropping namespace '{}'", (Object)namespace);
        String objectKeyPrefix = this.getObjectKeyPrefix(namespace);
        logger.debug("deleting all objects with prefix '{}.{}'", (Object)this.bucketName, (Object)objectKeyPrefix);
        S3Utils.deleteObjects(this.s3Client, this.bucketName, objectKeyPrefix);
        logger.debug("deleting namespace record from namespaces object '{}.{}'", (Object)this.bucketName, (Object)this.namespaceLookupKey);
        String remainingNamespacesQuery = String.format("select * from s3object s where NOT s.namespace = '%s'", namespace);
        InputStream namespacesCsv = S3Utils.S3Select.queryObjectCsv(this.s3Client, this.bucketName, this.namespaceLookupKey, remainingNamespacesQuery);
        ByteArrayOutputStream csvForNamespaces = new ByteArrayOutputStream();
        try (BufferedReader namespaceEntry = new BufferedReader(new InputStreamReader(namespacesCsv));){
            csvForNamespaces.write("namespace,key".getBytes(StandardCharsets.UTF_8));
            Iterator entries = namespaceEntry.lines().iterator();
            while (entries.hasNext()) {
                String entry = "\n" + (String)entries.next();
                csvForNamespaces.write(entry.getBytes(StandardCharsets.UTF_8));
            }
            ByteArrayInputStream updatedNamespaces = new ByteArrayInputStream(csvForNamespaces.toByteArray());
            S3Utils.putObject(this.s3Client, this.bucketName, this.namespaceLookupKey, updatedNamespaces, new ObjectMetadata());
        }
        this.namespaceCache.invalidate((Object)namespace);
    }

    private void refreshCache() {
        for (String key : this.namespaceCache.asMap().keySet()) {
            this.namespaceCache.refresh((Object)key);
        }
    }

    private void refreshNamespaces(Map<String, String> cachedNamespaces) throws IOException {
        cachedNamespaces.clear();
        InputStream namespacesCsv = S3Utils.getObjectStream(this.s3Client, this.bucketName, this.namespaceLookupKey);
        if (namespacesCsv == null) {
            return;
        }
        try (BufferedReader namespaceReader = new BufferedReader(new InputStreamReader(namespacesCsv));){
            Iterator namespaceCsv = namespaceReader.lines().skip(1L).iterator();
            while (namespaceCsv.hasNext()) {
                String entry = (String)namespaceCsv.next();
                String[] entries = entry.split(",");
                if (entries.length != 2) {
                    throw new IOException("Invalid entry in lookup table: " + entry);
                }
                cachedNamespaces.put(entries[0], entries[1]);
            }
        }
    }

    protected static String trim(String namespace) {
        String cleanName = namespace.replaceAll("[^A-Za-z0-9_\\-]", "").toLowerCase();
        return String.format("%s-%s", cleanName.substring(0, Math.min(64, cleanName.length())), Math.abs(namespace.hashCode()));
    }
}

