/*
 * Decompiled with CFR 0.152.
 */
package com.salesforce.cantor.s3;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import ch.qos.logback.classic.encoder.PatternLayoutEncoder;
import ch.qos.logback.classic.sift.MDCBasedDiscriminator;
import ch.qos.logback.classic.sift.SiftingAppender;
import ch.qos.logback.core.Appender;
import ch.qos.logback.core.Context;
import ch.qos.logback.core.FileAppender;
import ch.qos.logback.core.encoder.Encoder;
import ch.qos.logback.core.sift.Discriminator;
import ch.qos.logback.core.util.Duration;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.transfer.MultipleFileUpload;
import com.amazonaws.services.s3.transfer.TransferManager;
import com.amazonaws.services.s3.transfer.TransferManagerBuilder;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.salesforce.cantor.Events;
import com.salesforce.cantor.common.EventsPreconditions;
import com.salesforce.cantor.s3.AbstractBaseS3Namespaceable;
import com.salesforce.cantor.s3.S3Utils;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;

public class EventsOnS3
extends AbstractBaseS3Namespaceable
implements Events {
    private static final Logger logger = LoggerFactory.getLogger(EventsOnS3.class);
    private static final String defaultBufferDirectory = "cantor-events-s3-buffer";
    private static final long defaultFlushIntervalSeconds = 60L;
    private static final String dimensionKeyPayloadOffset = ".cantor-payload-offset";
    private static final String dimensionKeyPayloadLength = ".cantor-payload-length";
    private static final String siftingDiscriminatorKey = "path";
    private static final AtomicReference<String> currentFlushCycleGuid = new AtomicReference();
    private static final Gson parser = new GsonBuilder().create();
    private static final Logger siftingLogger = EventsOnS3.initSiftingLogger();
    private static final AtomicBoolean isInitialized = new AtomicBoolean(false);
    private static final ScheduledExecutorService executor = Executors.newSingleThreadScheduledExecutor();
    private final String bufferDirectory;
    private static final String objectKeyPrefix = "cantor-events";
    private static final DateFormat cycleNameFormatter = new SimpleDateFormat("YYYY-MM-dd_HH-mm-ss");
    private static final DateFormat directoryFormatterMin = new SimpleDateFormat("YYYY/MM/dd/HH/mm");
    private static final DateFormat directoryFormatterHour = new SimpleDateFormat("YYYY/MM/dd/HH/");
    private static final DateFormat directoryFormatterDay = new SimpleDateFormat("YYYY/MM/dd/");
    private final TransferManager s3TransferManager;
    private LoadingCache<String, AtomicLong> payloadOffset = CacheBuilder.newBuilder().build((CacheLoader)new CacheLoader<String, AtomicLong>(){

        public AtomicLong load(String ignoredPath) {
            return new AtomicLong(0L);
        }
    });

    public EventsOnS3(AmazonS3 s3Client, String bucketName) throws IOException {
        this(s3Client, bucketName, defaultBufferDirectory, 60L);
    }

    public EventsOnS3(AmazonS3 s3Client, String bucketName, String bufferDirectory) throws IOException {
        this(s3Client, bucketName, bufferDirectory, 60L);
    }

    public EventsOnS3(AmazonS3 s3Client, String bucketName, String bufferDirectory, long flushIntervalSeconds) throws IOException {
        super(s3Client, bucketName, "events");
        EventsPreconditions.checkArgument((flushIntervalSeconds > 0L ? 1 : 0) != 0, (String)"invalid flush interval");
        EventsPreconditions.checkString((String)bucketName, (String)"invalid bucket name");
        EventsPreconditions.checkString((String)bufferDirectory, (String)"invalid buffer directory");
        this.bufferDirectory = bufferDirectory;
        TransferManagerBuilder builder = TransferManagerBuilder.standard();
        builder.setS3Client(this.s3Client);
        this.s3TransferManager = builder.build();
        if (!isInitialized.getAndSet(true)) {
            EventsOnS3.rollover();
            executor.scheduleAtFixedRate(this::flush, 0L, flushIntervalSeconds, TimeUnit.SECONDS);
        }
    }

    public void store(String namespace, Collection<Events.Event> batch) throws IOException {
        EventsPreconditions.checkStore((String)namespace, batch);
        this.checkNamespace(namespace);
        try {
            this.doStore(namespace, batch);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception storing events to namespace: " + namespace, (Throwable)e);
            throw new IOException("exception storing events to namespace: " + namespace, e);
        }
    }

    public List<Events.Event> get(String namespace, long startTimestampMillis, long endTimestampMillis, Map<String, String> metadataQuery, Map<String, String> dimensionsQuery, boolean includePayloads, boolean ascending, int limit) throws IOException {
        EventsPreconditions.checkGet((String)namespace, (long)startTimestampMillis, (long)endTimestampMillis, metadataQuery, dimensionsQuery);
        this.checkNamespace(namespace);
        try {
            return this.doGet(namespace, startTimestampMillis, endTimestampMillis, metadataQuery != null ? metadataQuery : Collections.emptyMap(), dimensionsQuery != null ? dimensionsQuery : Collections.emptyMap(), includePayloads, ascending, limit);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception getting events from namespace: " + namespace, (Throwable)e);
            throw new IOException("exception getting events from namespace: " + namespace, e);
        }
    }

    public Set<String> metadata(String namespace, String metadataKey, long startTimestampMillis, long endTimestampMillis, Map<String, String> metadataQuery, Map<String, String> dimensionsQuery) throws IOException {
        EventsPreconditions.checkMetadata((String)namespace, (String)metadataKey, (long)startTimestampMillis, (long)endTimestampMillis, metadataQuery, dimensionsQuery);
        this.checkNamespace(namespace);
        try {
            return this.doMetadata(namespace, metadataKey, startTimestampMillis, endTimestampMillis, metadataQuery != null ? metadataQuery : Collections.emptyMap(), dimensionsQuery != null ? dimensionsQuery : Collections.emptyMap());
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception getting metadata from namespace: " + namespace, (Throwable)e);
            throw new IOException("exception getting metadata from namespace: " + namespace, e);
        }
    }

    public void expire(String namespace, long endTimestampMillis) throws IOException {
        EventsPreconditions.checkExpire((String)namespace, (long)endTimestampMillis);
        this.checkNamespace(namespace);
        try {
            this.doExpire(namespace, endTimestampMillis);
        }
        catch (AmazonS3Exception e) {
            logger.warn("exception expiring events from namespace: " + namespace, (Throwable)e);
            throw new IOException("exception expiring events from namespace: " + namespace, e);
        }
    }

    @Override
    protected String getObjectKeyPrefix(String namespace) {
        return String.format("%s/%s", objectKeyPrefix, EventsOnS3.trim(namespace));
    }

    private static Logger initSiftingLogger() {
        LoggerContext loggerContext = (LoggerContext)LoggerFactory.getILoggerFactory();
        SiftingAppender siftingAppender = new SiftingAppender();
        String loggerName = "cantor-s3-events-sifting-logger";
        siftingAppender.setName("cantor-s3-events-sifting-logger");
        siftingAppender.setContext((Context)loggerContext);
        MDCBasedDiscriminator discriminator = new MDCBasedDiscriminator();
        discriminator.setKey(siftingDiscriminatorKey);
        discriminator.setDefaultValue("unknown");
        discriminator.start();
        siftingAppender.setDiscriminator((Discriminator)discriminator);
        siftingAppender.setTimeout(Duration.buildBySeconds((double)3.0));
        siftingAppender.setAppenderFactory((context, discriminatingValue) -> {
            FileAppender fileAppender = new FileAppender();
            fileAppender.setName("file-" + discriminatingValue);
            fileAppender.setContext(context);
            fileAppender.setFile(discriminatingValue);
            PatternLayoutEncoder patternLayoutEncoder = new PatternLayoutEncoder();
            patternLayoutEncoder.setContext(context);
            patternLayoutEncoder.setPattern("%msg%n");
            patternLayoutEncoder.start();
            fileAppender.setEncoder((Encoder)patternLayoutEncoder);
            fileAppender.start();
            return fileAppender;
        });
        siftingAppender.start();
        ch.qos.logback.classic.Logger logger = loggerContext.getLogger("cantor-s3-events-sifting-logger");
        logger.setAdditive(false);
        logger.setLevel(Level.ALL);
        logger.addAppender((Appender)siftingAppender);
        return logger;
    }

    private synchronized void doStore(String namespace, Collection<Events.Event> batch) {
        for (Events.Event event : batch) {
            HashMap metadata = new HashMap(event.getMetadata());
            HashMap<String, Double> dimensions = new HashMap<String, Double>(event.getDimensions());
            byte[] payload = event.getPayload();
            String currentCycleName = this.getRolloverCycleName();
            String cyclePath = this.getPath(currentCycleName);
            String filePath = String.format("%s/%s/%s.%s", cyclePath, this.getObjectKeyPrefix(namespace), directoryFormatterMin.format(event.getTimestampMillis()), currentCycleName);
            String payloadFilePath = filePath + ".b64";
            String eventsFilePath = filePath + ".json";
            if (payload != null && payload.length > 0) {
                String payloadBase64 = Base64.getEncoder().encodeToString(payload);
                this.append(payloadFilePath, payloadBase64);
                long offset = ((AtomicLong)this.payloadOffset.getUnchecked((Object)payloadFilePath)).getAndAdd(payloadBase64.length() + 1);
                dimensions.put(dimensionKeyPayloadOffset, Double.valueOf(offset));
                dimensions.put(dimensionKeyPayloadLength, Double.valueOf(payloadBase64.length()));
            }
            Events.Event toWrite = new Events.Event(event.getTimestampMillis(), metadata, dimensions);
            this.append(eventsFilePath, parser.toJson((Object)toWrite));
        }
    }

    private synchronized void append(String path, String message) {
        MDC.put((String)siftingDiscriminatorKey, (String)path);
        siftingLogger.info(message);
        MDC.remove((String)siftingDiscriminatorKey);
    }

    private List<Events.Event> doGet(String namespace, long startTimestampMillis, long endTimestampMillis, Map<String, String> metadataQuery, Map<String, String> dimensionsQuery, boolean includePayloads, boolean ascending, int limit) throws IOException {
        CopyOnWriteArrayList<Events.Event> results = new CopyOnWriteArrayList<Events.Event>();
        ExecutorCompletionService<List> completionService = new ExecutorCompletionService<List>(Executors.newWorkStealingPool(64));
        ArrayList<Future<List>> futures = new ArrayList<Future<List>>();
        for (String objectKey : this.getMatchingKeys(namespace, startTimestampMillis, endTimestampMillis)) {
            if (!objectKey.endsWith("json")) continue;
            futures.add(completionService.submit(() -> this.doGetOnObject(objectKey, startTimestampMillis, endTimestampMillis, metadataQuery, dimensionsQuery, includePayloads)));
        }
        for (int i = 0; i < futures.size(); ++i) {
            try {
                results.addAll((Collection)completionService.take().get(5L, TimeUnit.SECONDS));
                continue;
            }
            catch (Exception e) {
                logger.warn("exception on get call to s3", (Throwable)e);
                throw new IOException(e);
            }
        }
        this.sortEventsByTimestamp(results, ascending);
        if (limit > 0) {
            return results.subList(0, Math.min(limit, results.size()));
        }
        return results;
    }

    private Set<String> doMetadata(String namespace, String metadataKey, long startTimestampMillis, long endTimestampMillis, Map<String, String> metadataQuery, Map<String, String> dimensionsQuery) throws IOException {
        CopyOnWriteArraySet<String> results = new CopyOnWriteArraySet<String>();
        ExecutorCompletionService<Set> completionService = new ExecutorCompletionService<Set>(Executors.newWorkStealingPool(64));
        ArrayList<Future<Set>> futures = new ArrayList<Future<Set>>();
        for (String objectKey : this.getMatchingKeys(namespace, startTimestampMillis, endTimestampMillis)) {
            if (!objectKey.endsWith("json")) continue;
            futures.add(completionService.submit(() -> this.doMetadataOnObject(objectKey, metadataKey, startTimestampMillis, endTimestampMillis, metadataQuery, dimensionsQuery)));
        }
        for (int i = 0; i < futures.size(); ++i) {
            try {
                results.addAll((Collection)completionService.take().get(5L, TimeUnit.SECONDS));
                continue;
            }
            catch (Exception e) {
                logger.warn("exception on metadata call to s3", (Throwable)e);
                throw new IOException(e);
            }
        }
        return results;
    }

    private void sortEventsByTimestamp(List<Events.Event> events, boolean ascending) {
        events.sort((event1, event2) -> {
            if (event1.getTimestampMillis() < event2.getTimestampMillis()) {
                return ascending ? -1 : 1;
            }
            if (event1.getTimestampMillis() > event2.getTimestampMillis()) {
                return ascending ? 1 : -1;
            }
            return 0;
        });
    }

    private List<Events.Event> doGetOnObject(String objectKey, long startTimestampMillis, long endTimestampMillis, Map<String, String> metadataQuery, Map<String, String> dimensionsQuery, boolean includePayloads) throws IOException {
        ArrayList<Events.Event> results = new ArrayList<Events.Event>();
        String query = this.generateGetQuery(startTimestampMillis, endTimestampMillis, metadataQuery, dimensionsQuery);
        InputStream jsonLines = S3Utils.S3Select.queryObjectJson(this.s3Client, this.bucketName, objectKey, query);
        Scanner lineReader = new Scanner(jsonLines);
        while (lineReader.hasNext()) {
            Events.Event event = (Events.Event)parser.fromJson(lineReader.nextLine(), Events.Event.class);
            if (includePayloads && event.getDimensions().containsKey(dimensionKeyPayloadOffset) && event.getDimensions().containsKey(dimensionKeyPayloadLength)) {
                long offset = ((Double)event.getDimensions().get(dimensionKeyPayloadOffset)).longValue();
                long length = ((Double)event.getDimensions().get(dimensionKeyPayloadLength)).longValue();
                String payloadFilename = objectKey.replace("json", "b64");
                byte[] payloadBase64Bytes = S3Utils.getObjectBytes(this.s3Client, this.bucketName, payloadFilename, offset, offset + length - 1L);
                if (payloadBase64Bytes == null || payloadBase64Bytes.length == 0) {
                    throw new IOException("failed to retrieve payload for event");
                }
                byte[] payload = Base64.getDecoder().decode(new String(payloadBase64Bytes));
                Events.Event eventWithPayload = new Events.Event(event.getTimestampMillis(), event.getMetadata(), event.getDimensions(), payload);
                results.add(eventWithPayload);
                continue;
            }
            results.add(event);
        }
        return results;
    }

    private Set<String> doMetadataOnObject(String objectKey, String metadataKey, long startTimestampMillis, long endTimestampMillis, Map<String, String> metadataQuery, Map<String, String> dimensionsQuery) throws IOException {
        HashSet<String> results = new HashSet<String>();
        String query = this.generateMetadataQuery(metadataKey, startTimestampMillis, endTimestampMillis, metadataQuery, dimensionsQuery);
        InputStream jsonLines = S3Utils.S3Select.queryObjectJson(this.s3Client, this.bucketName, objectKey, query);
        Scanner lineReader = new Scanner(jsonLines);
        while (lineReader.hasNext()) {
            Map metadata = (Map)parser.fromJson(lineReader.nextLine(), Map.class);
            if (!metadata.containsKey(metadataKey)) continue;
            results.add((String)metadata.get(metadataKey));
        }
        return results;
    }

    private void doExpire(String namespace, long endTimestampMillis) throws IOException {
        logger.info("expiring namespace '{}' with end timestamp of '{}'", (Object)namespace, (Object)endTimestampMillis);
        Set<String> keys = this.getMatchingKeys(namespace, 0L, endTimestampMillis);
        logger.info("expiring objects: {}", keys);
        S3Utils.deleteObjects(this.s3Client, this.bucketName, keys);
    }

    private String generateGetQuery(long startTimestampMillis, long endTmestampMillis, Map<String, String> metadataQuery, Map<String, String> dimensionsQuery) {
        String timestampClause = String.format("s.timestampMillis BETWEEN %d AND %d", startTimestampMillis, endTmestampMillis);
        return String.format("SELECT * FROM s3object[*] s WHERE %s %s %s", timestampClause, this.getMetadataQuerySql(metadataQuery), this.getDimensionsQuerySql(dimensionsQuery));
    }

    private String generateMetadataQuery(String metadataKey, long startTimestampMillis, long endTmestampMillis, Map<String, String> metadataQuery, Map<String, String> dimensionsQuery) {
        String timestampClause = String.format("s.timestampMillis BETWEEN %d AND %d", startTimestampMillis, endTmestampMillis);
        return String.format("SELECT s.metadata.\"%s\" FROM s3object[*] s WHERE %s %s %s", metadataKey, timestampClause, this.getMetadataQuerySql(metadataQuery), this.getDimensionsQuerySql(dimensionsQuery));
    }

    private Set<String> getMatchingKeys(String namespace, long startTimestampMillis, long endTimestampMillis) throws IOException {
        HashSet<String> prefixes = new HashSet<String>();
        long start = startTimestampMillis;
        while (start <= endTimestampMillis) {
            if (start + TimeUnit.DAYS.toMillis(1L) <= endTimestampMillis) {
                prefixes.add(String.format("%s/%s", this.getObjectKeyPrefix(namespace), directoryFormatterDay.format(start)));
                start += TimeUnit.DAYS.toMillis(1L);
                continue;
            }
            if (start + TimeUnit.HOURS.toMillis(1L) <= endTimestampMillis) {
                prefixes.add(String.format("%s/%s", this.getObjectKeyPrefix(namespace), directoryFormatterHour.format(start)));
                start += TimeUnit.HOURS.toMillis(1L);
                continue;
            }
            prefixes.add(String.format("%s/%s", this.getObjectKeyPrefix(namespace), directoryFormatterMin.format(start)));
            start += TimeUnit.MINUTES.toMillis(1L);
        }
        prefixes.add(String.format("%s/%s", this.getObjectKeyPrefix(namespace), directoryFormatterMin.format(endTimestampMillis)));
        ConcurrentSkipListSet<String> matchingKeys = new ConcurrentSkipListSet<String>();
        ExecutorService executor = Executors.newWorkStealingPool(64);
        ExecutorCompletionService<List> completionService = new ExecutorCompletionService<List>(executor);
        ArrayList<Future<List>> futures = new ArrayList<Future<List>>();
        for (String prefix : prefixes) {
            futures.add(completionService.submit(() -> {
                matchingKeys.addAll(S3Utils.getKeys(this.s3Client, this.bucketName, prefix));
                return null;
            }));
        }
        for (int i = 0; i < futures.size(); ++i) {
            try {
                completionService.take().get(5L, TimeUnit.SECONDS);
                continue;
            }
            catch (Exception e) {
                logger.warn("exception on get call to s3", (Throwable)e);
                throw new IOException(e);
            }
        }
        return matchingKeys;
    }

    private String getMetadataQuerySql(Map<String, String> metadataQuery) {
        if (metadataQuery.isEmpty()) {
            return "";
        }
        StringBuilder sql = new StringBuilder();
        for (Map.Entry<String, String> entry : metadataQuery.entrySet()) {
            String metadataName = this.prefixMetadata(entry.getKey());
            String query = entry.getValue();
            if (query.startsWith("~")) {
                sql.append(" AND ").append(metadataName).append(" LIKE ").append(this.quote(this.regexToSql(query.substring(1))));
                continue;
            }
            if (query.startsWith("!~")) {
                sql.append(" AND ").append(metadataName).append(" NOT LIKE ").append(this.quote(this.regexToSql(query.substring(2))));
                continue;
            }
            if (query.startsWith("=")) {
                sql.append(" AND ").append(metadataName).append("=").append(this.quote(query.substring(1)));
                continue;
            }
            if (query.startsWith("!=")) {
                sql.append(" AND ").append(metadataName).append("!=").append(this.quote(query.substring(2)));
                continue;
            }
            sql.append(" AND ").append(metadataName).append("=").append(this.quote(query));
        }
        return sql.toString();
    }

    private String regexToSql(String regex) {
        return regex.replace(".*", "%").replace("_", "\\\\_");
    }

    private String getDimensionsQuerySql(Map<String, String> dimensionsQuery) {
        if (dimensionsQuery.isEmpty()) {
            return "";
        }
        StringBuilder sql = new StringBuilder();
        for (Map.Entry<String, String> entry : dimensionsQuery.entrySet()) {
            String dimensionName = this.prefixDimension(entry.getKey());
            String query = entry.getValue();
            if (query.contains("..")) {
                sql.append(" AND ").append(dimensionName).append(" BETWEEN ").append(Double.valueOf(query.substring(0, query.indexOf("..")))).append(" AND ").append(Double.valueOf(query.substring(query.indexOf("..") + 2)));
                continue;
            }
            if (query.startsWith(">=")) {
                sql.append(" AND ").append(dimensionName).append(">=").append(query.substring(2));
                continue;
            }
            if (query.startsWith("<=")) {
                sql.append(" AND ").append(dimensionName).append("<=").append(query.substring(2));
                continue;
            }
            if (query.startsWith(">")) {
                sql.append(" AND ").append(dimensionName).append(">").append(query.substring(1));
                continue;
            }
            if (query.startsWith("<")) {
                sql.append(" AND ").append(dimensionName).append("<").append(query.substring(1));
                continue;
            }
            if (query.startsWith("!=")) {
                sql.append(" AND ").append(dimensionName).append("!=").append(query.substring(2));
                continue;
            }
            if (query.startsWith("=")) {
                sql.append(" AND ").append(dimensionName).append("=").append(query.substring(1));
                continue;
            }
            sql.append(" AND ").append(dimensionName).append("=").append(query);
        }
        return sql.toString();
    }

    private String quote(String key) {
        return String.format("'%s'", key);
    }

    private String prefixMetadata(String key) {
        return String.format("s.metadata.\"%s\"", key);
    }

    private String prefixDimension(String key) {
        return String.format("CAST ( s.dimensions.\"%s\" as decimal)", key);
    }

    private static String rollover() {
        String rolloverCycleName = String.format("%s.%s", cycleNameFormatter.format(System.currentTimeMillis()), UUID.randomUUID().toString().replaceAll("-", ""));
        logger.info("starting new cycle: {}", (Object)rolloverCycleName);
        return currentFlushCycleGuid.getAndSet(rolloverCycleName);
    }

    private String getRolloverCycleName() {
        return currentFlushCycleGuid.get();
    }

    private String getPath(String rolloverCycleName) {
        return this.bufferDirectory + File.separator + rolloverCycleName;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void flush() {
        long startMillis = System.currentTimeMillis();
        try {
            EventsOnS3.rollover();
            File bufferDirectoryFile = new File(this.bufferDirectory);
            if (!(bufferDirectoryFile.exists() && bufferDirectoryFile.canWrite() && bufferDirectoryFile.isDirectory())) {
                logger.warn("buffer directory '{}' does not exist or is not writable", (Object)this.bufferDirectory);
                return;
            }
            ArrayList<File> toDelete = new ArrayList<File>();
            for (File toUpload : bufferDirectoryFile.listFiles()) {
                logger.info("uploading buffer directory: {}", (Object)toUpload.getAbsolutePath());
                if (!toUpload.exists() || !toUpload.isDirectory()) {
                    logger.info("nothing to upload");
                    continue;
                }
                this.uploadDirectory(toUpload);
                logger.info("successfully uploaded buffer directory: {}", (Object)toUpload.getAbsolutePath());
                toDelete.add(toUpload);
            }
            for (File file : toDelete) {
                logger.info("deleting buffer file: {}", (Object)file.getAbsolutePath());
                this.delete(file);
            }
        }
        catch (InterruptedException e) {
            logger.warn("flush cycle interrupted; exiting");
        }
        catch (Exception e) {
            logger.warn("exception during flush", (Throwable)e);
        }
        finally {
            long endMillis = System.currentTimeMillis();
            long elapsedSeconds = (endMillis - startMillis) / 1000L;
            logger.info("flush cycle elapsed time: {}s", (Object)elapsedSeconds);
        }
    }

    private void uploadDirectory(File toUpload) throws InterruptedException {
        MultipleFileUpload upload = this.s3TransferManager.uploadDirectory(this.bucketName, null, toUpload, true, (file, metadata) -> metadata.setContentType("text/plain"));
        do {
            logger.info("s3 transfer progress of '{}': {}% of {}mb", new Object[]{toUpload.getAbsolutePath(), (int)upload.getProgress().getPercentTransferred(), upload.getProgress().getTotalBytesToTransfer() / 0x100000L});
            Thread.sleep(1000L);
        } while (!upload.isDone());
    }

    private void delete(File dir) {
        File[] files = dir.listFiles();
        if (files != null) {
            for (File file : files) {
                this.delete(file);
            }
        }
        dir.delete();
    }
}

