/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.helios.agent;

import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.base.Throwables;
import com.google.common.collect.Iterables;
import com.spotify.docker.client.DockerClient;
import com.spotify.docker.client.DockerHost;
import com.spotify.docker.client.LogStream;
import com.spotify.docker.client.exceptions.DockerException;
import com.spotify.helios.agent.HealthChecker;
import com.spotify.helios.agent.TaskConfig;
import com.spotify.helios.common.descriptors.ExecHealthCheck;
import com.spotify.helios.common.descriptors.HealthCheck;
import com.spotify.helios.common.descriptors.HttpHealthCheck;
import com.spotify.helios.common.descriptors.TcpHealthCheck;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.MalformedURLException;
import java.net.Socket;
import java.net.URL;
import java.util.List;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class HealthCheckerFactory {
    private HealthCheckerFactory() {
    }

    public static HealthChecker create(TaskConfig taskConfig, DockerClient docker, DockerHost dockerHost, boolean agentInContainer) {
        HealthCheck healthCheck = taskConfig.healthCheck();
        if (healthCheck == null) {
            return null;
        }
        if (healthCheck instanceof ExecHealthCheck) {
            return new ExecHealthChecker((ExecHealthCheck)healthCheck, docker);
        }
        if (healthCheck instanceof HttpHealthCheck) {
            return new HttpHealthChecker((HttpHealthCheck)healthCheck, taskConfig, docker, dockerHost, agentInContainer);
        }
        if (healthCheck instanceof TcpHealthCheck) {
            return new TcpHealthChecker((TcpHealthCheck)healthCheck, taskConfig, docker, dockerHost);
        }
        throw new IllegalArgumentException("Unknown healthCheck type");
    }

    private static class TcpHealthChecker
    extends NetworkHealthchecker {
        private static final Logger log = LoggerFactory.getLogger(TcpHealthChecker.class);
        private static final int CONNECT_TIMEOUT_MILLIS = 500;
        private final TcpHealthCheck healthCheck;
        private final TaskConfig taskConfig;
        private final DockerHost dockerHost;

        private TcpHealthChecker(TcpHealthCheck healthCheck, TaskConfig taskConfig, DockerClient docker, DockerHost dockerHost) {
            super(docker);
            this.healthCheck = healthCheck;
            this.taskConfig = taskConfig;
            this.dockerHost = dockerHost;
        }

        @Override
        public boolean check(String containerId) throws InterruptedException, DockerException {
            Integer port = this.taskConfig.ports().get(this.healthCheck.getPort()).getExternalPort();
            InetSocketAddress address = new InetSocketAddress(this.dockerHost.address(), (int)port);
            if (address.getAddress().isLoopbackAddress()) {
                address = new InetSocketAddress(this.getBridgeAddress(containerId), (int)port);
            }
            log.info("about to healthcheck containerId={} with address={} for task={}", new Object[]{containerId, address, this.taskConfig});
            try (Socket s = new Socket();){
                s.connect(address, 500);
            }
            catch (Exception e) {
                return false;
            }
            return true;
        }
    }

    private static class HttpHealthChecker
    extends NetworkHealthchecker {
        private static final Logger log = LoggerFactory.getLogger(HttpHealthChecker.class);
        private static final int CONNECT_TIMEOUT_MILLIS = 500;
        private static final long READ_TIMEOUT_MILLIS = TimeUnit.SECONDS.toMillis(10L);
        private final HttpHealthCheck healthCheck;
        private final TaskConfig taskConfig;
        private final DockerHost dockerHost;
        private final boolean agentInContainer;

        private HttpHealthChecker(HttpHealthCheck healthCheck, TaskConfig taskConfig, DockerClient dockerClient, DockerHost dockerHost, boolean agentInContainer) {
            super(dockerClient);
            this.healthCheck = healthCheck;
            this.taskConfig = taskConfig;
            this.dockerHost = dockerHost;
            this.agentInContainer = agentInContainer;
        }

        @Override
        public boolean check(String containerId) throws InterruptedException, DockerException {
            URL url;
            String host;
            if (this.agentInContainer && this.dockerHost.host().startsWith("unix://")) {
                host = this.getBridgeAddress(containerId);
                log.info("Using bridge address {} for healthchecks", (Object)host);
            } else {
                host = this.dockerHost.address();
            }
            Integer port = this.taskConfig.ports().get(this.healthCheck.getPort()).getExternalPort();
            try {
                url = new URL("http", host, port, this.healthCheck.getPath());
            }
            catch (MalformedURLException e) {
                throw Throwables.propagate((Throwable)e);
            }
            log.info("about to healthcheck containerId={} with url={} for task={}", new Object[]{containerId, url, this.taskConfig});
            try {
                HttpURLConnection conn = (HttpURLConnection)url.openConnection();
                conn.setConnectTimeout(500);
                conn.setReadTimeout((int)READ_TIMEOUT_MILLIS);
                int response = conn.getResponseCode();
                log.warn("healthcheck for containerId={} with url={} returned status={}", new Object[]{containerId, url, response});
                return response >= 200 && response <= 399;
            }
            catch (Exception e) {
                log.warn("exception in healthchecking containerId={} with url={}", new Object[]{containerId, url, e});
                return false;
            }
        }
    }

    private static abstract class NetworkHealthchecker
    implements HealthChecker {
        private final DockerClient dockerClient;

        protected NetworkHealthchecker(DockerClient dockerClient) {
            this.dockerClient = dockerClient;
        }

        protected String getBridgeAddress(String containerId) throws DockerException, InterruptedException {
            return this.dockerClient.inspectContainer(containerId).networkSettings().gateway();
        }
    }

    static class ExecHealthChecker
    implements HealthChecker {
        private static final Logger log = LoggerFactory.getLogger(ExecHealthChecker.class);
        private final ExecHealthCheck healthCheck;
        private final DockerClient docker;

        ExecHealthChecker(ExecHealthCheck healthCheck, DockerClient docker) {
            this.healthCheck = healthCheck;
            this.docker = docker;
        }

        @Override
        public boolean check(String containerId) {
            if (!ExecHealthChecker.compatibleDockerVersion(this.docker)) {
                throw new UnsupportedOperationException("docker exec healthcheck is not supported on your docker version");
            }
            try {
                String output;
                List cmd = this.healthCheck.getCommand();
                String execId = this.docker.execCreate(containerId, cmd.toArray(new String[cmd.size()]), new DockerClient.ExecCreateParam[]{DockerClient.ExecCreateParam.attachStdout(), DockerClient.ExecCreateParam.attachStderr()});
                try (LogStream stream = this.docker.execStart(execId, new DockerClient.ExecStartParameter[0]);){
                    output = stream.readFully();
                }
                int exitCode = this.docker.execInspect(execId).exitCode();
                if (exitCode != 0) {
                    log.info("healthcheck failed with exit code {}. output {}", (Object)exitCode, (Object)output);
                    return false;
                }
                return true;
            }
            catch (DockerException e) {
                return false;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return false;
            }
        }

        private static boolean compatibleDockerVersion(DockerClient docker) {
            String apiVersion;
            try {
                apiVersion = docker.version().apiVersion();
            }
            catch (DockerException e) {
                return false;
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                return false;
            }
            if (Strings.isNullOrEmpty((String)apiVersion)) {
                return false;
            }
            Iterable split = Splitter.on((String)".").split((CharSequence)apiVersion);
            int major = Integer.parseInt((String)Iterables.get((Iterable)split, (int)0, (Object)"0"));
            int minor = Integer.parseInt((String)Iterables.get((Iterable)split, (int)1, (Object)"0"));
            return major == 1 && minor >= 18;
        }
    }
}

