/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.helios.agent;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.MoreExecutors;
import com.spotify.docker.client.DockerClient;
import com.spotify.docker.client.exceptions.ContainerNotFoundException;
import com.spotify.docker.client.exceptions.DockerException;
import com.spotify.docker.client.messages.ContainerInfo;
import com.spotify.helios.agent.BoundedRandomExponentialBackoff;
import com.spotify.helios.agent.RestartPolicy;
import com.spotify.helios.agent.Result;
import com.spotify.helios.agent.RetryScheduler;
import com.spotify.helios.agent.Sleeper;
import com.spotify.helios.agent.StatusUpdater;
import com.spotify.helios.agent.TaskMonitor;
import com.spotify.helios.agent.TaskRunner;
import com.spotify.helios.agent.TaskRunnerFactory;
import com.spotify.helios.agent.ThreadSleeper;
import com.spotify.helios.common.descriptors.Goal;
import com.spotify.helios.common.descriptors.Job;
import com.spotify.helios.common.descriptors.TaskStatus;
import com.spotify.helios.servicescommon.DefaultReactor;
import com.spotify.helios.servicescommon.Reactor;
import com.spotify.helios.servicescommon.statistics.MetricsContext;
import com.spotify.helios.servicescommon.statistics.SupervisorMetrics;
import java.io.InterruptedIOException;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Supervisor {
    private static final Logger log = LoggerFactory.getLogger(Supervisor.class);
    @VisibleForTesting
    static final int DEFAULT_SECONDS_TO_WAIT_BEFORE_KILL = 120;
    private final DockerClient docker;
    private final Job job;
    private final RestartPolicy restartPolicy;
    private final SupervisorMetrics metrics;
    private final Reactor reactor;
    private final Listener listener;
    private final TaskRunnerFactory runnerFactory;
    private final StatusUpdater statusUpdater;
    private final TaskMonitor monitor;
    private final Sleeper sleeper;
    private volatile Goal goal;
    private volatile String containerId;
    private volatile TaskRunner runner;
    private volatile Command currentCommand;
    private volatile Command performedCommand;

    public Supervisor(Builder builder) {
        this.job = (Job)Preconditions.checkNotNull((Object)builder.job, (Object)"job");
        this.docker = (DockerClient)Preconditions.checkNotNull((Object)builder.dockerClient, (Object)"docker");
        this.restartPolicy = (RestartPolicy)Preconditions.checkNotNull((Object)builder.restartPolicy, (Object)"restartPolicy");
        this.metrics = (SupervisorMetrics)Preconditions.checkNotNull((Object)builder.metrics, (Object)"metrics");
        this.listener = (Listener)Preconditions.checkNotNull((Object)builder.listener, (Object)"listener");
        this.currentCommand = new Nop();
        this.containerId = builder.existingContainerId;
        this.runnerFactory = (TaskRunnerFactory)Preconditions.checkNotNull((Object)builder.runnerFactory, (Object)"runnerFactory");
        this.statusUpdater = (StatusUpdater)Preconditions.checkNotNull((Object)builder.statusUpdater, (Object)"statusUpdater");
        this.monitor = (TaskMonitor)Preconditions.checkNotNull((Object)builder.monitor, (Object)"monitor");
        this.reactor = new DefaultReactor("supervisor-" + this.job.getId(), new Update(), TimeUnit.SECONDS.toMillis(30L));
        this.reactor.startAsync();
        this.statusUpdater.setContainerId(this.containerId);
        this.sleeper = builder.sleeper;
    }

    public void setGoal(Goal goal) {
        if (this.goal == goal) {
            return;
        }
        log.debug("Supervisor {}: setting goal: {}", (Object)this.job.getId(), (Object)goal);
        this.goal = goal;
        this.statusUpdater.setGoal(goal);
        switch (goal) {
            case START: {
                this.currentCommand = new Start();
                this.reactor.signal();
                this.metrics.supervisorStarted();
                break;
            }
            case STOP: 
            case UNDEPLOY: {
                this.currentCommand = new Stop();
                this.reactor.signal();
                this.metrics.supervisorStopped();
            }
        }
    }

    public void close() {
        this.reactor.stopAsync();
        if (this.runner != null) {
            this.runner.stopAsync();
        }
        this.metrics.supervisorClosed();
        this.monitor.close();
    }

    public void join() {
        this.reactor.awaitTerminated();
        if (this.runner != null) {
            this.runner.stopAsync();
            this.runner.awaitTerminated();
        }
    }

    public boolean isStarting() {
        return this.currentCommand instanceof Start;
    }

    public boolean isStopping() {
        return this.currentCommand instanceof Stop;
    }

    public boolean isDone() {
        return this.currentCommand == this.performedCommand;
    }

    public String containerId() {
        return this.containerId;
    }

    private void fireStateChanged() {
        log.debug("Supervisor {}: state changed", (Object)this.job.getId());
        try {
            this.listener.stateChanged(this);
        }
        catch (Exception e) {
            log.error("Listener threw exception", (Throwable)e);
        }
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public String toString() {
        return "Supervisor{job=" + this.job + ", currentCommand=" + this.currentCommand + ", performedCommand=" + this.performedCommand + '}';
    }

    private class TaskListener
    extends TaskRunner.NopListener {
        private MetricsContext pullContext;

        private TaskListener() {
        }

        @Override
        public void failed(Throwable t, String containerError) {
            Supervisor.this.metrics.containersThrewException();
        }

        @Override
        public void pulling() {
            this.pullContext = Supervisor.this.metrics.containerPull();
        }

        @Override
        public void pullFailed() {
            if (this.pullContext != null) {
                this.pullContext.failure();
            }
        }

        @Override
        public void pulled() {
            if (this.pullContext != null) {
                this.pullContext.success();
            }
        }

        @Override
        public void created(String createdContainerId) {
            Supervisor.this.containerId = createdContainerId;
        }
    }

    private static class Nop
    implements Command {
        private Nop() {
        }

        @Override
        public void perform(boolean done) {
        }
    }

    private class Stop
    implements Command {
        private Stop() {
        }

        @Override
        public void perform(boolean done) throws InterruptedException {
            Integer gracePeriod;
            if (done) {
                return;
            }
            Supervisor.this.statusUpdater.setState(TaskStatus.State.STOPPING);
            Supervisor.this.statusUpdater.update();
            if (Supervisor.this.runner != null && (gracePeriod = Supervisor.this.job.getGracePeriod()) != null && gracePeriod > 0) {
                log.info("Unregistering from service discovery for {} seconds before stopping", (Object)gracePeriod);
                if (Supervisor.this.runner.unregister()) {
                    log.info("Unregistered. Now sleeping for {} seconds.", (Object)gracePeriod);
                    Supervisor.this.sleeper.sleep(TimeUnit.MILLISECONDS.convert(gracePeriod.intValue(), TimeUnit.SECONDS));
                }
            }
            log.info("stopping job: {}", (Object)Supervisor.this.job);
            if (Supervisor.this.runner != null) {
                Supervisor.this.runner.stop();
                Supervisor.this.runner = null;
            }
            RetryScheduler retryScheduler = BoundedRandomExponentialBackoff.newBuilder().setMinIntervalMillis(TimeUnit.SECONDS.toMillis(1L)).setMaxIntervalMillis(TimeUnit.SECONDS.toMillis(30L)).build().newScheduler();
            while (this.containerRunning()) {
                this.killContainer();
                Supervisor.this.sleeper.sleep(retryScheduler.nextMillis());
            }
            Supervisor.this.statusUpdater.setState(TaskStatus.State.STOPPED);
            Supervisor.this.statusUpdater.setContainerError(this.containerError());
            Supervisor.this.statusUpdater.update();
        }

        private void killContainer() throws InterruptedException {
            if (Supervisor.this.containerId == null) {
                return;
            }
            try {
                Supervisor.this.docker.killContainer(Supervisor.this.containerId);
            }
            catch (DockerException e) {
                log.error("failed to kill container {}", (Object)Supervisor.this.containerId, (Object)e);
            }
        }

        private boolean containerRunning() throws InterruptedException {
            ContainerInfo containerInfo;
            if (Supervisor.this.containerId == null) {
                return false;
            }
            try {
                containerInfo = Supervisor.this.docker.inspectContainer(Supervisor.this.containerId);
            }
            catch (ContainerNotFoundException e) {
                return false;
            }
            catch (DockerException e) {
                log.error("failed to query container {}", (Object)Supervisor.this.containerId, (Object)e);
                return true;
            }
            return containerInfo.state().running();
        }

        private String containerError() throws InterruptedException {
            ContainerInfo containerInfo;
            if (Supervisor.this.containerId == null) {
                return null;
            }
            try {
                containerInfo = Supervisor.this.docker.inspectContainer(Supervisor.this.containerId);
            }
            catch (ContainerNotFoundException e) {
                return null;
            }
            catch (DockerException e) {
                log.error("failed to query container {}", (Object)Supervisor.this.containerId, (Object)e);
                return null;
            }
            return containerInfo.state().error();
        }
    }

    private class Start
    implements Command {
        private Start() {
        }

        @Override
        public void perform(boolean done) throws InterruptedException {
            if (Supervisor.this.runner == null) {
                this.startAfter(0L);
                return;
            }
            if (Supervisor.this.runner.isRunning()) {
                return;
            }
            Result<Integer> result = Supervisor.this.runner.result();
            if (!result.isSuccess()) {
                Throwable t = result.getException();
                if (t instanceof InterruptedException || t instanceof InterruptedIOException) {
                    log.debug("task runner interrupted");
                    Supervisor.this.runner = null;
                    Supervisor.this.reactor.signal();
                    return;
                }
                if (t instanceof DockerException) {
                    log.error("docker error", t);
                } else {
                    log.error("task runner threw exception", t);
                }
            }
            this.startAfter(Supervisor.this.restartPolicy.delay(Supervisor.this.monitor.throttle()));
        }

        private void startAfter(long delay) {
            log.debug("starting job (delay={}): {}", (Object)delay, (Object)Supervisor.this.job);
            int waitBeforeKill = Optional.ofNullable(Supervisor.this.job.getSecondsToWaitBeforeKill()).orElse(120);
            Supervisor.this.runner = Supervisor.this.runnerFactory.create(delay, Supervisor.this.containerId, new TaskListener(), waitBeforeKill);
            Supervisor.this.runner.startAsync();
            Supervisor.this.runner.resultFuture().addListener(Supervisor.this.reactor.signalRunnable(), MoreExecutors.directExecutor());
        }
    }

    private static interface Command {
        public void perform(boolean var1) throws InterruptedException;
    }

    public static class Builder {
        private Job job;
        private String existingContainerId;
        private DockerClient dockerClient;
        private RestartPolicy restartPolicy;
        private SupervisorMetrics metrics;
        private Listener listener = new NopListener();
        private TaskRunnerFactory runnerFactory;
        private StatusUpdater statusUpdater;
        private TaskMonitor monitor;
        private Sleeper sleeper = new ThreadSleeper();

        private Builder() {
        }

        public Builder setJob(Job job) {
            this.job = job;
            return this;
        }

        public Builder setExistingContainerId(String existingContainerId) {
            this.existingContainerId = existingContainerId;
            return this;
        }

        public Builder setRestartPolicy(RestartPolicy restartPolicy) {
            this.restartPolicy = restartPolicy;
            return this;
        }

        public Builder setDockerClient(DockerClient dockerClient) {
            this.dockerClient = dockerClient;
            return this;
        }

        public Builder setMetrics(SupervisorMetrics metrics) {
            this.metrics = metrics;
            return this;
        }

        public Builder setListener(Listener listener) {
            this.listener = listener;
            return this;
        }

        public Builder setRunnerFactory(TaskRunnerFactory runnerFactory) {
            this.runnerFactory = runnerFactory;
            return this;
        }

        public Builder setStatusUpdater(StatusUpdater statusUpdater) {
            this.statusUpdater = statusUpdater;
            return this;
        }

        public Builder setMonitor(TaskMonitor monitor) {
            this.monitor = monitor;
            return this;
        }

        public Builder setSleeper(Sleeper sleeper) {
            this.sleeper = sleeper;
            return this;
        }

        public Supervisor build() {
            return new Supervisor(this);
        }

        private class NopListener
        implements Listener {
            private NopListener() {
            }

            @Override
            public void stateChanged(Supervisor supervisor) {
            }
        }
    }

    private class Update
    implements Reactor.Callback {
        private Update() {
        }

        @Override
        public void run(boolean timeout) throws InterruptedException {
            Command command = Supervisor.this.currentCommand;
            boolean done = Supervisor.this.performedCommand == command;
            log.debug("Supervisor {}: update: performedCommand={}, command={}, done={}", new Object[]{Supervisor.this.job.getId(), Supervisor.this.performedCommand, command, done});
            command.perform(done);
            if (!done) {
                Supervisor.this.performedCommand = command;
                Supervisor.this.fireStateChanged();
            }
        }
    }

    public static interface Listener {
        public void stateChanged(Supervisor var1);
    }
}

