/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.styx.util;

import com.google.common.base.Preconditions;
import com.spotify.styx.model.Workflow;
import com.spotify.styx.model.WorkflowConfiguration;
import com.spotify.styx.util.DockerImageValidator;
import com.spotify.styx.util.TimeUtil;
import java.time.Duration;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.List;

public class WorkflowValidator {
    static final int MAX_ID_LENGTH = 256;
    static final int MAX_DOCKER_ARGS_TOTAL = 1000000;
    static final int MAX_RESOURCES = 5;
    static final int MAX_RESOURCE_LENGTH = 256;
    static final int MAX_COMMIT_SHA_LENGTH = 256;
    static final int MAX_SECRET_NAME_LENGTH = 253;
    static final int MAX_SECRET_MOUNT_PATH_LENGTH = 1024;
    static final int MAX_SERVICE_ACCOUNT_LENGTH = 256;
    static final int MAX_ENV_VARS = 128;
    static final int MAX_ENV_SIZE = 16384;
    static final Duration MIN_RUNNING_TIMEOUT = Duration.ofMinutes(1L);
    private final DockerImageValidator dockerImageValidator;
    private final Duration maybeMaxRunningTimeout;

    private WorkflowValidator(DockerImageValidator dockerImageValidator, Duration maybeMaxRunningTimeout) {
        Preconditions.checkArgument(maybeMaxRunningTimeout == null || !maybeMaxRunningTimeout.isNegative(), "Max Running timeout should be positive");
        this.dockerImageValidator = dockerImageValidator;
        this.maybeMaxRunningTimeout = maybeMaxRunningTimeout;
    }

    public static Builder newBuilder(DockerImageValidator dockerImageValidator) {
        return new Builder(dockerImageValidator);
    }

    public List<String> validateWorkflow(Workflow workflow) {
        WorkflowConfiguration configuration = workflow.configuration();
        return this.validateWorkflowConfiguration(configuration);
    }

    public List<String> validateWorkflowConfiguration(WorkflowConfiguration cfg) {
        ArrayList<String> e = new ArrayList<String>();
        this.upperLimit(e, cfg.id().length(), 256, "id too long");
        this.upperLimit(e, cfg.commitSha().map(String::length).orElse(0), 256, "commitSha too long");
        this.upperLimit(e, cfg.secret().map(s2 -> s2.name().length()).orElse(0), 253, "secret name too long");
        this.upperLimit(e, cfg.secret().map(s2 -> s2.mountPath().length()).orElse(0), 1024, "secret mount path too long");
        this.upperLimit(e, cfg.serviceAccount().map(String::length).orElse(0), 256, "service account too long");
        this.upperLimit(e, cfg.resources().size(), 5, "too many resources");
        this.upperLimit(e, cfg.env().size(), 128, "too many env vars");
        this.upperLimit(e, cfg.env().entrySet().stream().mapToInt(entry -> ((String)entry.getKey()).length() + ((String)entry.getValue()).length()).sum(), 16384, "env too big");
        cfg.dockerImage().ifPresent(image -> this.dockerImageValidator.validateImageReference((String)image).stream().map(s2 -> "invalid image: " + s2).forEach(e::add));
        cfg.resources().stream().map(String::length).forEach(v -> this.upperLimit((List<String>)e, (Comparable)v, 256, "resource name too long"));
        cfg.dockerArgs().ifPresent(args -> {
            int dockerArgs = args.size() + args.stream().mapToInt(String::length).sum();
            this.upperLimit(e, dockerArgs, 1000000, "docker args is too large");
        });
        cfg.offset().ifPresent(offset -> {
            try {
                TimeUtil.addOffset(ZonedDateTime.now(), offset);
            }
            catch (DateTimeParseException ex) {
                e.add(String.format("invalid offset: %s", ex.getMessage()));
            }
        });
        try {
            TimeUtil.cron(cfg.schedule());
        }
        catch (IllegalArgumentException ex) {
            e.add("invalid schedule");
        }
        cfg.runningTimeout().ifPresent(timeout -> {
            this.lowerLimit((List<String>)e, (Comparable)timeout, (Comparable)MIN_RUNNING_TIMEOUT, "running timeout is too small");
            if (this.maybeMaxRunningTimeout != null) {
                this.upperLimit((List<String>)e, (Comparable)timeout, (Comparable)this.maybeMaxRunningTimeout, "running timeout is too big");
            }
        });
        return e;
    }

    private <T extends Comparable<T>> void lowerLimit(List<String> errors, T value, T limit, String message) {
        this.limit(errors, value.compareTo(limit) < 0, value, limit, message);
    }

    private <T extends Comparable<T>> void upperLimit(List<String> errors, T value, T limit, String message) {
        this.limit(errors, value.compareTo(limit) > 0, value, limit, message);
    }

    private <T extends Comparable<T>> void limit(List<String> errors, boolean isError, T value, T limit, String message) {
        if (isError) {
            errors.add(message + ": " + value + ", limit = " + limit);
        }
    }

    public static class Builder {
        private final DockerImageValidator dockerImageValidator;
        private Duration maxRunningTimeout;

        public Builder(DockerImageValidator dockerImageValidator) {
            this.dockerImageValidator = dockerImageValidator;
        }

        public Builder withMaxRunningTimeoutLimit(Duration maxRunningTimeout) {
            this.maxRunningTimeout = maxRunningTimeout;
            return this;
        }

        public WorkflowValidator build() {
            return new WorkflowValidator(this.dockerImageValidator, this.maxRunningTimeout);
        }
    }
}

