/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.styx.storage;

import com.google.cloud.datastore.Datastore;
import com.google.cloud.datastore.DatastoreException;
import com.google.cloud.datastore.DatastoreReader;
import com.google.cloud.datastore.DateTime;
import com.google.cloud.datastore.Entity;
import com.google.cloud.datastore.EntityQuery;
import com.google.cloud.datastore.FullEntity;
import com.google.cloud.datastore.Key;
import com.google.cloud.datastore.KeyFactory;
import com.google.cloud.datastore.PathElement;
import com.google.cloud.datastore.Query;
import com.google.cloud.datastore.QueryResults;
import com.google.cloud.datastore.StructuredQuery;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.spotify.styx.model.Backfill;
import com.spotify.styx.model.Partitioning;
import com.spotify.styx.model.Resource;
import com.spotify.styx.model.Workflow;
import com.spotify.styx.model.WorkflowId;
import com.spotify.styx.model.WorkflowInstance;
import com.spotify.styx.model.WorkflowState;
import com.spotify.styx.serialization.Json;
import com.spotify.styx.util.FnWithException;
import com.spotify.styx.util.ResourceNotFoundException;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DatastoreStorage {
    private static final Logger LOG = LoggerFactory.getLogger(DatastoreStorage.class);
    public static final String KIND_STYX_CONFIG = "StyxConfig";
    public static final String KIND_COMPONENT = "Component";
    public static final String KIND_WORKFLOW = "Workflow";
    public static final String KIND_ACTIVE_WORKFLOW_INSTANCE = "ActiveWorkflowInstance";
    public static final String KIND_RESOURCE = "Resource";
    public static final String KIND_BACKFILL = "Backfill";
    public static final String PROPERTY_CONFIG_ENABLED = "enabled";
    public static final String PROPERTY_CONFIG_DOCKER_RUNNER_ID = "dockerRunnerId";
    public static final String PROPERTY_WORKFLOW_JSON = "json";
    public static final String PROPERTY_WORKFLOW_ENABLED = "enabled";
    public static final String PROPERTY_NEXT_NATURAL_TRIGGER = "nextNaturalTrigger";
    public static final String PROPERTY_DOCKER_IMAGE = "dockerImage";
    public static final String PROPERTY_COUNTER = "counter";
    public static final String PROPERTY_COMPONENT = "component";
    public static final String PROPERTY_WORKFLOW = "workflow";
    public static final String PROPERTY_PARAMETER = "parameter";
    public static final String PROPERTY_COMMIT_SHA = "commitSha";
    public static final String PROPERTY_CONCURRENCY = "concurrency";
    public static final String PROPERTY_START = "start";
    public static final String PROPERTY_END = "end";
    public static final String PROPERTY_RESOURCE = "resource";
    public static final String PROPERTY_NEXT_TRIGGER = "nextTrigger";
    public static final String PROPERTY_PARTITIONING = "partitioning";
    public static final String PROPERTY_COMPLETED = "completed";
    public static final String PROPERTY_HALTED = "halted";
    public static final String PROPERTY_DEBUG_ENABLED = "debug";
    public static final String KEY_GLOBAL_CONFIG = "styxGlobal";
    public static final boolean DEFAULT_CONFIG_ENABLED = true;
    public static final String DEFAULT_CONFIG_DOCKER_RUNNER_ID = "default";
    public static final boolean DEFAULT_WORKFLOW_ENABLED = false;
    public static final boolean DEFAULT_DEBUG_ENABLED = false;
    public static final int MAX_RETRIES = 100;
    private final Datastore datastore;
    private final Duration retryBaseDelay;
    private final KeyFactory componentKeyFactory;
    @VisibleForTesting
    final Key globalConfigKey;

    DatastoreStorage(Datastore datastore, Duration retryBaseDelay) {
        this.datastore = Objects.requireNonNull(datastore);
        this.retryBaseDelay = Objects.requireNonNull(retryBaseDelay);
        this.componentKeyFactory = (KeyFactory)datastore.newKeyFactory().kind(KIND_COMPONENT);
        this.globalConfigKey = ((KeyFactory)datastore.newKeyFactory().kind(KIND_STYX_CONFIG)).newKey(KEY_GLOBAL_CONFIG);
    }

    private String readConfigString(String property, String defaultValue) {
        return this.getOpt((DatastoreReader)this.datastore, this.globalConfigKey).filter(w -> w.contains(property)).map(config -> config.getString(property)).orElse(defaultValue);
    }

    private boolean readConfigBoolean(String property, boolean defaultValue) {
        return this.getOpt((DatastoreReader)this.datastore, this.globalConfigKey).filter(w -> w.contains(property)).map(config -> config.getBoolean(property)).orElse(defaultValue);
    }

    boolean globalEnabled() throws IOException {
        return this.readConfigBoolean("enabled", true);
    }

    boolean debugEnabled() throws IOException {
        return this.readConfigBoolean(PROPERTY_DEBUG_ENABLED, false);
    }

    public String globalDockerRunnerId() {
        return this.readConfigString(PROPERTY_CONFIG_DOCKER_RUNNER_ID, DEFAULT_CONFIG_DOCKER_RUNNER_ID);
    }

    boolean setGlobalEnabled(boolean globalEnabled) throws IOException {
        return this.storeWithRetries(() -> (Boolean)this.datastore.runInTransaction(transaction -> {
            Optional<Entity> configOpt = this.getOpt((DatastoreReader)transaction, this.globalConfigKey);
            Entity.Builder config = this.asBuilderOrNew(configOpt, this.globalConfigKey);
            boolean oldValue = configOpt.filter(w -> w.contains("enabled")).map(c -> c.getBoolean("enabled")).orElse(true);
            config.set("enabled", globalEnabled);
            transaction.put((FullEntity)config.build());
            return oldValue;
        }));
    }

    boolean enabled(WorkflowId workflowId) throws IOException {
        Key workflowKey = this.workflowKey(workflowId);
        return this.getOpt((DatastoreReader)this.datastore, workflowKey).filter(w -> w.contains("enabled")).map(workflow -> workflow.getBoolean("enabled")).orElse(false);
    }

    Set<WorkflowId> enabled() throws IOException {
        EntityQuery queryWorkflows = ((EntityQuery.Builder)EntityQuery.entityQueryBuilder().kind(KIND_WORKFLOW)).build();
        QueryResults result = this.datastore.run((Query)queryWorkflows);
        HashSet enabledWorkflows = Sets.newHashSet();
        while (result.hasNext()) {
            Entity workflow = (Entity)result.next();
            boolean enabled = workflow.contains("enabled") && workflow.getBoolean("enabled");
            if (!enabled) continue;
            enabledWorkflows.add(this.parseWorkflowId(workflow));
        }
        return enabledWorkflows;
    }

    void store(Workflow workflow) throws IOException {
        this.storeWithRetries(() -> (Entity)this.datastore.runInTransaction(transaction -> {
            String json = Json.OBJECT_MAPPER.writeValueAsString((Object)workflow);
            Key componentKey = this.componentKeyFactory.newKey(workflow.componentId());
            Entity retrievedComponent = transaction.get(componentKey);
            if (retrievedComponent == null) {
                transaction.put((FullEntity)Entity.builder((Key)componentKey).build());
            }
            Key workflowKey = this.workflowKey(workflow.id());
            Optional<Entity> workflowOpt = this.getOpt((DatastoreReader)transaction, workflowKey);
            Entity workflowEntity = ((Entity.Builder)this.asBuilderOrNew(workflowOpt, workflowKey).set(PROPERTY_WORKFLOW_JSON, json)).build();
            return transaction.put((FullEntity)workflowEntity);
        }));
    }

    Optional<Workflow> workflow(WorkflowId workflowId) throws IOException {
        Key workflowKey = this.workflowKey(workflowId);
        return this.getOpt((DatastoreReader)this.datastore, workflowKey).filter(e -> e.contains(PROPERTY_WORKFLOW_JSON)).map(e -> {
            try {
                return (Workflow)Json.OBJECT_MAPPER.readValue(e.getString(PROPERTY_WORKFLOW_JSON), Workflow.class);
            }
            catch (IOException e1) {
                LOG.info("Failed to read workflow for {}, {}", (Object)workflowId.componentId(), (Object)workflowId.endpointId());
                return null;
            }
        });
    }

    void delete(WorkflowId workflowId) throws IOException {
        this.storeWithRetries(() -> {
            this.datastore.delete(new Key[]{this.workflowKey(workflowId)});
            return null;
        });
    }

    public void updateNextNaturalTrigger(WorkflowId workflowId, Instant nextNaturalTrigger) throws IOException {
        this.storeWithRetries(() -> (Entity)this.datastore.runInTransaction(transaction -> {
            Key workflowKey = this.workflowKey(workflowId);
            Optional<Entity> workflowOpt = this.getOpt((DatastoreReader)transaction, workflowKey);
            if (!workflowOpt.isPresent()) {
                throw new ResourceNotFoundException(String.format("%s:%s doesn't exist.", workflowId.componentId(), workflowId.endpointId()));
            }
            Entity.Builder builder = (Entity.Builder)Entity.builder((Entity)workflowOpt.get()).set(PROPERTY_NEXT_NATURAL_TRIGGER, DatastoreStorage.instantToDatetime(nextNaturalTrigger));
            return transaction.put((FullEntity)builder.build());
        }));
    }

    public Map<Workflow, Optional<Instant>> workflowsWithNextNaturalTrigger() throws IOException {
        HashMap map = Maps.newHashMap();
        EntityQuery query = ((EntityQuery.Builder)Query.entityQueryBuilder().kind(KIND_WORKFLOW)).build();
        QueryResults result = this.datastore.run((Query)query);
        while (result.hasNext()) {
            Workflow workflow;
            Entity entity = (Entity)result.next();
            try {
                workflow = (Workflow)Json.OBJECT_MAPPER.readValue(entity.getString(PROPERTY_WORKFLOW_JSON), Workflow.class);
            }
            catch (IOException e) {
                LOG.warn("Failed to read workflow {}.", (Object)entity.key());
                continue;
            }
            map.put(workflow, entity.contains(PROPERTY_NEXT_NATURAL_TRIGGER) ? Optional.of(DatastoreStorage.datetimeToInstant(entity.getDateTime(PROPERTY_NEXT_NATURAL_TRIGGER))) : Optional.empty());
        }
        return map;
    }

    Map<WorkflowInstance, Long> allActiveStates() throws IOException {
        EntityQuery query = ((EntityQuery.Builder)Query.entityQueryBuilder().kind(KIND_ACTIVE_WORKFLOW_INSTANCE)).build();
        return this.queryActiveStates(query);
    }

    Map<WorkflowInstance, Long> activeStates(String componentId) throws IOException {
        EntityQuery query = ((EntityQuery.Builder)((EntityQuery.Builder)Query.entityQueryBuilder().kind(KIND_ACTIVE_WORKFLOW_INSTANCE)).filter((StructuredQuery.Filter)StructuredQuery.PropertyFilter.eq((String)PROPERTY_COMPONENT, (String)componentId))).build();
        return this.queryActiveStates(query);
    }

    private Map<WorkflowInstance, Long> queryActiveStates(EntityQuery activeStatesQuery) throws IOException {
        ImmutableMap.Builder mapBuilder = ImmutableMap.builder();
        QueryResults results = this.datastore.run((Query)activeStatesQuery);
        while (results.hasNext()) {
            Entity entity = (Entity)results.next();
            long counter = entity.getLong(PROPERTY_COUNTER);
            WorkflowInstance instance = this.parseWorkflowInstance(entity);
            mapBuilder.put((Object)instance, (Object)counter);
        }
        return mapBuilder.build();
    }

    void writeActiveState(WorkflowInstance workflowInstance, long counter) throws IOException {
        this.storeWithRetries(() -> {
            Key key = this.activeWorkflowInstanceKey(workflowInstance);
            Entity entity = ((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)Entity.builder((Key)key).set(PROPERTY_COMPONENT, workflowInstance.workflowId().componentId())).set(PROPERTY_WORKFLOW, workflowInstance.workflowId().endpointId())).set(PROPERTY_PARAMETER, workflowInstance.parameter())).set(PROPERTY_COUNTER, counter)).build();
            return this.datastore.put((FullEntity)entity);
        });
    }

    void deleteActiveState(WorkflowInstance workflowInstance) throws IOException {
        this.storeWithRetries(() -> {
            this.datastore.delete(new Key[]{this.activeWorkflowInstanceKey(workflowInstance)});
            return null;
        });
    }

    void patchState(WorkflowId workflowId, WorkflowState state) throws IOException {
        this.storeWithRetries(() -> (Entity)this.datastore.runInTransaction(transaction -> {
            Key workflowKey = this.workflowKey(workflowId);
            Optional<Entity> workflowOpt = this.getOpt((DatastoreReader)transaction, workflowKey);
            if (!workflowOpt.isPresent()) {
                throw new ResourceNotFoundException(String.format("%s:%s doesn't exist.", workflowId.componentId(), workflowId.endpointId()));
            }
            Entity.Builder builder = Entity.builder((Entity)workflowOpt.get());
            state.enabled().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set("enabled", x.booleanValue());
            });
            state.dockerImage().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_DOCKER_IMAGE, x);
            });
            state.commitSha().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_COMMIT_SHA, x);
            });
            state.nextNaturalTrigger().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_NEXT_NATURAL_TRIGGER, DatastoreStorage.instantToDatetime(x));
            });
            return transaction.put((FullEntity)builder.build());
        }));
    }

    void patchState(String componentId, WorkflowState state) throws IOException {
        this.storeWithRetries(() -> (Entity)this.datastore.runInTransaction(transaction -> {
            Key componentKey = this.componentKeyFactory.newKey(componentId);
            Optional<Entity> componentOpt = this.getOpt((DatastoreReader)transaction, componentKey);
            if (!componentOpt.isPresent()) {
                throw new ResourceNotFoundException(String.format("%s doesn't exist.", componentId));
            }
            Entity.Builder builder = Entity.builder((Entity)componentOpt.get());
            state.dockerImage().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_DOCKER_IMAGE, x);
            });
            state.commitSha().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_COMMIT_SHA, x);
            });
            return transaction.put((FullEntity)builder.build());
        }));
    }

    Optional<String> getDockerImage(WorkflowId workflowId) throws IOException {
        Key workflowKey = this.workflowKey(workflowId);
        Optional<String> dockerImage = this.getOptStringProperty((DatastoreReader)this.datastore, workflowKey, PROPERTY_DOCKER_IMAGE);
        if (dockerImage.isPresent()) {
            return dockerImage;
        }
        Key componentKey = this.componentKeyFactory.newKey(workflowId.componentId());
        dockerImage = this.getOptStringProperty((DatastoreReader)this.datastore, componentKey, PROPERTY_DOCKER_IMAGE);
        if (dockerImage.isPresent()) {
            return dockerImage;
        }
        return this.workflow(workflowId).flatMap(wf -> wf.schedule().dockerImage());
    }

    public WorkflowState workflowState(WorkflowId workflowId) throws IOException {
        WorkflowState.Builder builder = WorkflowState.builder().enabled(this.enabled(workflowId));
        this.getDockerImage(workflowId).ifPresent(builder::dockerImage);
        this.getCommitSha(workflowId).ifPresent(builder::commitSha);
        return builder.build();
    }

    private Optional<String> getCommitSha(WorkflowId workflowId) {
        Key workflowKey = this.workflowKey(workflowId);
        Optional<String> commitSha = this.getOptStringProperty((DatastoreReader)this.datastore, workflowKey, PROPERTY_COMMIT_SHA);
        if (commitSha.isPresent()) {
            return commitSha;
        }
        Key componentKey = this.componentKeyFactory.newKey(workflowId.componentId());
        return this.getOptStringProperty((DatastoreReader)this.datastore, componentKey, PROPERTY_COMMIT_SHA);
    }

    private <T> T storeWithRetries(FnWithException<T, IOException> storingOperation) throws IOException {
        int storeRetries = 0;
        while (storeRetries < 100) {
            try {
                return storingOperation.apply();
            }
            catch (ResourceNotFoundException e) {
                throw e;
            }
            catch (DatastoreException | IOException e) {
                if (e.getCause() instanceof ResourceNotFoundException) {
                    throw (ResourceNotFoundException)e.getCause();
                }
                if (++storeRetries == 100) {
                    throw e;
                }
                LOG.warn(String.format("Failed to read/write from/to Datastore (attempt #%d)", storeRetries), e);
                try {
                    Thread.sleep(this.retryBaseDelay.toMillis());
                }
                catch (InterruptedException e1) {
                    throw Throwables.propagate((Throwable)e1);
                }
            }
        }
        throw new IOException("This should never happen");
    }

    private Key workflowKey(WorkflowId workflowId) {
        return ((KeyFactory)((KeyFactory)this.datastore.newKeyFactory().ancestors(PathElement.of((String)KIND_COMPONENT, (String)workflowId.componentId()))).kind(KIND_WORKFLOW)).newKey(workflowId.endpointId());
    }

    private Key activeWorkflowInstanceKey(WorkflowInstance workflowInstance) {
        return ((KeyFactory)this.datastore.newKeyFactory().kind(KIND_ACTIVE_WORKFLOW_INSTANCE)).newKey(workflowInstance.toKey());
    }

    private WorkflowInstance parseWorkflowInstance(Entity activeWorkflowInstance) {
        String componentId = activeWorkflowInstance.getString(PROPERTY_COMPONENT);
        String workflowId = activeWorkflowInstance.getString(PROPERTY_WORKFLOW);
        String parameter = activeWorkflowInstance.getString(PROPERTY_PARAMETER);
        return WorkflowInstance.create(WorkflowId.create(componentId, workflowId), parameter);
    }

    private WorkflowId parseWorkflowId(Entity workflow) {
        String componentId = ((PathElement)((Key)workflow.key()).ancestors().get(0)).name();
        String endpointId = ((Key)workflow.key()).name();
        return WorkflowId.create(componentId, endpointId);
    }

    private Optional<Entity> getOpt(DatastoreReader datastoreReader, Key key) {
        return Optional.ofNullable(datastoreReader.get(key));
    }

    private Optional<String> getOptStringProperty(DatastoreReader datastoreReader, Key key, String property) {
        return this.getOpt(datastoreReader, key).filter(e -> e.contains(property)).map(e -> e.getString(property));
    }

    private Entity.Builder asBuilderOrNew(Optional<Entity> entityOpt, Key key) {
        return entityOpt.map(c -> Entity.builder((Entity)c)).orElse(Entity.builder((Key)key));
    }

    void setEnabled(WorkflowId workflowId1, boolean enabled) throws IOException {
        this.patchState(workflowId1, WorkflowState.patchEnabled(enabled));
    }

    private static DateTime instantToDatetime(Instant instant) {
        return DateTime.copyFrom((Date)Date.from(instant));
    }

    private static Instant datetimeToInstant(DateTime dateTime) {
        return dateTime.toDate().toInstant();
    }

    Optional<Resource> getResource(String id) {
        Entity entity = this.datastore.get(((KeyFactory)this.datastore.newKeyFactory().kind(KIND_RESOURCE)).newKey(id));
        if (entity == null) {
            return Optional.empty();
        }
        return Optional.of(this.entityToResource(entity));
    }

    void postResource(Resource resource) throws IOException {
        this.storeWithRetries(() -> this.datastore.put((FullEntity)this.resourceToEntity(resource)));
    }

    List<Resource> getResources() {
        EntityQuery query = ((EntityQuery.Builder)Query.entityQueryBuilder().kind(KIND_RESOURCE)).build();
        QueryResults results = this.datastore.run((Query)query);
        ImmutableList.Builder resources = ImmutableList.builder();
        while (results.hasNext()) {
            resources.add((Object)this.entityToResource((Entity)results.next()));
        }
        return resources.build();
    }

    private Resource entityToResource(Entity entity) {
        return Resource.create(((Key)entity.key()).name(), entity.getLong(PROPERTY_CONCURRENCY));
    }

    private Entity resourceToEntity(Resource resource) {
        Key key = ((KeyFactory)this.datastore.newKeyFactory().kind(KIND_RESOURCE)).newKey(resource.id());
        return ((Entity.Builder)Entity.builder((Key)key).set(PROPERTY_CONCURRENCY, resource.concurrency())).build();
    }

    void deleteResource(String id) throws IOException {
        Key key = ((KeyFactory)this.datastore.newKeyFactory().kind(KIND_RESOURCE)).newKey(id);
        this.storeWithRetries(() -> {
            this.datastore.delete(new Key[]{key});
            return null;
        });
    }

    Optional<Backfill> getBackfill(String id) {
        Entity entity = this.datastore.get(((KeyFactory)this.datastore.newKeyFactory().kind(KIND_BACKFILL)).newKey(id));
        if (entity == null) {
            return Optional.empty();
        }
        return Optional.of(this.entityToBackfill(entity));
    }

    List<Backfill> getBackfills() {
        EntityQuery query = ((EntityQuery.Builder)Query.entityQueryBuilder().kind(KIND_BACKFILL)).build();
        QueryResults results = this.datastore.run((Query)query);
        ImmutableList.Builder resources = ImmutableList.builder();
        while (results.hasNext()) {
            resources.add((Object)this.entityToBackfill((Entity)results.next()));
        }
        return resources.build();
    }

    private Backfill entityToBackfill(Entity entity) {
        WorkflowId workflowId = WorkflowId.create(entity.getString(PROPERTY_COMPONENT), entity.getString(PROPERTY_WORKFLOW));
        return Backfill.newBuilder().id(((Key)entity.key()).name()).start(DatastoreStorage.datetimeToInstant(entity.getDateTime(PROPERTY_START))).end(DatastoreStorage.datetimeToInstant(entity.getDateTime(PROPERTY_END))).workflowId(workflowId).concurrency((int)entity.getLong(PROPERTY_CONCURRENCY)).resource(entity.getString(PROPERTY_RESOURCE)).nextTrigger(DatastoreStorage.datetimeToInstant(entity.getDateTime(PROPERTY_NEXT_TRIGGER))).partitioning(Partitioning.valueOf(entity.getString(PROPERTY_PARTITIONING))).completed(entity.getBoolean(PROPERTY_COMPLETED)).halted(entity.getBoolean(PROPERTY_HALTED)).build();
    }

    void storeBackfill(Backfill backfill) throws IOException {
        this.storeWithRetries(() -> this.datastore.put((FullEntity)this.backfillToEntity(backfill)));
    }

    private Entity backfillToEntity(Backfill backfill) {
        Key key = ((KeyFactory)this.datastore.newKeyFactory().kind(KIND_BACKFILL)).newKey(backfill.id());
        Entity.Builder builder = (Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)Entity.builder((Key)key).set(PROPERTY_CONCURRENCY, (long)backfill.concurrency())).set(PROPERTY_START, DatastoreStorage.instantToDatetime(backfill.start()))).set(PROPERTY_END, DatastoreStorage.instantToDatetime(backfill.end()))).set(PROPERTY_COMPONENT, backfill.workflowId().componentId())).set(PROPERTY_WORKFLOW, backfill.workflowId().endpointId())).set(PROPERTY_RESOURCE, backfill.resource())).set(PROPERTY_PARTITIONING, backfill.partitioning().name())).set(PROPERTY_NEXT_TRIGGER, DatastoreStorage.instantToDatetime(backfill.nextTrigger()))).set(PROPERTY_COMPLETED, backfill.completed())).set(PROPERTY_HALTED, backfill.halted());
        return builder.build();
    }
}

