/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.styx.storage;

import com.google.cloud.Timestamp;
import com.google.cloud.datastore.Datastore;
import com.google.cloud.datastore.DatastoreException;
import com.google.cloud.datastore.DatastoreReader;
import com.google.cloud.datastore.Entity;
import com.google.cloud.datastore.EntityQuery;
import com.google.cloud.datastore.FullEntity;
import com.google.cloud.datastore.Key;
import com.google.cloud.datastore.KeyFactory;
import com.google.cloud.datastore.PathElement;
import com.google.cloud.datastore.Query;
import com.google.cloud.datastore.QueryResults;
import com.google.cloud.datastore.StructuredQuery;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.spotify.styx.model.Backfill;
import com.spotify.styx.model.Resource;
import com.spotify.styx.model.Schedule;
import com.spotify.styx.model.Workflow;
import com.spotify.styx.model.WorkflowId;
import com.spotify.styx.model.WorkflowInstance;
import com.spotify.styx.model.WorkflowState;
import com.spotify.styx.serialization.Json;
import com.spotify.styx.util.FnWithException;
import com.spotify.styx.util.ResourceNotFoundException;
import com.spotify.styx.util.TimeUtil;
import com.spotify.styx.util.TriggerInstantSpec;
import java.io.IOException;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class DatastoreStorage {
    private static final Logger LOG = LoggerFactory.getLogger(DatastoreStorage.class);
    public static final String KIND_STYX_CONFIG = "StyxConfig";
    public static final String KIND_COMPONENT = "Component";
    public static final String KIND_WORKFLOW = "Workflow";
    public static final String KIND_ACTIVE_WORKFLOW_INSTANCE = "ActiveWorkflowInstance";
    public static final String KIND_RESOURCE = "Resource";
    public static final String KIND_BACKFILL = "Backfill";
    public static final String PROPERTY_CONFIG_ENABLED = "enabled";
    public static final String PROPERTY_CONFIG_DOCKER_RUNNER_ID = "dockerRunnerId";
    public static final String PROPERTY_CONFIG_CONCURRENCY = "concurrency";
    public static final String PROPERTY_CONFIG_CLIENT_BLACKLIST = "clientBlacklist";
    public static final String PROPERTY_WORKFLOW_JSON = "json";
    public static final String PROPERTY_WORKFLOW_ENABLED = "enabled";
    public static final String PROPERTY_NEXT_NATURAL_TRIGGER = "nextNaturalTrigger";
    public static final String PROPERTY_NEXT_NATURAL_OFFSET_TRIGGER = "nextNaturalOffsetTrigger";
    public static final String PROPERTY_DOCKER_IMAGE = "dockerImage";
    public static final String PROPERTY_COUNTER = "counter";
    public static final String PROPERTY_COMPONENT = "component";
    public static final String PROPERTY_WORKFLOW = "workflow";
    public static final String PROPERTY_PARAMETER = "parameter";
    public static final String PROPERTY_COMMIT_SHA = "commitSha";
    public static final String PROPERTY_CONCURRENCY = "concurrency";
    public static final String PROPERTY_START = "start";
    public static final String PROPERTY_END = "end";
    public static final String PROPERTY_NEXT_TRIGGER = "nextTrigger";
    public static final String PROPERTY_SCHEDULE = "schedule";
    public static final String PROPERTY_ALL_TRIGGERED = "allTriggered";
    public static final String PROPERTY_HALTED = "halted";
    public static final String PROPERTY_DEBUG_ENABLED = "debug";
    public static final String PROPERTY_SUBMISSION_RATE_LIMIT = "submissionRateLimit";
    public static final String KEY_GLOBAL_CONFIG = "styxGlobal";
    public static final boolean DEFAULT_CONFIG_ENABLED = true;
    public static final String DEFAULT_CONFIG_DOCKER_RUNNER_ID = "default";
    public static final boolean DEFAULT_WORKFLOW_ENABLED = false;
    public static final boolean DEFAULT_DEBUG_ENABLED = false;
    public static final int MAX_RETRIES = 100;
    private final Datastore datastore;
    private final Duration retryBaseDelay;
    private final KeyFactory componentKeyFactory;
    @VisibleForTesting
    final Key globalConfigKey;

    DatastoreStorage(Datastore datastore, Duration retryBaseDelay) {
        this.datastore = Objects.requireNonNull(datastore);
        this.retryBaseDelay = Objects.requireNonNull(retryBaseDelay);
        this.componentKeyFactory = (KeyFactory)datastore.newKeyFactory().setKind(KIND_COMPONENT);
        this.globalConfigKey = ((KeyFactory)datastore.newKeyFactory().setKind(KIND_STYX_CONFIG)).newKey(KEY_GLOBAL_CONFIG);
    }

    private String readConfigString(String property, String defaultValue) {
        return this.getOpt((DatastoreReader)this.datastore, this.globalConfigKey).filter(w -> w.contains(property)).map(config -> config.getString(property)).orElse(defaultValue);
    }

    private boolean readConfigBoolean(String property, boolean defaultValue) {
        return this.getOpt((DatastoreReader)this.datastore, this.globalConfigKey).filter(w -> w.contains(property)).map(config -> config.getBoolean(property)).orElse(defaultValue);
    }

    boolean globalEnabled() throws IOException {
        return this.readConfigBoolean("enabled", true);
    }

    boolean debugEnabled() throws IOException {
        return this.readConfigBoolean(PROPERTY_DEBUG_ENABLED, false);
    }

    public String globalDockerRunnerId() {
        return this.readConfigString(PROPERTY_CONFIG_DOCKER_RUNNER_ID, DEFAULT_CONFIG_DOCKER_RUNNER_ID);
    }

    boolean setGlobalEnabled(boolean globalEnabled) throws IOException {
        return this.storeWithRetries(() -> (Boolean)this.datastore.runInTransaction(transaction -> {
            Optional<Entity> configOpt = this.getOpt((DatastoreReader)transaction, this.globalConfigKey);
            Entity.Builder config = this.asBuilderOrNew(configOpt, this.globalConfigKey);
            boolean oldValue = configOpt.filter(w -> w.contains("enabled")).map(c -> c.getBoolean("enabled")).orElse(true);
            config.set("enabled", globalEnabled);
            transaction.put((FullEntity)config.build());
            return oldValue;
        }));
    }

    boolean enabled(WorkflowId workflowId) throws IOException {
        Key workflowKey = this.workflowKey(workflowId);
        return this.getOpt((DatastoreReader)this.datastore, workflowKey).filter(w -> w.contains("enabled")).map(workflow -> workflow.getBoolean("enabled")).orElse(false);
    }

    Set<WorkflowId> enabled() throws IOException {
        EntityQuery queryWorkflows = ((EntityQuery.Builder)EntityQuery.newEntityQueryBuilder().setKind(KIND_WORKFLOW)).build();
        QueryResults result = this.datastore.run((Query)queryWorkflows);
        HashSet enabledWorkflows = Sets.newHashSet();
        while (result.hasNext()) {
            Entity workflow = (Entity)result.next();
            boolean enabled = workflow.contains("enabled") && workflow.getBoolean("enabled");
            if (!enabled) continue;
            enabledWorkflows.add(this.parseWorkflowId(workflow));
        }
        return enabledWorkflows;
    }

    void store(Workflow workflow) throws IOException {
        this.storeWithRetries(() -> (Entity)this.datastore.runInTransaction(transaction -> {
            String json = Json.OBJECT_MAPPER.writeValueAsString((Object)workflow);
            Key componentKey = this.componentKeyFactory.newKey(workflow.componentId());
            Entity retrievedComponent = transaction.get(componentKey);
            if (retrievedComponent == null) {
                transaction.put((FullEntity)Entity.newBuilder((Key)componentKey).build());
            }
            Key workflowKey = this.workflowKey(workflow.id());
            Optional<Entity> workflowOpt = this.getOpt((DatastoreReader)transaction, workflowKey);
            Entity workflowEntity = ((Entity.Builder)this.asBuilderOrNew(workflowOpt, workflowKey).set(PROPERTY_WORKFLOW_JSON, json)).build();
            return transaction.put((FullEntity)workflowEntity);
        }));
    }

    Optional<Workflow> workflow(WorkflowId workflowId) throws IOException {
        Key workflowKey = this.workflowKey(workflowId);
        return this.getOpt((DatastoreReader)this.datastore, workflowKey).filter(e -> e.contains(PROPERTY_WORKFLOW_JSON)).map(e -> this.parseWorkflowJson((Entity)e, workflowId));
    }

    void delete(WorkflowId workflowId) throws IOException {
        this.storeWithRetries(() -> {
            this.datastore.delete(new Key[]{this.workflowKey(workflowId)});
            return null;
        });
    }

    public void updateNextNaturalTrigger(WorkflowId workflowId, TriggerInstantSpec triggerSpec) throws IOException {
        this.storeWithRetries(() -> (Entity)this.datastore.runInTransaction(transaction -> {
            Key workflowKey = this.workflowKey(workflowId);
            Optional<Entity> workflowOpt = this.getOpt((DatastoreReader)transaction, workflowKey);
            if (!workflowOpt.isPresent()) {
                throw new ResourceNotFoundException(String.format("%s:%s doesn't exist.", workflowId.componentId(), workflowId.id()));
            }
            Entity.Builder builder = (Entity.Builder)((Entity.Builder)Entity.newBuilder((Entity)workflowOpt.get()).set(PROPERTY_NEXT_NATURAL_TRIGGER, DatastoreStorage.instantToTimestamp(triggerSpec.instant()))).set(PROPERTY_NEXT_NATURAL_OFFSET_TRIGGER, DatastoreStorage.instantToTimestamp(triggerSpec.offsetInstant()));
            return transaction.put((FullEntity)builder.build());
        }));
    }

    @Deprecated
    @VisibleForTesting
    public void updateNextNaturalTrigger(WorkflowId workflowId, Instant instant) throws IOException {
        this.storeWithRetries(() -> (Entity)this.datastore.runInTransaction(transaction -> {
            Key workflowKey = this.workflowKey(workflowId);
            Optional<Entity> workflowOpt = this.getOpt((DatastoreReader)transaction, workflowKey);
            if (!workflowOpt.isPresent()) {
                throw new ResourceNotFoundException(String.format("%s:%s doesn't exist.", workflowId.componentId(), workflowId.id()));
            }
            Entity.Builder builder = (Entity.Builder)Entity.newBuilder((Entity)workflowOpt.get()).set(PROPERTY_NEXT_NATURAL_TRIGGER, DatastoreStorage.instantToTimestamp(instant));
            return transaction.put((FullEntity)builder.build());
        }));
    }

    public Map<Workflow, TriggerInstantSpec> workflowsWithNextNaturalTrigger() throws IOException {
        HashMap map = Maps.newHashMap();
        EntityQuery query = ((EntityQuery.Builder)Query.newEntityQueryBuilder().setKind(KIND_WORKFLOW)).build();
        QueryResults result = this.datastore.run((Query)query);
        while (result.hasNext()) {
            Instant triggerInstant;
            Workflow workflow;
            Entity entity = (Entity)result.next();
            try {
                workflow = (Workflow)Json.OBJECT_MAPPER.readValue(entity.getString(PROPERTY_WORKFLOW_JSON), Workflow.class);
            }
            catch (IOException e) {
                LOG.warn("Failed to read workflow {}.", (Object)entity.getKey());
                continue;
            }
            if (!entity.contains(PROPERTY_NEXT_NATURAL_TRIGGER)) continue;
            Instant instant = DatastoreStorage.timestampToInstant(entity.getTimestamp(PROPERTY_NEXT_NATURAL_TRIGGER));
            if (!entity.contains(PROPERTY_NEXT_NATURAL_OFFSET_TRIGGER)) {
                Schedule schedule = workflow.configuration().schedule();
                if (TimeUtil.isAligned(instant, schedule)) {
                    instant = TimeUtil.previousInstant(instant, schedule);
                }
                triggerInstant = workflow.configuration().addOffset(instant);
            } else {
                triggerInstant = DatastoreStorage.timestampToInstant(entity.getTimestamp(PROPERTY_NEXT_NATURAL_OFFSET_TRIGGER));
            }
            map.put(workflow, TriggerInstantSpec.create(instant, triggerInstant));
        }
        return map;
    }

    public Map<WorkflowId, Workflow> workflows() {
        HashMap map = Maps.newHashMap();
        EntityQuery query = ((EntityQuery.Builder)Query.newEntityQueryBuilder().setKind(KIND_WORKFLOW)).build();
        QueryResults result = this.datastore.run((Query)query);
        while (result.hasNext()) {
            Workflow workflow;
            Entity entity = (Entity)result.next();
            try {
                workflow = (Workflow)Json.OBJECT_MAPPER.readValue(entity.getString(PROPERTY_WORKFLOW_JSON), Workflow.class);
            }
            catch (IOException e) {
                LOG.warn("Failed to read workflow {}.", (Object)entity.getKey());
                continue;
            }
            map.put(workflow.id(), workflow);
        }
        return map;
    }

    Map<WorkflowInstance, Long> allActiveStates() throws IOException {
        EntityQuery query = ((EntityQuery.Builder)Query.newEntityQueryBuilder().setKind(KIND_ACTIVE_WORKFLOW_INSTANCE)).build();
        return this.queryActiveStates(query);
    }

    Map<WorkflowInstance, Long> activeStates(String componentId) throws IOException {
        EntityQuery query = ((EntityQuery.Builder)((EntityQuery.Builder)Query.newEntityQueryBuilder().setKind(KIND_ACTIVE_WORKFLOW_INSTANCE)).setFilter((StructuredQuery.Filter)StructuredQuery.PropertyFilter.eq((String)PROPERTY_COMPONENT, (String)componentId))).build();
        return this.queryActiveStates(query);
    }

    private Map<WorkflowInstance, Long> queryActiveStates(EntityQuery activeStatesQuery) throws IOException {
        ImmutableMap.Builder mapBuilder = ImmutableMap.builder();
        QueryResults results = this.datastore.run((Query)activeStatesQuery);
        while (results.hasNext()) {
            Entity entity = (Entity)results.next();
            long counter = entity.getLong(PROPERTY_COUNTER);
            WorkflowInstance instance = this.parseWorkflowInstance(entity);
            mapBuilder.put((Object)instance, (Object)counter);
        }
        return mapBuilder.build();
    }

    void writeActiveState(WorkflowInstance workflowInstance, long counter) throws IOException {
        this.storeWithRetries(() -> {
            Key key = this.activeWorkflowInstanceKey(workflowInstance);
            Entity entity = ((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)Entity.newBuilder((Key)key).set(PROPERTY_COMPONENT, workflowInstance.workflowId().componentId())).set(PROPERTY_WORKFLOW, workflowInstance.workflowId().id())).set(PROPERTY_PARAMETER, workflowInstance.parameter())).set(PROPERTY_COUNTER, counter)).build();
            return this.datastore.put((FullEntity)entity);
        });
    }

    void deleteActiveState(WorkflowInstance workflowInstance) throws IOException {
        this.storeWithRetries(() -> {
            this.datastore.delete(new Key[]{this.activeWorkflowInstanceKey(workflowInstance)});
            return null;
        });
    }

    void patchState(WorkflowId workflowId, WorkflowState state) throws IOException {
        this.storeWithRetries(() -> (Entity)this.datastore.runInTransaction(transaction -> {
            Key workflowKey = this.workflowKey(workflowId);
            Optional<Entity> workflowOpt = this.getOpt((DatastoreReader)transaction, workflowKey);
            if (!workflowOpt.isPresent()) {
                throw new ResourceNotFoundException(String.format("%s:%s doesn't exist.", workflowId.componentId(), workflowId.id()));
            }
            Entity.Builder builder = Entity.newBuilder((Entity)workflowOpt.get());
            state.enabled().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set("enabled", x.booleanValue());
            });
            state.dockerImage().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_DOCKER_IMAGE, x);
            });
            state.commitSha().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_COMMIT_SHA, x);
            });
            state.nextNaturalTrigger().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_NEXT_NATURAL_TRIGGER, DatastoreStorage.instantToTimestamp(x));
            });
            state.nextNaturalOffsetTrigger().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_NEXT_NATURAL_OFFSET_TRIGGER, DatastoreStorage.instantToTimestamp(x));
            });
            return transaction.put((FullEntity)builder.build());
        }));
    }

    void patchState(String componentId, WorkflowState state) throws IOException {
        this.storeWithRetries(() -> (Entity)this.datastore.runInTransaction(transaction -> {
            Key componentKey = this.componentKeyFactory.newKey(componentId);
            Optional<Entity> componentOpt = this.getOpt((DatastoreReader)transaction, componentKey);
            if (!componentOpt.isPresent()) {
                throw new ResourceNotFoundException(String.format("%s doesn't exist.", componentId));
            }
            Entity.Builder builder = Entity.newBuilder((Entity)componentOpt.get());
            state.dockerImage().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_DOCKER_IMAGE, x);
            });
            state.commitSha().ifPresent(x -> {
                Entity.Builder cfr_ignored_0 = (Entity.Builder)builder.set(PROPERTY_COMMIT_SHA, x);
            });
            return transaction.put((FullEntity)builder.build());
        }));
    }

    Optional<String> getDockerImage(WorkflowId workflowId) throws IOException {
        Optional<Entity> workflowEntity = this.getOpt((DatastoreReader)this.datastore, this.workflowKey(workflowId));
        Optional<String> dockerImage = this.getOptStringProperty(workflowEntity, PROPERTY_DOCKER_IMAGE);
        if (dockerImage.isPresent()) {
            return dockerImage;
        }
        dockerImage = this.getOptStringProperty((DatastoreReader)this.datastore, this.componentKeyFactory.newKey(workflowId.componentId()), PROPERTY_DOCKER_IMAGE);
        if (dockerImage.isPresent()) {
            return dockerImage;
        }
        return workflowEntity.map(w -> this.parseWorkflowJson((Entity)w, workflowId)).flatMap(wf -> wf.configuration().dockerImage());
    }

    public WorkflowState workflowState(WorkflowId workflowId) throws IOException {
        WorkflowState.Builder builder = WorkflowState.builder();
        Optional<Entity> workflowEntity = this.getOpt((DatastoreReader)this.datastore, this.workflowKey(workflowId));
        builder.enabled(workflowEntity.filter(w -> w.contains("enabled")).map(workflow -> workflow.getBoolean("enabled")).orElse(false));
        this.getOptInstantProperty(workflowEntity, PROPERTY_NEXT_NATURAL_TRIGGER).ifPresent(builder::nextNaturalTrigger);
        this.getOptInstantProperty(workflowEntity, PROPERTY_NEXT_NATURAL_OFFSET_TRIGGER).ifPresent(builder::nextNaturalOffsetTrigger);
        Optional<String> dockerImage = this.getOptStringProperty(workflowEntity, PROPERTY_DOCKER_IMAGE);
        Optional<String> commitSha = this.getOptStringProperty(workflowEntity, PROPERTY_COMMIT_SHA);
        if (!dockerImage.isPresent() || !commitSha.isPresent()) {
            Optional<Entity> componentEntity = this.getOpt((DatastoreReader)this.datastore, this.componentKeyFactory.newKey(workflowId.componentId()));
            if (!dockerImage.isPresent() && !(dockerImage = this.getOptStringProperty(componentEntity, PROPERTY_DOCKER_IMAGE)).isPresent()) {
                dockerImage = workflowEntity.map(w -> this.parseWorkflowJson((Entity)w, workflowId)).flatMap(wf -> wf.configuration().dockerImage());
            }
            if (!commitSha.isPresent()) {
                commitSha = this.getOptStringProperty(componentEntity, PROPERTY_COMMIT_SHA);
            }
        }
        dockerImage.ifPresent(builder::dockerImage);
        commitSha.ifPresent(builder::commitSha);
        return builder.build();
    }

    private <T> T storeWithRetries(FnWithException<T, IOException> storingOperation) throws IOException {
        int storeRetries = 0;
        while (storeRetries < 100) {
            try {
                return storingOperation.apply();
            }
            catch (ResourceNotFoundException e) {
                throw e;
            }
            catch (DatastoreException | IOException e) {
                if (e.getCause() instanceof ResourceNotFoundException) {
                    throw (ResourceNotFoundException)e.getCause();
                }
                if (++storeRetries == 100) {
                    throw e;
                }
                LOG.warn(String.format("Failed to read/write from/to Datastore (attempt #%d)", storeRetries), e);
                try {
                    Thread.sleep(this.retryBaseDelay.toMillis());
                }
                catch (InterruptedException e1) {
                    throw Throwables.propagate((Throwable)e1);
                }
            }
        }
        throw new IOException("This should never happen");
    }

    private Key workflowKey(WorkflowId workflowId) {
        return ((KeyFactory)((KeyFactory)this.datastore.newKeyFactory().addAncestor(PathElement.of((String)KIND_COMPONENT, (String)workflowId.componentId()))).setKind(KIND_WORKFLOW)).newKey(workflowId.id());
    }

    private Key activeWorkflowInstanceKey(WorkflowInstance workflowInstance) {
        return ((KeyFactory)this.datastore.newKeyFactory().setKind(KIND_ACTIVE_WORKFLOW_INSTANCE)).newKey(workflowInstance.toKey());
    }

    private WorkflowInstance parseWorkflowInstance(Entity activeWorkflowInstance) {
        String componentId = activeWorkflowInstance.getString(PROPERTY_COMPONENT);
        String workflowId = activeWorkflowInstance.getString(PROPERTY_WORKFLOW);
        String parameter = activeWorkflowInstance.getString(PROPERTY_PARAMETER);
        return WorkflowInstance.create(WorkflowId.create(componentId, workflowId), parameter);
    }

    private WorkflowId parseWorkflowId(Entity workflow) {
        String componentId = ((PathElement)((Key)workflow.getKey()).getAncestors().get(0)).getName();
        String id = ((Key)workflow.getKey()).getName();
        return WorkflowId.create(componentId, id);
    }

    private Workflow parseWorkflowJson(Entity entity, WorkflowId workflowId) {
        try {
            return (Workflow)Json.OBJECT_MAPPER.readValue(entity.getString(PROPERTY_WORKFLOW_JSON), Workflow.class);
        }
        catch (IOException e1) {
            LOG.info("Failed to read workflow for {}, {}", (Object)workflowId.componentId(), (Object)workflowId.id());
            return null;
        }
    }

    private Optional<Entity> getOpt(DatastoreReader datastoreReader, Key key) {
        return Optional.ofNullable(datastoreReader.get(key));
    }

    private Optional<String> getOptStringProperty(DatastoreReader datastoreReader, Key key, String property) {
        return this.getOpt(datastoreReader, key).filter(e -> e.contains(property)).map(e -> e.getString(property));
    }

    private Optional<String> getOptStringProperty(Optional<Entity> entity, String property) {
        return entity.filter(e -> e.contains(property)).map(e -> e.getString(property));
    }

    private Optional<Instant> getOptInstantProperty(Optional<Entity> entity, String property) {
        return entity.filter(w -> w.contains(property)).map(workflow -> DatastoreStorage.timestampToInstant(workflow.getTimestamp(property)));
    }

    private Entity.Builder asBuilderOrNew(Optional<Entity> entityOpt, Key key) {
        return entityOpt.map(c -> Entity.newBuilder((Entity)c)).orElse(Entity.newBuilder((Key)key));
    }

    void setEnabled(WorkflowId workflowId1, boolean enabled) throws IOException {
        this.patchState(workflowId1, WorkflowState.patchEnabled(enabled));
    }

    private static Timestamp instantToTimestamp(Instant instant) {
        return Timestamp.of((Date)Date.from(instant));
    }

    private static Instant timestampToInstant(Timestamp ts) {
        return Instant.ofEpochSecond(ts.getSeconds(), ts.getNanos());
    }

    Optional<Resource> getResource(String id) {
        Entity entity = this.datastore.get(((KeyFactory)this.datastore.newKeyFactory().setKind(KIND_RESOURCE)).newKey(id));
        if (entity == null) {
            return Optional.empty();
        }
        return Optional.of(this.entityToResource(entity));
    }

    void postResource(Resource resource) throws IOException {
        this.storeWithRetries(() -> this.datastore.put((FullEntity)this.resourceToEntity(resource)));
    }

    List<Resource> getResources() {
        EntityQuery query = ((EntityQuery.Builder)Query.newEntityQueryBuilder().setKind(KIND_RESOURCE)).build();
        QueryResults results = this.datastore.run((Query)query);
        ImmutableList.Builder resources = ImmutableList.builder();
        while (results.hasNext()) {
            resources.add((Object)this.entityToResource((Entity)results.next()));
        }
        return resources.build();
    }

    private Resource entityToResource(Entity entity) {
        return Resource.create(((Key)entity.getKey()).getName(), entity.getLong("concurrency"));
    }

    private Entity resourceToEntity(Resource resource) {
        Key key = ((KeyFactory)this.datastore.newKeyFactory().setKind(KIND_RESOURCE)).newKey(resource.id());
        return ((Entity.Builder)Entity.newBuilder((Key)key).set("concurrency", resource.concurrency())).build();
    }

    void deleteResource(String id) throws IOException {
        Key key = ((KeyFactory)this.datastore.newKeyFactory().setKind(KIND_RESOURCE)).newKey(id);
        this.storeWithRetries(() -> {
            this.datastore.delete(new Key[]{key});
            return null;
        });
    }

    Optional<Backfill> getBackfill(String id) {
        Entity entity = this.datastore.get(((KeyFactory)this.datastore.newKeyFactory().setKind(KIND_BACKFILL)).newKey(id));
        if (entity == null) {
            return Optional.empty();
        }
        return Optional.of(this.entityToBackfill(entity));
    }

    private EntityQuery.Builder backfillQueryBuilder(boolean showAll, StructuredQuery.Filter ... filters) {
        EntityQuery.Builder queryBuilder = (EntityQuery.Builder)Query.newEntityQueryBuilder().setKind(KIND_BACKFILL);
        ArrayList andedFilters = Lists.newArrayList((Object[])filters);
        if (!showAll) {
            andedFilters.add(StructuredQuery.PropertyFilter.eq((String)PROPERTY_ALL_TRIGGERED, (boolean)false));
            andedFilters.add(StructuredQuery.PropertyFilter.eq((String)PROPERTY_HALTED, (boolean)false));
        }
        if (!andedFilters.isEmpty()) {
            StructuredQuery.Filter head = (StructuredQuery.Filter)andedFilters.get(0);
            StructuredQuery.Filter[] tail = (StructuredQuery.Filter[])andedFilters.stream().skip(1L).toArray(StructuredQuery.Filter[]::new);
            queryBuilder.setFilter((StructuredQuery.Filter)StructuredQuery.CompositeFilter.and((StructuredQuery.Filter)head, (StructuredQuery.Filter[])tail));
        }
        return queryBuilder;
    }

    private List<Backfill> backfillsForQuery(Query<Entity> query) {
        QueryResults results = this.datastore.run(query);
        ImmutableList.Builder resources = ImmutableList.builder();
        results.forEachRemaining(entity -> resources.add((Object)this.entityToBackfill((Entity)entity)));
        return resources.build();
    }

    List<Backfill> getBackfills(boolean showAll) {
        return this.backfillsForQuery((Query<Entity>)this.backfillQueryBuilder(showAll, new StructuredQuery.Filter[0]).build());
    }

    List<Backfill> getBackfillsForComponent(boolean showAll, String component) {
        EntityQuery query = this.backfillQueryBuilder(showAll, new StructuredQuery.Filter[]{StructuredQuery.PropertyFilter.eq((String)PROPERTY_COMPONENT, (String)component)}).build();
        return this.backfillsForQuery((Query<Entity>)query);
    }

    List<Backfill> getBackfillsForWorkflow(boolean showAll, String workflow) {
        EntityQuery query = this.backfillQueryBuilder(showAll, new StructuredQuery.Filter[]{StructuredQuery.PropertyFilter.eq((String)PROPERTY_WORKFLOW, (String)workflow)}).build();
        return this.backfillsForQuery((Query<Entity>)query);
    }

    List<Backfill> getBackfillsForWorkflowId(boolean showAll, WorkflowId workflowId) {
        EntityQuery query = this.backfillQueryBuilder(showAll, new StructuredQuery.Filter[]{StructuredQuery.PropertyFilter.eq((String)PROPERTY_COMPONENT, (String)workflowId.componentId()), StructuredQuery.PropertyFilter.eq((String)PROPERTY_WORKFLOW, (String)workflowId.id())}).build();
        return this.backfillsForQuery((Query<Entity>)query);
    }

    private Backfill entityToBackfill(Entity entity) {
        WorkflowId workflowId = WorkflowId.create(entity.getString(PROPERTY_COMPONENT), entity.getString(PROPERTY_WORKFLOW));
        return Backfill.newBuilder().id(((Key)entity.getKey()).getName()).start(DatastoreStorage.timestampToInstant(entity.getTimestamp(PROPERTY_START))).end(DatastoreStorage.timestampToInstant(entity.getTimestamp(PROPERTY_END))).workflowId(workflowId).concurrency((int)entity.getLong("concurrency")).nextTrigger(DatastoreStorage.timestampToInstant(entity.getTimestamp(PROPERTY_NEXT_TRIGGER))).schedule(Schedule.parse(entity.getString(PROPERTY_SCHEDULE))).allTriggered(entity.getBoolean(PROPERTY_ALL_TRIGGERED)).halted(entity.getBoolean(PROPERTY_HALTED)).build();
    }

    void storeBackfill(Backfill backfill) throws IOException {
        this.storeWithRetries(() -> this.datastore.put((FullEntity)this.backfillToEntity(backfill)));
    }

    private Entity backfillToEntity(Backfill backfill) {
        Key key = ((KeyFactory)this.datastore.newKeyFactory().setKind(KIND_BACKFILL)).newKey(backfill.id());
        Entity.Builder builder = (Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)((Entity.Builder)Entity.newBuilder((Key)key).set("concurrency", (long)backfill.concurrency())).set(PROPERTY_START, DatastoreStorage.instantToTimestamp(backfill.start()))).set(PROPERTY_END, DatastoreStorage.instantToTimestamp(backfill.end()))).set(PROPERTY_COMPONENT, backfill.workflowId().componentId())).set(PROPERTY_WORKFLOW, backfill.workflowId().id())).set(PROPERTY_SCHEDULE, backfill.schedule().toString())).set(PROPERTY_NEXT_TRIGGER, DatastoreStorage.instantToTimestamp(backfill.nextTrigger()))).set(PROPERTY_ALL_TRIGGERED, backfill.allTriggered())).set(PROPERTY_HALTED, backfill.halted());
        return builder.build();
    }

    Optional<Long> globalConcurrency() {
        return this.getOpt((DatastoreReader)this.datastore, this.globalConfigKey).filter(e -> e.contains("concurrency")).map(e -> e.getLong("concurrency"));
    }

    Optional<Double> submissionRate() {
        return this.getOpt((DatastoreReader)this.datastore, this.globalConfigKey).filter(e -> e.contains(PROPERTY_SUBMISSION_RATE_LIMIT)).map(e -> e.getDouble(PROPERTY_SUBMISSION_RATE_LIMIT));
    }

    Optional<List<String>> clientBlacklist() {
        return this.getOpt((DatastoreReader)this.datastore, this.globalConfigKey).filter(e -> e.contains(PROPERTY_CONFIG_CLIENT_BLACKLIST)).map(e -> e.getList(PROPERTY_CONFIG_CLIENT_BLACKLIST).stream().map(v -> (String)v.get()).collect(Collectors.toList()));
    }
}

