/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.styx.util;

import com.google.common.base.Throwables;
import com.spotify.styx.model.SequenceEvent;
import com.spotify.styx.model.WorkflowInstance;
import com.spotify.styx.state.OutputHandler;
import com.spotify.styx.state.RunState;
import com.spotify.styx.storage.Storage;
import com.spotify.styx.util.EventUtil;
import com.spotify.styx.util.Time;
import com.spotify.styx.util.TriggerUtil;
import java.io.IOException;
import java.time.Instant;
import java.util.Iterator;
import java.util.Map;
import java.util.Optional;
import java.util.SortedSet;
import java.util.stream.Collectors;
import javaslang.Tuple;
import javaslang.Tuple2;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ReplayEvents {
    private static final Logger LOG = LoggerFactory.getLogger(ReplayEvents.class);

    private ReplayEvents() {
    }

    public static Map<RunState, Long> replayActiveStates(Map<WorkflowInstance, Long> instances, Storage storage, boolean printLogs) throws IOException {
        LOG.info("Replaying active states");
        OutputHandler replayLogger = printLogs ? ReplayEvents.transitionLogger("  ") : OutputHandler.NOOP;
        return instances.entrySet().parallelStream().map(entry -> {
            SortedSet<SequenceEvent> sequenceEvents;
            WorkflowInstance workflowInstance = (WorkflowInstance)entry.getKey();
            long lastConsumedEvent = (Long)entry.getValue();
            SettableTime time = new SettableTime();
            if (printLogs) {
                LOG.info("Replaying {} up to #{}", (Object)workflowInstance.toKey(), (Object)lastConsumedEvent);
            }
            try {
                sequenceEvents = storage.readEvents(workflowInstance);
            }
            catch (IOException e) {
                throw Throwables.propagate((Throwable)e);
            }
            RunState restoredState = RunState.fresh(workflowInstance, (Time)time, new OutputHandler[0]);
            for (SequenceEvent sequenceEvent : sequenceEvents) {
                if (sequenceEvent.counter() > lastConsumedEvent) {
                    if (!printLogs) break;
                    LOG.error("Got unexpected newer event than the last consumed event {} > {} for {}", new Object[]{sequenceEvent.counter(), lastConsumedEvent, workflowInstance.toKey()});
                    break;
                }
                time.set(Instant.ofEpochMilli(sequenceEvent.timestamp()));
                if ("triggerExecution".equals(EventUtil.name(sequenceEvent.event()))) {
                    restoredState = RunState.fresh(workflowInstance, (Time)time, new OutputHandler[0]);
                }
                if (printLogs) {
                    LOG.info("  replaying #{} {}", (Object)sequenceEvent.counter(), (Object)sequenceEvent.event());
                }
                restoredState = restoredState.transition(sequenceEvent.event());
                replayLogger.transitionInto(restoredState);
            }
            return Tuple.of((Object)restoredState, (Object)lastConsumedEvent);
        }).collect(Collectors.toMap(Tuple2::_1, Tuple2::_2));
    }

    public static Optional<RunState> getBackfillRunState(WorkflowInstance workflowInstance, Map<WorkflowInstance, Long> activeWorkflowInstances, Storage storage, String backfillId) {
        SequenceEvent sequenceEvent;
        SortedSet<SequenceEvent> sequenceEvents;
        SettableTime time = new SettableTime();
        boolean backfillFound = false;
        try {
            sequenceEvents = storage.readEvents(workflowInstance);
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
        if (sequenceEvents.isEmpty()) {
            return Optional.empty();
        }
        RunState restoredState = RunState.fresh(workflowInstance, (Time)time, new OutputHandler[0]);
        long lastConsumedEvent = activeWorkflowInstances.getOrDefault(workflowInstance, sequenceEvents.last().counter());
        Iterator iterator = sequenceEvents.iterator();
        while (iterator.hasNext() && (sequenceEvent = (SequenceEvent)iterator.next()).counter() <= lastConsumedEvent) {
            time.set(Instant.ofEpochMilli(sequenceEvent.timestamp()));
            if ("triggerExecution".equals(EventUtil.name(sequenceEvent.event()))) {
                if (backfillFound) {
                    return Optional.of(restoredState);
                }
                restoredState = RunState.fresh(workflowInstance, (Time)time, new OutputHandler[0]);
            }
            restoredState = restoredState.transition(sequenceEvent.event());
            if (!"triggerExecution".equals(EventUtil.name(sequenceEvent.event())) || !restoredState.data().trigger().isPresent() || !backfillId.equals(TriggerUtil.triggerId(restoredState.data().trigger().get()))) continue;
            backfillFound = true;
        }
        return backfillFound ? Optional.of(restoredState) : Optional.empty();
    }

    public static OutputHandler transitionLogger(String prefix) {
        return state -> {
            String instanceKey = state.workflowInstance().toKey();
            LOG.info("{}{} transition -> {} {}", new Object[]{prefix, instanceKey, state.state().name().toLowerCase(), ReplayEvents.stateInfo(state)});
        };
    }

    private static String stateInfo(RunState state) {
        switch (state.state()) {
            case NEW: 
            case PREPARE: 
            case ERROR: 
            case DONE: {
                return String.format("tries:%d", state.data().tries());
            }
            case SUBMITTED: 
            case RUNNING: 
            case FAILED: {
                return String.format("tries:%d execId:%s", state.data().tries(), state.data().executionId());
            }
            case TERMINATED: {
                return String.format("tries:%d execId:%s exitCode:%s", state.data().tries(), state.data().executionId(), state.data().lastExit().map(String::valueOf).orElse("-"));
            }
            case QUEUED: {
                return String.format("tries:%d delayMs:%s", state.data().tries(), state.data().retryDelayMillis());
            }
        }
        return "";
    }

    private static final class SettableTime
    implements Time {
        private Instant now = Instant.now();

        private SettableTime() {
        }

        @Override
        public Instant get() {
            return this.now;
        }

        void set(Instant time) {
            this.now = time;
        }
    }
}

