/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.styx.api;

import com.google.api.client.googleapis.auth.oauth2.GoogleIdToken;
import com.google.api.client.googleapis.auth.oauth2.GoogleIdTokenVerifier;
import com.google.api.client.googleapis.json.GoogleJsonResponseException;
import com.google.api.services.cloudresourcemanager.CloudResourceManager;
import com.google.api.services.cloudresourcemanager.model.Ancestor;
import com.google.api.services.cloudresourcemanager.model.GetAncestryRequest;
import com.google.api.services.cloudresourcemanager.model.GetAncestryResponse;
import com.google.api.services.cloudresourcemanager.model.ListProjectsResponse;
import com.google.api.services.cloudresourcemanager.model.Project;
import com.google.api.services.cloudresourcemanager.model.ResourceId;
import com.google.api.services.iam.v1.Iam;
import com.google.api.services.iam.v1.model.ServiceAccount;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Sets;
import com.spotify.styx.api.AuthenticatorConfiguration;
import java.io.IOException;
import java.security.GeneralSecurityException;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Authenticator {
    private static final Logger logger = LoggerFactory.getLogger(Authenticator.class);
    private static final Pattern SERVICE_ACCOUNT_PATTERN = Pattern.compile("^.+\\.gserviceaccount\\.com$");
    private static final Pattern USER_CREATED_SERVICE_ACCOUNT_PATTERN = Pattern.compile("^.+@(.+)\\.iam\\.gserviceaccount\\.com$");
    private static final long VALIDATED_EMAIL_CACHE_SIZE = 1000L;
    private final GoogleIdTokenVerifier googleIdTokenVerifier;
    private final CloudResourceManager cloudResourceManager;
    private final Iam iam;
    private final Set<String> domainWhitelist;
    private final Set<ResourceId> resourceWhitelist;
    private final Set<ResourceId> resourceCache = Sets.newConcurrentHashSet();
    private final Cache<String, String> validatedEmailCache = CacheBuilder.newBuilder().maximumSize(1000L).build();

    Authenticator(GoogleIdTokenVerifier googleIdTokenVerifier, CloudResourceManager cloudResourceManager, Iam iam, AuthenticatorConfiguration configuration) {
        this.googleIdTokenVerifier = Objects.requireNonNull(googleIdTokenVerifier, "googleIdTokenVerifier");
        this.cloudResourceManager = Objects.requireNonNull(cloudResourceManager, "cloudResourceManager");
        this.iam = Objects.requireNonNull(iam, "iam");
        this.domainWhitelist = configuration.domainWhitelist();
        this.resourceWhitelist = configuration.resourceWhitelist();
    }

    void cacheResources() throws IOException {
        ListProjectsResponse response;
        CloudResourceManager.Projects.List request = this.cloudResourceManager.projects().list();
        do {
            if ((response = (ListProjectsResponse)request.execute()).getProjects() == null) continue;
            for (Project project : response.getProjects()) {
                boolean access = this.resolveProject(project);
                logger.info("Resolved project: {}, access={}", (Object)project.getProjectId(), (Object)access);
            }
            request.setPageToken(response.getNextPageToken());
        } while (response.getNextPageToken() != null);
        logger.info("Resource cache loaded");
    }

    GoogleIdToken authenticate(String token) {
        String projectId;
        GoogleIdToken googleIdToken;
        try {
            googleIdToken = this.verifyIdToken(token);
        }
        catch (IOException e) {
            logger.warn("Failed to verify token");
            return null;
        }
        if (googleIdToken == null) {
            return null;
        }
        String email = googleIdToken.getPayload().getEmail();
        if (email == null) {
            logger.debug("No email in id token");
            return null;
        }
        String domain = Authenticator.getDomain(email);
        if (domain != null) {
            if (this.domainWhitelist.contains(domain)) {
                logger.debug("Domain {} in whitelist", (Object)domain);
                return googleIdToken;
            }
        } else {
            logger.warn("Invalid email address {}", (Object)email);
            return null;
        }
        if (this.validatedEmailCache.getIfPresent((Object)email) != null) {
            return googleIdToken;
        }
        if (!SERVICE_ACCOUNT_PATTERN.matcher(email).matches()) {
            return null;
        }
        try {
            projectId = this.checkServiceAccountProject(email);
        }
        catch (IOException e) {
            logger.info("Cannot authenticate {}", (Object)email);
            return null;
        }
        if (projectId != null) {
            this.validatedEmailCache.put((Object)email, (Object)projectId);
            return googleIdToken;
        }
        return null;
    }

    @VisibleForTesting
    void clearResourceCache() {
        this.resourceCache.clear();
    }

    private boolean isWhitelisted(ResourceId resourceId) {
        return this.resourceWhitelist.contains(resourceId) || this.resourceCache.contains(resourceId);
    }

    @VisibleForTesting
    static ResourceId resourceId(String type, String id) {
        return new ResourceId().setType(type).setId(id);
    }

    @VisibleForTesting
    static ResourceId resourceId(Project project) {
        return Authenticator.resourceId("project", project.getProjectId());
    }

    private GoogleIdToken verifyIdToken(String token) throws IOException {
        try {
            return this.googleIdTokenVerifier.verify(token);
        }
        catch (GeneralSecurityException e) {
            return null;
        }
    }

    private String checkServiceAccountProject(String email) throws IOException {
        String projectId = null;
        Matcher matcher = USER_CREATED_SERVICE_ACCOUNT_PATTERN.matcher(email);
        if (matcher.matches()) {
            projectId = matcher.group(1);
        }
        if (projectId == null) {
            logger.debug("Email {} doesn't contain project id, looking up its project", (Object)email);
            projectId = this.getProjectIdOfServiceAccount(email);
        }
        if (projectId == null) {
            return null;
        }
        if (this.isWhitelisted(Authenticator.resourceId("project", projectId))) {
            logger.debug("Hit cache for project id {}", (Object)projectId);
            return projectId;
        }
        if (this.resolveProjectAccess(projectId)) {
            return projectId;
        }
        return null;
    }

    private String getProjectIdOfServiceAccount(String email) throws IOException {
        try {
            ServiceAccount serviceAccount = (ServiceAccount)this.iam.projects().serviceAccounts().get("projects/-/serviceAccounts/" + email).execute();
            return serviceAccount.getProjectId();
        }
        catch (GoogleJsonResponseException e) {
            if (e.getStatusCode() == 404) {
                logger.debug("Service account {} doesn't exist", (Object)email, (Object)e);
                return null;
            }
            logger.info("Cannot get project id for service account {}", (Object)email, (Object)e);
            return null;
        }
    }

    private boolean resolveProjectAccess(String projectId) throws IOException {
        GetAncestryResponse ancestry;
        try {
            ancestry = (GetAncestryResponse)this.cloudResourceManager.projects().getAncestry(projectId, new GetAncestryRequest()).execute();
        }
        catch (GoogleJsonResponseException e) {
            if (e.getStatusCode() == 404) {
                logger.debug("Project {} doesn't exist", (Object)projectId, (Object)e);
                return false;
            }
            logger.info("Cannot get project with id {}", (Object)projectId, (Object)e);
            return false;
        }
        return this.resolveAccess(ancestry.getAncestor());
    }

    private boolean resolveProject(Project project) throws IOException {
        ResourceId resourceId = Authenticator.resourceId(project);
        if (this.isWhitelisted(resourceId)) {
            return true;
        }
        if (project.getParent() != null && this.isWhitelisted(project.getParent())) {
            return true;
        }
        return this.resolveProjectAccess(project.getProjectId());
    }

    private boolean resolveAccess(List<Ancestor> ancestry) {
        for (int i = 0; i < ancestry.size(); ++i) {
            Ancestor ancestor = ancestry.get(i);
            if (!this.isWhitelisted(ancestor.getResourceId())) continue;
            for (Ancestor descendant : ancestry.subList(0, i)) {
                if (!this.resourceCache.add(descendant.getResourceId())) continue;
                logger.debug("Whitelist cached {}/{}, descendant of {}/{}", new Object[]{descendant.getResourceId().getType(), descendant.getResourceId().getId(), ancestor.getResourceId().getType(), ancestor.getResourceId().getId()});
            }
            return true;
        }
        return false;
    }

    private static String getDomain(String email) {
        int index = email.indexOf(64);
        if (index == -1) {
            return null;
        }
        return email.substring(index + 1);
    }
}

