/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.styx.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.collect.Range;
import com.spotify.styx.monitoring.Stats;
import com.spotify.styx.storage.Storage;
import com.spotify.styx.storage.StorageTransaction;
import com.spotify.styx.util.CounterCapacityException;
import com.spotify.styx.util.CounterSnapshot;
import com.spotify.styx.util.CounterSnapshotFactory;
import com.spotify.styx.util.Shard;
import com.spotify.styx.util.ShardNotFoundException;
import java.io.IOException;
import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Random;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ShardedCounter {
    private static final Logger LOG = LoggerFactory.getLogger(ShardedCounter.class);
    public static final int NUM_SHARDS = 128;
    private static final Duration CACHE_EXPIRY_DURATION = Duration.ofMillis(1000L);
    public static final String KIND_COUNTER_LIMIT = "CounterLimit";
    public static final String PROPERTY_LIMIT = "limit";
    public static final String KIND_COUNTER_SHARD = "CounterShard";
    public static final String PROPERTY_SHARD_VALUE = "value";
    public static final String PROPERTY_SHARD_INDEX = "index";
    public static final String PROPERTY_COUNTER_ID = "counterId";
    private final Stats stats;
    @VisibleForTesting
    final Cache<String, CounterSnapshot> inMemSnapshot = CacheBuilder.newBuilder().maximumSize(100000L).expireAfterWrite(CACHE_EXPIRY_DURATION.toMillis(), TimeUnit.MILLISECONDS).build();
    private CounterSnapshotFactory counterSnapshotFactory;

    public ShardedCounter(Stats stats, CounterSnapshotFactory counterSnapshotFactory) {
        this.stats = Objects.requireNonNull(stats);
        this.counterSnapshotFactory = Objects.requireNonNull(counterSnapshotFactory);
    }

    public CounterSnapshot getCounterSnapshot(String counterId) throws IOException {
        CounterSnapshot snapshot = (CounterSnapshot)this.inMemSnapshot.getIfPresent((Object)counterId);
        if (snapshot != null) {
            this.stats.recordCounterCacheHit();
            return snapshot;
        }
        this.stats.recordCounterCacheMiss();
        return this.refreshCounterSnapshot(counterId);
    }

    private CounterSnapshot refreshCounterSnapshot(String counterId) throws IOException {
        CounterSnapshot newSnapshot = this.counterSnapshotFactory.create(counterId);
        this.inMemSnapshot.put((Object)counterId, (Object)newSnapshot);
        return newSnapshot;
    }

    public boolean counterHasSpareCapacity(String resourceId) throws IOException {
        try {
            CounterSnapshot counterSnapshot = this.getCounterSnapshot(resourceId);
            counterSnapshot.pickShardWithSpareCapacity(1L);
            return true;
        }
        catch (CounterCapacityException e) {
            return false;
        }
    }

    public void updateLimit(StorageTransaction tx, String counterId, long limit) throws IOException {
        tx.updateLimitForCounter(counterId, limit);
    }

    static long getLimit(Storage storage, String counterId) throws IOException {
        return storage.getLimitForCounter(counterId);
    }

    public void updateCounter(StorageTransaction transaction, String counterId, long delta) throws IOException {
        Optional<Integer> shardIndex;
        CounterSnapshot snapshot = this.getCounterSnapshot(counterId);
        if (delta < 0L && (shardIndex = snapshot.pickShardWithExcessUsage(delta)).isPresent()) {
            this.updateCounterShard(transaction, counterId, delta, shardIndex.get(), snapshot.shardCapacity(shardIndex.get()));
            return;
        }
        int shardIndex2 = snapshot.pickShardWithSpareCapacity(delta);
        this.updateCounterShard(transaction, counterId, delta, shardIndex2, snapshot.shardCapacity(shardIndex2));
    }

    /*
     * Enabled aggressive block sorting
     */
    @VisibleForTesting
    void updateCounterShard(StorageTransaction transaction, String counterId, long delta, int shardIndex, long shardCapacity) throws IOException {
        Optional<Shard> shard = transaction.shard(counterId, shardIndex);
        if (!shard.isPresent()) {
            String message = String.format("Could not find shard %s-%s. Unexpected Datastore corruption or ourbug - the code should've called initialize() before reaching thispoint, and any particular shard should strongly be get()-ablethereafter", counterId, shardIndex);
            LOG.error(message);
            throw new ShardNotFoundException(message);
        }
        long newShardValue = (long)shard.get().value() + delta;
        if (delta < 0L && newShardValue >= 0L) {
            transaction.store(Shard.create(counterId, shardIndex, (int)newShardValue));
        } else if (delta > 0L && Range.closed((Comparable)Long.valueOf(0L), (Comparable)Long.valueOf(shardCapacity)).contains((Comparable)Long.valueOf(newShardValue))) {
            transaction.store(Shard.create(counterId, shardIndex, (int)newShardValue));
        } else {
            String message = String.format("Chosen shard %s-%s has no more capacity: capacity=%d, value=%d, delta=%d, newValue=%d", counterId, shardIndex, shardCapacity, shard.get().value(), delta, newShardValue);
            LOG.info(message);
            throw new CounterCapacityException(message);
        }
        String operation = delta > 0L ? "increment" : "decrement";
        LOG.info("Updating counter shard ({}): {}-{}: capacity={}, value={}, delta={}, newValue={}", new Object[]{operation, counterId, shardIndex, shardCapacity, shard.get().value(), delta, newShardValue});
    }

    public long getCounter(String counterId) throws IOException {
        return this.getCounterSnapshot(counterId).getTotalUsage();
    }

    public static class Snapshot
    implements CounterSnapshot {
        private final String counterId;
        private final Long limit;
        private final Map<Integer, Long> shards;

        Snapshot(String counterId, long limit, Map<Integer, Long> shards) {
            this.counterId = Objects.requireNonNull(counterId);
            this.limit = limit;
            this.shards = Objects.requireNonNull(shards);
        }

        @Override
        public long shardCapacity(int shardIndex) {
            return this.limit / 128L + (long)((long)shardIndex < this.limit % 128L ? 1 : 0);
        }

        @Override
        public long getLimit() {
            return this.limit;
        }

        @Override
        public long getTotalUsage() {
            return this.shards.values().stream().mapToLong(i -> i).sum();
        }

        @Override
        public Optional<Integer> pickShardWithExcessUsage(long delta) {
            List candidates = this.shards.keySet().stream().filter(index -> this.shards.get(index) > this.shardCapacity((int)index)).filter(index -> this.shards.get(index) + delta >= 0L).collect(Collectors.toList());
            if (candidates.isEmpty()) {
                return Optional.empty();
            }
            return Optional.of((Integer)candidates.get(new Random().nextInt(candidates.size())));
        }

        @Override
        public Map<Integer, Long> getShards() {
            return this.shards;
        }

        @Override
        public int pickShardWithSpareCapacity(long delta) {
            if (delta > 0L && this.getTotalUsage() >= this.getLimit()) {
                String message = String.format("No shard for counter %s has capacity for delta %s", this.counterId, delta);
                LOG.info(message);
                throw new CounterCapacityException(message);
            }
            List candidates = this.shards.keySet().stream().filter(index -> Range.closed((Comparable)Long.valueOf(0L), (Comparable)Long.valueOf(this.shardCapacity((int)index))).contains((Comparable)Long.valueOf(this.shards.get(index) + delta))).collect(Collectors.toList());
            if (candidates.isEmpty()) {
                if (this.shards.size() == 0) {
                    String message = "Trying to operate with a potentially uninitialized counter " + this.counterId + ". Cache needs to be updated first.";
                    LOG.error(message);
                    throw new ShardNotFoundException(message);
                }
                String message = String.format("No shard for counter %s has capacity for delta %s", this.counterId, delta);
                LOG.info(message);
                throw new CounterCapacityException(message);
            }
            return (Integer)candidates.get(new Random().nextInt(candidates.size()));
        }
    }
}

