/*
 * Decompiled with CFR 0.152.
 */
package com.spotify.styx.util;

import com.spotify.styx.model.FlyteIdentifier;
import com.spotify.styx.model.Workflow;
import com.spotify.styx.model.WorkflowConfiguration;
import com.spotify.styx.model.WorkflowId;
import com.spotify.styx.util.DockerImageValidator;
import com.spotify.styx.util.TimeUtil;
import com.spotify.styx.util.WorkflowValidator;
import java.time.Duration;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.time.format.DateTimeParseException;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.springframework.expression.spel.standard.SpelExpressionParser;

public class BasicWorkflowValidator
implements WorkflowValidator {
    private static final int MAX_ID_LENGTH = 256;
    private static final int MAX_DOCKER_ARGS_TOTAL = 1000000;
    private static final int MAX_RESOURCES = 5;
    private static final int MAX_RESOURCE_LENGTH = 256;
    private static final int MAX_COMMIT_SHA_LENGTH = 256;
    private static final int MAX_SERVICE_ACCOUNT_LENGTH = 256;
    private static final int MAX_RETRY_CONDITION_LENGTH = 256;
    private static final int MAX_ENV_VARS = 128;
    private static final int MAX_ENV_SIZE = 16384;
    private static final Pattern VALID_EMAIL_ADDRESS_REGEX = Pattern.compile("^[_A-Za-z0-9-+]+(\\.[_A-Za-z0-9-]+)*@[A-Za-z0-9-]+(\\.[A-Za-z0-9]+)*(\\.[A-Za-z]{2,})$", 2);
    private static final String RESOURCE_TYPE_LAUNCH_PLAN = "LAUNCH_PLAN";
    private static final Duration MIN_RUNNING_TIMEOUT = Duration.ofMinutes(1L);
    private final DockerImageValidator dockerImageValidator;

    public BasicWorkflowValidator(DockerImageValidator dockerImageValidator) {
        this.dockerImageValidator = Objects.requireNonNull(dockerImageValidator);
    }

    @Override
    public List<String> validateWorkflow(Workflow workflow) {
        WorkflowId workflowId = workflow.id();
        WorkflowConfiguration cfg = workflow.configuration();
        ArrayList<String> e = new ArrayList<String>();
        String componentId = workflowId.componentId();
        this.empty(e, "component id", workflowId.componentId());
        if (componentId.contains("#")) {
            e.add("component id cannot contain #");
        }
        this.empty(e, "workflow id", workflowId.id());
        if (!workflowId.id().equals(cfg.id())) {
            e.add("workflow id mismatch");
        }
        WorkflowValidator.upperLimit(e, cfg.id().length(), 256, "id too long");
        WorkflowValidator.upperLimit(e, cfg.commitSha().map(String::length).orElse(0), 256, "commitSha too long");
        WorkflowValidator.upperLimit(e, cfg.serviceAccount().map(String::length).orElse(0), 256, "service account too long");
        WorkflowValidator.upperLimit(e, cfg.resources().size(), 5, "too many resources");
        WorkflowValidator.upperLimit(e, cfg.env().size(), 128, "too many env vars");
        WorkflowValidator.upperLimit(e, cfg.env().entrySet().stream().mapToInt(entry -> ((String)entry.getKey()).length() + ((String)entry.getValue()).length()).sum(), 16384, "env too big");
        WorkflowValidator.upperLimit(e, cfg.retryCondition().map(String::length).orElse(0), 256, "retry condition too long");
        cfg.resources().stream().map(String::length).forEach(v -> WorkflowValidator.upperLimit(e, v, 256, "resource name too long"));
        cfg.offset().ifPresent(offset -> {
            try {
                TimeUtil.addOffset(ZonedDateTime.now(ZoneOffset.UTC), offset);
            }
            catch (DateTimeParseException ex) {
                e.add(String.format("invalid offset: %s", ex.getMessage()));
            }
        });
        try {
            TimeUtil.cron(cfg.schedule());
        }
        catch (IllegalArgumentException ex) {
            e.add("invalid schedule");
        }
        cfg.runningTimeout().ifPresent(timeout -> WorkflowValidator.lowerLimit(e, timeout, MIN_RUNNING_TIMEOUT, "running timeout is too small"));
        cfg.serviceAccount().ifPresent(serviceAccount -> {
            if (!BasicWorkflowValidator.validateServiceAccount(serviceAccount)) {
                e.add("service account is not a valid email address: " + serviceAccount);
            }
        });
        cfg.retryCondition().ifPresent(retryCondition -> {
            try {
                new SpelExpressionParser().parseRaw(retryCondition);
            }
            catch (RuntimeException ex) {
                e.add(String.format("invalid retry condition: %s", ex.getMessage()));
            }
        });
        this.validateConflictingExecConf(e, cfg);
        this.validateDockerConf(e, cfg);
        this.validateFlyteExecConf(e, cfg);
        return e;
    }

    private static boolean validateServiceAccount(String serviceAccount) {
        Matcher matcher = VALID_EMAIL_ADDRESS_REGEX.matcher(serviceAccount);
        return matcher.matches();
    }

    private void validateConflictingExecConf(List<String> e, WorkflowConfiguration cfg) {
        boolean hasDockerConf;
        boolean bl = hasDockerConf = cfg.dockerImage().isPresent() || cfg.dockerArgs().isPresent();
        if (cfg.flyteExecConf().isPresent() && hasDockerConf) {
            e.add("configuration cannot specify both docker and flyte parameters");
        }
    }

    private void validateDockerConf(List<String> errors, WorkflowConfiguration config) {
        config.dockerArgs().ifPresent(args -> this.validateDockerArgs(errors, (List<String>)args));
        config.dockerImage().ifPresent(image -> this.validateDockerImage(errors, (String)image));
    }

    private void validateDockerImage(List<String> errors, String image) {
        this.dockerImageValidator.validateImageReference(image).forEach(error -> errors.add("invalid image: " + error));
    }

    private void validateDockerArgs(List<String> errors, List<String> args) {
        int dockerArgs = args.size() + args.stream().mapToInt(String::length).sum();
        WorkflowValidator.upperLimit(errors, dockerArgs, 1000000, "docker args is too large");
    }

    private void validateFlyteExecConf(List<String> errors, WorkflowConfiguration config) {
        config.flyteExecConf().ifPresent(flyteExecConf -> {
            FlyteIdentifier flyteIdentifier = flyteExecConf.referenceId();
            String resourceType = flyteIdentifier.resourceType();
            if (!RESOURCE_TYPE_LAUNCH_PLAN.equals(resourceType)) {
                errors.add("only launch plans (\"LAUNCH_PLAN\") are supported as resource type, but received: " + resourceType);
            }
            this.empty(errors, "domain", flyteIdentifier.domain());
            this.empty(errors, "project", flyteIdentifier.project());
            this.empty(errors, "name", flyteIdentifier.name());
            this.empty(errors, "version", flyteIdentifier.version());
        });
    }

    private void empty(List<String> errors, String fieldName, String value) {
        if (value.isEmpty()) {
            errors.add(fieldName + " cannot be empty");
        }
    }
}

