[workflow-core](../../index.md) / [com.squareup.workflow](../index.md) / [Worker](./index.md)

# Worker

`interface Worker<out OutputT>`

Represents a unit of asynchronous work that can have zero, one, or multiple outputs.

Workers allow you to execute arbitrary, possibly asynchronous tasks in a declarative manner. To
perform their tasks, workers return a [Flow](#). Workers are effectively [Flow](#)s that can be
[compared](does-same-work-as.md) to determine equivalence between render passes. A [Workflow](../-workflow/index.md) uses
Workers to perform asynchronous work during the render pass by calling
[RenderContext.runningWorker](../-render-context/running-worker.md).

See the documentation on [run](run.md) for more information on the returned [Flow](#) is consumed and how
to implement asynchronous work.

See the documentation on [doesSameWorkAs](does-same-work-as.md) for more details on how and when workers are compared
and the worker lifecycle.

## Example: Network request

Let's say you have a network service with an API that returns a number, and you want to
call that service from a [Workflow](../-workflow/index.md).

```
interface TimeService {
  suspend fun getTime(timezone: String): Long
}
```

The first step is to define a Worker that can call this service, and maybe an extension
function on your service class:

```
fun TimeService.getTimeWorker(timezone: String): Worker<Long> = TimeWorker(timezone, this)

private class TimeWorker(
  val timezone: String,
  val service: TimeService
): Worker<Long> {

  override fun run(): Flow<Long> = flow {
    val time = service.getTime(timezone)
    emit(time)
  }
}
```

You also need to define how to determine if a previous Worker is already doing the same work.
This will ensure that if the same request is made by the same [Workflow](../-workflow/index.md) in adjacent render
passes, we'll keep the request alive from the first pass.

```
override fun doesSameWorkAs(otherWorker: Worker<*>): Boolean =
  otherWorker is TimeWorker &&
    timezone == otherWorker.timezone
```

Now you can request the time from your [Workflow](../-workflow/index.md):

```
class MyWorkflow(private val timeWorker: TimeWorker) {
  override fun render(…): Foo {
    context.onWorkerOutput(timeWorker) { time -> emitOutput("The time is $time") }
  }
```

Alternatively, if the response is a unique type, unlikely to be shared by any other workers,
you don't even need to create your own Worker class, you can use a builder, and the worker
will automatically be distinguished by that response type:

```
interface TimeService {
  fun getTime(timezone: String): Deferred<TimeResponse>
}

fun TimeService.getTimeWorker(timezone: String): Worker<TimeResponse> =
  Worker.from { getTime(timezone).await()) }
```

**See Also**

[create](create.md)

[from](from.md)

[fromNullable](from-nullable.md)

[Deferred.asWorker](../kotlinx.coroutines.-deferred/as-worker.md)

[BroadcastChannel.asWorker](../kotlinx.coroutines.channels.-broadcast-channel/as-worker.md)

### Functions

| Name | Summary |
|---|---|
| [doesSameWorkAs](does-same-work-as.md) | Override this method to define equivalence between [Worker](./index.md)s.`open fun doesSameWorkAs(otherWorker: `[`Worker`](./index.md)`<*>): `[`Boolean`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html) |
| [run](run.md) | Returns a [Flow](#) to execute the work represented by this worker.`abstract fun run(): Flow<OutputT>` |

### Companion Object Functions

| Name | Summary |
|---|---|
| [create](create.md) | Shorthand for `flow { block() }.asWorker()`.`fun <OutputT> create(block: suspend FlowCollector<OutputT>.() -> `[`Unit`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-unit/index.html)`): `[`Worker`](./index.md)`<OutputT>` |
| [createSideEffect](create-side-effect.md) | Creates a [Worker](./index.md) that just performs some side effects and doesn't emit anything. Run the worker from your `render` method using [RenderContext.runningWorker](../-render-context/running-worker.md).`fun createSideEffect(block: suspend () -> `[`Unit`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-unit/index.html)`): `[`Worker`](./index.md)`<`[`Nothing`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-nothing/index.html)`>` |
| [finished](finished.md) | Returns a [Worker](./index.md) that finishes immediately without emitting anything.`fun <T> finished(): `[`Worker`](./index.md)`<T>` |
| [from](from.md) | Creates a [Worker](./index.md) from a function that returns a single value.`fun <OutputT> from(block: suspend () -> OutputT): `[`Worker`](./index.md)`<OutputT>` |
| [fromNullable](from-nullable.md) | Creates a [Worker](./index.md) from a function that returns a single value. The worker will emit the value **if and only if the value is not null**, then finish.`fun <OutputT : `[`Any`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-any/index.html)`> fromNullable(block: suspend () -> OutputT?): `[`Worker`](./index.md)`<OutputT>` |
| [timer](timer.md) | Creates a [Worker](./index.md) that will emit [Unit](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-unit/index.html) and then finish after [delayMs](timer.md#com.squareup.workflow.Worker.Companion$timer(kotlin.Long, kotlin.String)/delayMs) milliseconds. Negative delays are clamped to zero.`fun timer(delayMs: `[`Long`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-long/index.html)`, key: `[`String`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)` = ""): `[`Worker`](./index.md)`<`[`Unit`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-unit/index.html)`>` |

### Extension Functions

| Name | Summary |
|---|---|
| [transform](../transform.md) | Returns a [Worker](./index.md) that transforms this [Worker](./index.md)'s [Flow](#) by calling [transform](../transform.md#com.squareup.workflow$transform(com.squareup.workflow.Worker((com.squareup.workflow.transform.T)), kotlin.Function1((kotlinx.coroutines.flow.Flow((com.squareup.workflow.transform.T)), kotlinx.coroutines.flow.Flow((com.squareup.workflow.transform.R)))))/transform).`fun <T, R> `[`Worker`](./index.md)`<T>.transform(transform: (Flow<T>) -> Flow<R>): `[`Worker`](./index.md)`<R>` |

### Inheritors

| Name | Summary |
|---|---|
| [LifecycleWorker](../-lifecycle-worker/index.md) | [Worker](./index.md) that performs some action when the worker is started and/or stopped.`abstract class LifecycleWorker : `[`Worker`](./index.md)`<`[`Nothing`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-nothing/index.html)`>` |
