[workflow-testing](../../index.md) / [com.squareup.workflow.testing](../index.md) / [RenderTester](./index.md)

# RenderTester

`interface RenderTester<PropsT, StateT, OutputT : `[`Any`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-any/index.html)`, RenderingT>`

The props must be specified, the initial state may be specified, and then all child workflows
and workers that are expected to run, and any outputs from them, must be specified with
[expectWorkflow](expect-workflow.md) and [expectWorker](expect-worker.md) calls. Then call [render](render.md) and perform any assertions on the
rendering. An event may also be sent to the rendering if no workflows or workers emitted an
output. Lastly, the [RenderTestResult](../-render-test-result/index.md) returned by `render` may be used to assert on the
[WorkflowAction](#)s processed to handle events or outputs by calling
[verifyAction](../-render-test-result/verify-action.md) or
[verifyActionResult](../-render-test-result/verify-action-result.md).

* All workflows and workers that are rendered/ran by this workflow must be specified.
* It is an error if more than one workflow or worker specifies an output.
* It is a test failure if any workflows or workers that were expected were not ran.
* It is a test failure if the workflow tried to run any workflows or workers that were not
expected.
* It is a test failure if no workflow or workflow emitted an output, no rendering event was
invoked, and any of the action verification methods on [RenderTestResult](../-render-test-result/index.md) is called.

## Examples

### Worker output

The following example tests a render pass that runs one worker, `SubmitLoginWorker`, which
is configured to have "emitted" an output, and one workflow, `ChildWorkflow`, which expects a
props containing "test@foo.com" and returning a `ChildRendering` as its rendering.

It checks that the rendering properties are expected and that the output handler for the
`SubmitLoginWorker` returned the `CompleteLogin` action.

```
workflow
  .renderTester(
    props = MyProps(…),
    initialState = MyState(…)
  )
  .expectWorker(
    matchesWhen = { it is SubmitLoginWorker },
    key = "signin",
    output = EmittedOutput(LoginResponse(success = true))
  )
  .expectWorkflow(
    workflowType = ChildWorkflow::class,
    key = "child",
    assertProps = { assertThat(it.email).isEqualTo("test@foo.com") },
    rendering = ChildRendering("message")
  )
  .render { rendering ->
    assertThat(rendering.text).isEqualTo("foo")
  }
  .verifyAction { action ->
    assertThat(action).isEqualTo(Action.CompleteLogin(success = true))
  }
```

### Rendering event

This is similar to the example above, but will test an event sent to the rendering instead.

```
workflow
  .renderTester(
    props = MyProps(…),
    initialState = MyState(…)
  )
  .expectWorker(
    matchesWhen = { it is SubmitLoginWorker },
    key = "signin"
  )
  .expectWorkflow(
    workflowType = ChildWorkflow::class,
    key = "child",
    assertProps = { assertThat(it.email).isEqualTo("test@foo.com") },
    rendering = ChildRendering("message")
  )
  .render { rendering ->
    rendering.onCancelClicked()
    assertThat(rendering.text).isEqualTo("foo")
  }
  .verifyAction { action ->
    assertThat(action).isEqualTo(Action.CancelLogin)
  }
```

### Verify action result

This test verifies the action *result* instead of the action itself. This technique is useful
if the [WorkflowAction](#) is anonymous or inline.

```
val currentState = …
val previousState = …

workflow
  .renderTester(
    props = MyProps(…),
    initialState = currentState
  )
  .render { rendering ->
    rendering.onCancelClicked()
  }
  .verifyActionResult { newState, output ->
    // Check that the workflow navigated back correctly.
    assertThat(newState).isEqualTo(previousState)

    // Check that the workflow didn't emit any output from the button click.
    assertThat(output).isNull()
  }
```

### Too many outputs

This is an example of what **not** to do – this test will error out because a worker is emitting
and output *and* a rendering event is sent.

```
workflow
  .renderTester(
    props = MyProps(…),
    initialState = MyState(…)
  )
  .expectWorker(
    matchesWhen = { it is SubmitLoginWorker },
    key = "signin",
    output = EmittedOutput(LoginResponse(success = true))
  )
  .expectWorkflow(
    workflowType = ChildWorkflow::class,
    key = "child",
    assertProps = { assertThat(it.email).isEqualTo("test@foo.com") },
    rendering = ChildRendering("message")
  )
  .render { rendering ->
    // This will throw and fail the test because the SubmitLoginWorker is also configured to emit
    // an output.
    rendering.onCancelClicked()
```

### Functions

| Name | Summary |
|---|---|
| [expectWorker](expect-worker.md) | Specifies that this render pass is expected to run a particular worker.`abstract fun expectWorker(matchesWhen: (otherWorker: Worker<*>) -> `[`Boolean`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-boolean/index.html)`, key: `[`String`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)` = "", output: `[`EmittedOutput`](../-emitted-output/index.md)`<`[`Any`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-any/index.html)`?>? = null): `[`RenderTester`](./index.md)`<PropsT, StateT, OutputT, RenderingT>` |
| [expectWorkflow](expect-workflow.md) | Specifies that this render pass is expected to render a particular child workflow.`abstract fun <ChildPropsT, ChildOutputT : `[`Any`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-any/index.html)`, ChildRenderingT> expectWorkflow(workflowType: `[`KClass`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin.reflect/-k-class/index.html)`<out Workflow<ChildPropsT, ChildOutputT, ChildRenderingT>>, rendering: ChildRenderingT, key: `[`String`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)` = "", assertProps: (props: ChildPropsT) -> `[`Unit`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-unit/index.html)` = {}, output: `[`EmittedOutput`](../-emitted-output/index.md)`<ChildOutputT>? = null): `[`RenderTester`](./index.md)`<PropsT, StateT, OutputT, RenderingT>` |
| [render](render.md) | Execute the workflow's `render` method and run [block](render.md#com.squareup.workflow.testing.RenderTester$render(kotlin.Function1((com.squareup.workflow.testing.RenderTester.RenderingT, kotlin.Unit)))/block) to perform assertions on and send events to the resulting rendering.`abstract fun render(block: (rendering: RenderingT) -> `[`Unit`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-unit/index.html)` = {}): `[`RenderTestResult`](../-render-test-result/index.md)`<StateT, OutputT>` |

### Extension Functions

| Name | Summary |
|---|---|
| [expectWorker](../expect-worker.md) | Specifies that this render pass is expected to run a particular worker.`fun <PropsT, StateT, OutputT : `[`Any`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-any/index.html)`, RenderingT> `[`RenderTester`](./index.md)`<PropsT, StateT, OutputT, RenderingT>.expectWorker(doesSameWorkAs: Worker<OutputT>, key: `[`String`](https://kotlinlang.org/api/latest/jvm/stdlib/kotlin/-string/index.html)` = "", output: `[`EmittedOutput`](../-emitted-output/index.md)`<OutputT>? = null): `[`RenderTester`](./index.md)`<PropsT, StateT, OutputT, RenderingT>` |
