package com.twitter.finagle.netty4.channel

import com.twitter.finagle.stats.StatsReceiver
import io.netty.channel._
import io.netty.channel.ChannelHandler.Sharable
import io.netty.util.AttributeKey
import java.util.concurrent.atomic.AtomicInteger

private[finagle] object ChannelRequestStatsHandler {
  private[channel] val ConnectionRequestsKey: AttributeKey[AtomicInteger] =
    AttributeKey.valueOf("ChannelRequestStatsHandler.connection_requests")
}

/**
 * A channel stats handler that keeps per-connection request
 * statistics. This handler should be after the request codec in the
 * stack as it assumes messages are POJOs with request/responses.
 *
 * @param statsReceiver the [[StatsReceiver]] to which stats are reported
 */
@Sharable
private[finagle] class ChannelRequestStatsHandler(statsReceiver: StatsReceiver)
    extends ChannelInboundHandlerAdapter {
  import ChannelRequestStatsHandler.ConnectionRequestsKey

  private[this] val requestCount = statsReceiver.stat("connection_requests")

  override def handlerAdded(ctx: ChannelHandlerContext): Unit = {
    ctx.channel.attr(ConnectionRequestsKey).set(new AtomicInteger(0))
    super.handlerAdded(ctx)
  }

  override def channelInactive(ctx: ChannelHandlerContext): Unit = {
    requestCount.add(ctx.channel.attr(ConnectionRequestsKey).get.get)
    super.channelInactive(ctx)
  }

  override def channelRead(ctx: ChannelHandlerContext, msg: Any): Unit = {
    val readCount = ctx.channel.attr(ConnectionRequestsKey).get
    readCount.incrementAndGet()
    super.channelRead(ctx, msg)
  }
}
