package com.vaadin.spring.roo.addon;

import java.io.IOException;
import java.util.HashSet;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.springframework.roo.model.JavaPackage;
import org.springframework.roo.model.JavaSymbolName;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.shell.CliAvailabilityIndicator;
import org.springframework.roo.shell.CliCommand;
import org.springframework.roo.shell.CliOption;
import org.springframework.roo.shell.CommandMarker;
import org.springframework.roo.support.util.StringUtils;

import com.vaadin.spring.roo.addon.addons.VaadinAddonArtifactId;
import com.vaadin.spring.roo.addon.addons.VaadinAddonOperations;

/**
 * Vaadin add-on command class.
 *
 * The command class is registered by the Roo shell following an automatic
 * classpath scan.
 *
 * @since 1.1.0-M1
 */
@Component
@Service
public class VaadinCommands implements CommandMarker {

	private static Logger logger = Logger.getLogger(VaadinCommands.class
			.getName());

	@Reference
	private VaadinOperations operations;
	@Reference
	private VaadinAddonOperations addonOperations;

	@CliAvailabilityIndicator("vaadin setup")
	public boolean isVaadinSetupAvailable() {
		// allow re-performing "vaadin setup" also if already done
		return operations.isProjectAvailable();
	}

	@CliAvailabilityIndicator({ "vaadin generate all", "vaadin scaffold" })
	public boolean isVaadinGenerateViewAvailable() {
		return operations.isProjectAvailable() && operations.isVaadinSetup()
				&& operations.isPersistenceSetup();
	}

	@CliAvailabilityIndicator("vaadin addon install")
	public boolean isVaadinInstallAddonAvailable() {
		return operations.isProjectAvailable() && operations.isVaadinSetup()
				&& operations.isPersistenceSetup();
	}

	@CliAvailabilityIndicator({ "vaadin widgetset update",
			"vaadin widgetset compile" })
	public boolean isVaadinWidgetsetConfigured() {
		return operations.isProjectAvailable() && operations.isVaadinSetup()
				&& operations.isWidgetsetSetup();
	}

	@CliCommand(value = "vaadin setup", help = "Adds Vaadin to a project and creates an Application and a Window")
	public void vaadinSetup(
			@CliOption(key = { "applicationPackage", "" }, mandatory = true, help = "The package in which a Vaadin application should be created") JavaPackage applicationPackage,
			@CliOption(key = { "baseName", "" }, mandatory = false, help = "Base name for classes created (application, window and view)") JavaSymbolName baseName,
			@CliOption(key = { "themeName", "" }, mandatory = false, help = "Name of the Vaadin theme to create (default is based on application base name)") JavaSymbolName themeName,
			@CliOption(key = "applicationNameHtml", mandatory = false, unspecifiedDefaultValue = "Welcome", help = "Application name HTML string to show on the UI") String appNameHtml,
			@CliOption(key = "useJpaContainer", mandatory = false, unspecifiedDefaultValue = "false", help = "Use the lazily loading JPAContainer instead of in-memory entity lists (AGPL3 / Commercial Vaadin add-on)") boolean useJpaContainer) {
		operations.vaadinSetup(applicationPackage, baseName, themeName,
				appNameHtml, useJpaContainer);
	}

	@CliCommand(value = "vaadin generate all", help = "Scaffold a view for all existing entities without an entity view")
	public void vaadinGenerateAll(
			@CliOption(key = { "package", "" }, mandatory = true, help = "The package in which the views will be created") JavaPackage viewPackage,
			@CliOption(key = { "baseClass", "" }, mandatory = false, help = "The path and name of the abstract base class for entity views to create or use, by default AbstractEntityView") JavaType baseClass,
			@CliOption(key = "visuallyComposable", mandatory = false, unspecifiedDefaultValue = "false", help = "Make the form layout editable with the Visual Editor (note: there are no automatic updates to visually composable forms based on entity changes)") boolean visuallyComposable) {
		operations.generateAll(viewPackage, visuallyComposable, baseClass);
	}

	@CliCommand(value = "vaadin scaffold", help = "Create a new scaffold view for an entity class (ie where we maintain CRUD automatically)")
	public void vaadinScaffold(
			@CliOption(key = { "class", "" }, mandatory = true, help = "The path and name of the entity view class to be created") JavaType view,
			@CliOption(key = "entity", mandatory = false, optionContext = "update,project", unspecifiedDefaultValue = "*", help = "The name of the entity object which the view manages") JavaType entity,
			@CliOption(key = { "baseClass", "" }, mandatory = false, help = "The path and name of the abstract base class for entity views to create or use, by default AbstractEntityView") JavaType baseClass,
			@CliOption(key = "visuallyComposable", mandatory = false, unspecifiedDefaultValue = "false", help = "Make the form layout editable with the Visual Editor (note: there are no automatic updates to visually composable forms based on entity changes)") boolean visuallyComposable,
			@CliOption(key = "disallowedOperations", mandatory = false, help = "A comma separated list of operations (only create, update, delete allowed) that should not be generated in the view") String disallowedOperations) {
		if (view.getSimpleTypeName().equalsIgnoreCase(
				entity.getSimpleTypeName())) {
			logger.warning("View class name needs to be different from the class name of the form backing object (suggestion: '"
					+ entity.getSimpleTypeName() + "View')");
			return;
		}

		Set<String> disallowedOperationSet = new HashSet<String>();
		if (!"".equals(disallowedOperations)) {
			for (String operation : StringUtils
					.commaDelimitedListToSet(disallowedOperations)) {
				if (!("create".equals(operation) || "update".equals(operation) || "delete"
						.equals(operation))) {
					logger.warning("-disallowedOperations options can only contain 'create', 'update', 'delete': -disallowedOperations update,delete");
					return;
				}
				disallowedOperationSet.add(operation.toLowerCase());
			}
		}

		operations.createEntityView(view, visuallyComposable, entity,
				baseClass, disallowedOperationSet);
	}

	@CliCommand(value = "vaadin widgetset create", help = "Create a widgetset and configure GWT compilation for the project")
	public void vaadinCreateWidgetset(
			@CliOption(key = "browsers", mandatory = false, help = "GWT permutations (user agents) to compile the widgetset for, all supported browsers if omitted") String browsers)
			throws IOException {
		operations.createWidgetset(browsers);
	}

	@CliCommand(value = "vaadin widgetset update", help = "Update and compile a widgetset based on Vaadin add-on dependencies for the project")
	public void vaadinUpdateWidgetset() throws IOException {
		operations.updateWidgetset();
	}

	@CliCommand(value = "vaadin widgetset compile", help = "Compile a Vaadin widgetset created with \"vaadin widgetset create\"")
	public void vaadinCompileWidgetset() throws IOException {
		operations.compileWidgetset();
	}

	@CliCommand(value = "vaadin addon install", help = "Add a dependency to a Vaadin add-on")
	public void vaadinInstallAddon(
			@CliOption(key = "artifactId", mandatory = true, help = "Maven artifact id for the add-on") VaadinAddonArtifactId artifactId,
			@CliOption(key = "groupId", mandatory = false, help = "Maven group id for the add-on") String groupId,
			@CliOption(key = "version", mandatory = false, help = "Maven artifact version number for the add-on or LATEST, defaults to the latest version number currently available") String version) {
		addonOperations.installAddon(groupId, artifactId, version);
	}

}