package com.vaadin.spring.roo.addon;

import java.io.IOException;
import java.util.Set;

import org.springframework.roo.model.JavaPackage;
import org.springframework.roo.model.JavaSymbolName;
import org.springframework.roo.model.JavaType;

import com.vaadin.spring.roo.addon.annotations.RooVaadinEntityView;

/**
 * Interface of Vaadin Roo commands that are available via the Roo shell.
 *
 * @since 1.1.0-M1
 */
public interface VaadinOperations {

	/**
	 * Default name for the abstract base class of entity views.
	 */
	public static final String ABSTRACT_ENTITY_VIEW_CLASS = "AbstractEntityView";

	/**
	 * Checks whether the shell is in a valid project.
	 *
	 * @return
	 */
	boolean isProjectAvailable();

	/**
	 * Checks if Vaadin already been added to a project.
	 *
	 * @return true if Vaadin is configured in the project
	 */
	boolean isVaadinSetup();

	/**
	 * Checks if the persistence has been setup for the project.
	 *
	 * @return true if persistence.xml exists
	 */
	boolean isPersistenceSetup();

	/**
	 * Checks if a widgetset has been configured for the project.
	 *
	 * @return true if widgetset exists in the project
	 */
	public boolean isWidgetsetSetup();

	/**
	 * Installs Vaadin to a project.
	 *
	 * Adds a dependency to Vaadin.
	 *
	 * If the application package is specified (not null), creates web.xml, a
	 * Vaadin application, a window, a main view and a trivial theme. If some
	 * class already exists, they are not modified.
	 *
	 * @param applicationPackage
	 *            the package in which to create an application related classes
	 *            or null for none
	 * @param baseName
	 *            optional beginning of class name for the classes created -
	 *            generated from package name if null; no classes are generated
	 *            if package path is null
	 * @param themeName
	 *            optional name of the theme to create, deduced from baseName if
	 *            not given
	 * @param appNameHtml
	 *            HTML string to show as the application name in the entity
	 *            manager view
	 * @param useJpaContainer
	 *            if true, JPAContainer is used where applicable instead of an
	 *            in-memory container and explicit JPA operations
	 */
	void vaadinSetup(JavaPackage applicationPackage, JavaSymbolName baseName,
			JavaSymbolName themeName, String appNameHtml,
			boolean useJpaContainer);

	/**
	 * Generates Vaadin CRUD view for all the entities in the project that don't
	 * have one.
	 *
	 * The views are marked with the {@link RooVaadinEntityView} annotation and
	 * support all CRUD operations.
	 *
	 * The entity editor forms can optionally be editable with the Vaadin Visual
	 * Editor.
	 *
	 * @param viewPackage
	 *            the package in which the views should be created
	 * @param visuallyComposable
	 *            true for the forms to be editable with the visual editor,
	 *            false to use an automatic forms constructed at runtime
	 * @param baseClass
	 *            the class name of the abstract entity view base class to
	 *            create or use (optional, using default name and view package
	 *            if null)
	 */
	void generateAll(JavaPackage viewPackage, boolean visuallyComposable,
			JavaType baseClass);

	/**
	 * Generates Vaadin CRUD view for an entity class.
	 *
	 * The view is marked with the {@link RooVaadinEntityView} annotation and
	 * support all CRUD operations except those in disallowedOperations (allowed
	 * values are "create", "update" and "delete").
	 *
	 * The entity editor form can optionally be editable with the Vaadin Visual
	 * Editor.
	 *
	 * @param view
	 *            the class name of the entity view to create (required)
	 * @param visuallyComposable
	 *            true for the form to be editable with the visual editor, false
	 *            to use an automatic form constructed at runtime
	 * @param entity
	 *            the entity class that the view shows and manages (required)
	 * @param baseClass
	 *            the class name of the abstract entity view base class to
	 *            create or use (optional, using default name and view package
	 *            if null)
	 * @param disallowedOperations
	 *            set of disallowed operations (required, but can be empty)
	 */
	void createEntityView(JavaType view, boolean visuallyComposable,
			JavaType entity, JavaType baseClass,
			Set<String> disallowedOperations);

	/**
	 * Creates a Vaadin widgetset and configures any related settings (web.xml,
	 * POM plugins).
	 *
	 * @param browsers
	 *            permutations to use in GWT comma separated user agent list
	 *            format, null for all permutations
	 */
	void createWidgetset(String browsers) throws IOException;

	/**
	 * Updates a Vaadin widgetset in the project based on the Vaadin add-on
	 * dependencies of the project.
	 */
	void updateWidgetset() throws IOException;

	/**
	 * Compiles a Vaadin widgetset using the GWT Maven plugin.
	 */
	void compileWidgetset() throws IOException;

}