package com.vaadin.spring.roo.addon;

import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.apache.felix.scr.annotations.Component;
import org.apache.felix.scr.annotations.Reference;
import org.apache.felix.scr.annotations.Service;
import org.osgi.service.component.ComponentContext;
import org.springframework.roo.addon.entity.EntityMetadata;
import org.springframework.roo.addon.entity.RooEntity;
import org.springframework.roo.classpath.PhysicalTypeIdentifier;
import org.springframework.roo.classpath.TypeLocationService;
import org.springframework.roo.classpath.details.ClassOrInterfaceTypeDetails;
import org.springframework.roo.classpath.scanner.MemberDetailsScanner;
import org.springframework.roo.metadata.MetadataDependencyRegistry;
import org.springframework.roo.metadata.MetadataService;
import org.springframework.roo.model.JavaPackage;
import org.springframework.roo.model.JavaSymbolName;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.process.manager.ActiveProcessManager;
import org.springframework.roo.process.manager.FileManager;
import org.springframework.roo.process.manager.ProcessManager;
import org.springframework.roo.project.Dependency;
import org.springframework.roo.project.Path;
import org.springframework.roo.project.PathResolver;
import org.springframework.roo.project.Plugin;
import org.springframework.roo.project.ProjectMetadata;
import org.springframework.roo.project.ProjectOperations;
import org.springframework.roo.project.ProjectType;
import org.springframework.roo.project.Repository;
import org.springframework.roo.support.util.Assert;
import org.springframework.roo.support.util.StringUtils;
import org.springframework.roo.support.util.TemplateUtils;
import org.springframework.roo.support.util.WebXmlUtils;
import org.springframework.roo.support.util.XmlElementBuilder;
import org.springframework.roo.support.util.XmlUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;

import com.vaadin.spring.roo.addon.abstractentityview.VaadinAbstractEntityViewMetadata;
import com.vaadin.spring.roo.addon.annotations.RooVaadinAbstractEntityView;
import com.vaadin.spring.roo.addon.entityform.VaadinClassNames;
import com.vaadin.spring.roo.addon.entityform.VisuallyComposableFormCreationHelper;
import com.vaadin.spring.roo.addon.entityform.VisuallyComposableFormMethodBuilder;
import com.vaadin.spring.roo.addon.entitymanagerview.VaadinEntityManagerViewMetadata;
import com.vaadin.spring.roo.addon.entityview.VaadinEntityViewMetadata;

/**
 * Implementation of Vaadin Roo commands that are available via the Roo shell.
 *
 * @since 1.1.0M1
 */
@Component
@Service
public class VaadinOperationsImpl implements VaadinOperations {

	private static final String ABSTRACT_ENTITY_VIEW_CLASS = "AbstractEntityView";
	private static final String AUTOMATIC_ENTITY_FORM_CLASS = "AutomaticEntityForm";

	private static final String VAADIN_ARTIFACT_ID = "vaadin";
	private static final String VAADIN_GROUP_ID = "com.vaadin";

	private static final String OPEN_ENTITYMANAGER_IN_VIEW_FILTER_NAME = "Spring OpenEntityManagerInViewFilter";

	private static final String ENTITY_TABLE_COLUMN_GENERATOR_CLASS = "EntityTableColumnGenerator";

	private static final String ENTITY_EDITOR_INTERFACE = "EntityEditor";

	// for JPAContainer
	public static final String ENTITY_PROVIDER_UTIL_CLASS = "EntityProviderUtil";

	private static Logger logger = Logger.getLogger(VaadinOperations.class
			.getName());

	@Reference
	private FileManager fileManager;
	@Reference
	private PathResolver pathResolver;
	@Reference
	private MetadataService metadataService;
	@Reference
	private MemberDetailsScanner memberDetailsScanner;
	@Reference
	private ProjectOperations projectOperations;
	@Reference
	private TypeLocationService typeLocationService;
	@Reference
	private MetadataDependencyRegistry dependencyRegistry;

	// for maven execution
	@Reference
	private ProcessManager processManager;

	private ComponentContext context;

	protected void activate(ComponentContext context) {
		this.context = context;
	}

	public boolean isProjectAvailable() {
		ProjectMetadata projectMetadata = (ProjectMetadata) metadataService
				.get(ProjectMetadata.getProjectIdentifier());
		return projectMetadata != null;
	}

	public boolean isVaadinSetup() {
		ProjectMetadata projectMetadata = (ProjectMetadata) metadataService
				.get(ProjectMetadata.getProjectIdentifier());
		Set<Dependency> dependencies = projectMetadata.getDependencies();
		for (Dependency d : dependencies) {
			if (VAADIN_ARTIFACT_ID.equals(d.getArtifactId())
					&& VAADIN_GROUP_ID.equals(d.getGroupId())) {
				return true;
			}
		}
		return false;
	}

	public boolean isPersistenceSetup() {
		return pathResolver != null
				&& fileManager.exists(pathResolver.getIdentifier(
						Path.SRC_MAIN_RESOURCES, "META-INF/persistence.xml"));
	}

	public boolean isWidgetsetSetup() {
		// find *Widgetset.gwt.xml in project source directory
		// inefficient but good enough
		String antPath = pathResolver.getRoot(Path.SRC_MAIN_JAVA)
				+ "/**/*Widgetset.gwt.xml";
		return !fileManager.findMatchingAntPath(antPath).isEmpty();
	}

	public void vaadinSetup(JavaPackage applicationPackage,
			JavaSymbolName baseName, JavaSymbolName themeName,
			String appNameHtml, boolean useJpaContainer) {
		logger.info("Performing vaadin setup");

		// Parse the configuration.xml file
		Element configuration = XmlUtils.getConfiguration(getClass());

		addVaadinDependencies(configuration);
		addBeanValidationDependencies(configuration);

		if (applicationPackage != null) {

			String pkg = applicationPackage.getFullyQualifiedPackageName();

			String baseNameString;
			if (baseName == null) {
				baseNameString = generateApplicationBaseName(applicationPackage);
			} else {
				baseNameString = baseName.getSymbolNameCapitalisedFirstLetter();
			}

			String themeNameString;
			if (themeName != null) {
				themeNameString = themeName.getSymbolName();
			} else {
				themeNameString = baseNameString.toLowerCase();
			}

			// create theme
			createTheme(themeNameString);

			// create/modify web.xml
			copyWebXml();
			manageWebXml(pkg + "." + getApplicationClassName(baseNameString));

			// make this a WAR project
			projectOperations.updateProjectType(ProjectType.WAR);

			// create a Vaadin application and any related classes
			createApplication(applicationPackage, baseNameString,
					themeNameString, appNameHtml, useJpaContainer);
		}
	}

	private void addVaadinDependencies(Element configuration) {
		List<Element> vaadinDependencies = XmlUtils.findElements(
				"/configuration/vaadin/dependencies/dependency", configuration);
		List<Dependency> dependencies = new ArrayList<Dependency>();
		for (Element dependency : vaadinDependencies) {
			dependencies.add(new Dependency(dependency));
		}
		projectOperations.addDependencies(dependencies);


		List<Element> vaadinRepositories = XmlUtils.findElements(
				"/configuration/vaadin/repositories/repository", configuration);

		List<Repository> repositories = new ArrayList<Repository>();
		for (Element repositoryElement : vaadinRepositories) {
			repositories.add(new Repository(repositoryElement));
		}
		projectOperations.addRepositories(repositories);
	}

	private void addBeanValidationDependencies(Element configuration) {
		List<Element> beanValidationDependencies = XmlUtils.findElements(
				"/configuration/beanvalidation/dependencies/dependency",
				configuration);
		List<Dependency> dependencies = new ArrayList<Dependency>();
		for (Element dependency : beanValidationDependencies) {
			dependencies.add(new Dependency(dependency));
		}
		projectOperations.addDependencies(dependencies);
	}

	private static void addJpaContainerDependencies(Element configuration,
			ProjectOperations projectOperations) {
		List<Element> jpaContainerDependencies = XmlUtils.findElements(
				"/configuration/jpacontainer/dependencies/dependency",
				configuration);
		List<Dependency> dependencies = new ArrayList<Dependency>();
		for (Element dependency : jpaContainerDependencies) {
			dependencies.add(new Dependency(dependency));
		}
		projectOperations.addDependencies(dependencies);
	}

	private void createTheme(String themeName) {
		Assert.notNull(pathResolver, "Path resolver not found");

		String themePath = pathResolver.getIdentifier(Path.SRC_MAIN_WEBAPP,
				"VAADIN/themes/" + themeName);
		if (!fileManager.exists(themePath)) {
			VaadinRooUtils.copyDirectoryContents(fileManager, context, "theme",
					themePath);
			VaadinRooUtils.copyDirectoryContents(fileManager, context,
					"theme/img", themePath + "/img");
			VaadinRooUtils.copyDirectoryContents(fileManager, context,
					"theme/entityviews", themePath + "/entityviews");
		} else {
			logger.info("Theme " + themeName + " already exists");
		}
	}

	private void copyWebXml() {
		ProjectMetadata projectMetadata = (ProjectMetadata) metadataService
				.get(ProjectMetadata.getProjectIdentifier());
		Assert.notNull(projectMetadata, "Project metadata required");
		Assert.notNull(pathResolver, "Path resolver required");

		if (fileManager.exists(pathResolver.getIdentifier(Path.SRC_MAIN_WEBAPP,
				"WEB-INF/web.xml"))) {
			// file exists, so nothing to do
			return;
		}

		InputStream templateInputStream = TemplateUtils.getTemplate(getClass(),
				"web-template.xml");
		Document webXml;
		try {
			webXml = XmlUtils.getDocumentBuilder().parse(templateInputStream);
		} catch (Exception ex) {
			throw new IllegalStateException(ex);
		}

		WebXmlUtils.setDisplayName(projectMetadata.getProjectName(), webXml,
				null);
		WebXmlUtils.setDescription(
				"Roo generated " + projectMetadata.getProjectName()
						+ " application", webXml, null);

		VaadinRooUtils
				.writeXmlToDiskIfNecessary(fileManager,
						pathResolver.getIdentifier(Path.SRC_MAIN_WEBAPP,
								"WEB-INF/web.xml"), webXml);

		fileManager.scan();
	}

	private void manageWebXml(String applicationName) {
		ProjectMetadata projectMetadata = (ProjectMetadata) metadataService
				.get(ProjectMetadata.getProjectIdentifier());
		Assert.notNull(projectMetadata, "Project metadata required");
		Assert.notNull(pathResolver, "Path resolver required");

		// Verify that the web.xml already exists
		String webXmlFile = pathResolver.getIdentifier(Path.SRC_MAIN_WEBAPP,
				"WEB-INF/web.xml");
		Assert.isTrue(fileManager.exists(webXmlFile), "'" + webXmlFile
				+ "' does not exist");

		Document webXml;
		try {
			webXml = XmlUtils.getDocumentBuilder().parse(
					fileManager.getInputStream(webXmlFile));
		} catch (Exception ex) {
			throw new IllegalStateException(ex);
		}

		WebXmlUtils.addContextParam(new WebXmlUtils.WebXmlParam(
				"productionMode", "false"), webXml, "Vaadin production mode");

		// can have multiple application context files separated by spaces
		WebXmlUtils.addContextParam(new WebXmlUtils.WebXmlParam(
				"contextConfigLocation",
				"classpath*:META-INF/spring/applicationContext*.xml"), webXml,
				null);

		// optionally make the EntityManager available in each request
		if (fileManager.exists(pathResolver.getIdentifier(
				Path.SRC_MAIN_RESOURCES, "META-INF/persistence.xml"))) {
			WebXmlUtils
					.addFilterAtPosition(
							WebXmlUtils.FilterPosition.FIRST,
							null,
							null,
							OPEN_ENTITYMANAGER_IN_VIEW_FILTER_NAME,
							"org.springframework.orm.jpa.support.OpenEntityManagerInViewFilter",
							"/*", webXml, null);
		}

		// load the Spring context from the appropriate file
		WebXmlUtils
				.addListener(
						"org.springframework.web.context.ContextLoaderListener",
						webXml,
						"Creates the Spring Container shared by all Servlets and Filters");

		// for now, register servlet for /*
		WebXmlUtils.addServlet(projectMetadata.getProjectName()
				+ " Vaadin Application Servlet",
				VaadinClassNames.APPLICATION_SERVLET_CLASS, "/*", null, webXml,
				null, new WebXmlUtils.WebXmlParam("application",
						applicationName));

		VaadinRooUtils.writeXmlToDiskIfNecessary(fileManager, webXmlFile,
				webXml);
	}

	/**
	 * Add the widgetset parameter for the Vaadin servlet, or modify it to refer
	 * to the given widgetset if the parameter is already present.
	 *
	 * @param widgetsetName
	 */
	public void addWidgetsetParamToWebXml(String widgetsetName) {
		ProjectMetadata projectMetadata = (ProjectMetadata) metadataService
				.get(ProjectMetadata.getProjectIdentifier());
		Assert.notNull(projectMetadata, "Project metadata required");
		Assert.notNull(pathResolver, "Path resolver required");

		// Verify that the web.xml already exists
		String webXmlFile = pathResolver.getIdentifier(Path.SRC_MAIN_WEBAPP,
				"WEB-INF/web.xml");
		Assert.isTrue(fileManager.exists(webXmlFile), "'" + webXmlFile
				+ "' does not exist");

		Document webXml;
		try {
			webXml = XmlUtils.getDocumentBuilder().parse(
					fileManager.getInputStream(webXmlFile));
		} catch (Exception ex) {
			throw new IllegalStateException(ex);
		}

		Element servlet = XmlUtils.findFirstElement(
				"/web-app/servlet[servlet-name = '"
						+ projectMetadata.getProjectName()
						+ " Vaadin Application Servlet" + "']",
				webXml.getDocumentElement());

		// update or create init-param element
		Element widgetsetInitParam = XmlUtils.findFirstElement(
				"init-param[param-name = 'widgetset']", servlet);
		if (widgetsetInitParam != null) {
			// update existing widgetset parameter
			Element widgetsetParamValue = XmlUtils.findFirstElement(
					"param-value", widgetsetInitParam);
			widgetsetParamValue.setTextContent(widgetsetName);
		} else {
			// append init-param in the servlet
			Element initParam = new XmlElementBuilder("init-param", webXml)
					.addChild(
							new XmlElementBuilder("param-name", webXml)
									.setText("widgetset").build())
					.addChild(
							new XmlElementBuilder("param-value", webXml)
									.setText(widgetsetName).build()).build();
			servlet.appendChild(initParam);
		}

		VaadinRooUtils.writeXmlToDiskIfNecessary(fileManager, webXmlFile,
				webXml);
	}

	public void createApplication(JavaPackage applicationPackage,
			String baseName, String themeName, String appNameHtml,
			boolean useJpaContainer) {
		Assert.notNull(applicationPackage, "Application package required");
		Assert.notNull(typeLocationService, "Type location service required");
		Assert.notNull(pathResolver, "Path resolver required");

		String pkg = applicationPackage.getFullyQualifiedPackageName();

		String applicationName = getApplicationClassName(baseName);
		String windowName = baseName + "Window";
		String viewName = baseName + "EntityManagerView";
		String abstractEntityViewName = ABSTRACT_ENTITY_VIEW_CLASS;

		String annotationParameters = "";
		if (useJpaContainer) {
			annotationParameters = "useJpaContainer = true";
		}

		Map<String, String> substitutions = new HashMap<String, String>();
		substitutions.put("__TOP_LEVEL_PACKAGE__", pkg);
		substitutions.put("__APPLICATION_CLASS_NAME__", applicationName);
		substitutions.put("__WINDOW_CLASS_NAME__", windowName);
		substitutions.put("__VIEW_CLASS_NAME__", viewName);
		substitutions.put("__THEME_NAME__", themeName);
		substitutions.put("__APP_NAME_HTML__", appNameHtml);
		substitutions.put("__ABSTRACT_ENTITY_VIEW_PACKAGE__", pkg);
		substitutions.put("__ABSTRACT_ENTITY_VIEW_CLASS__",
				abstractEntityViewName);
		substitutions.put("__ABSTRACT_ENTITY_VIEW_ANNOTATION_PARAMETERS__",
				annotationParameters);

		createClassFromTemplate(pkg + "." + applicationName,
				"Application.java", substitutions);
		createClassFromTemplate(pkg + "." + windowName, "Window.java",
				substitutions);
		createClassFromTemplate(pkg + "." + viewName, "EntityManagerView.java",
				substitutions);
		createClassFromTemplate(pkg + "." + abstractEntityViewName,
				"AbstractEntityView.java", substitutions);
		createClassFromTemplate(pkg + "." + AUTOMATIC_ENTITY_FORM_CLASS,
				"AutomaticEntityForm.java", substitutions);

		// UI table column generator
		createClassFromTemplate(
				pkg + "." + ENTITY_TABLE_COLUMN_GENERATOR_CLASS,
				"EntityTableColumnGenerator.java", substitutions);

		// entity editor interface
		createClassFromTemplate(pkg + "." + ENTITY_EDITOR_INTERFACE,
				"EntityEditor.java", substitutions);
	}

	private void createClassFromTemplate(String className,
			String templateBaseName, Map<String, String> substitutions) {
		JavaType classToCreate = new JavaType(className);
		String classResourceIdentifier = typeLocationService
				.getPhysicalLocationCanonicalPath(classToCreate,
						Path.SRC_MAIN_JAVA);
		VaadinRooUtils.installFromTemplateIfNeeded(fileManager,
				classResourceIdentifier, templateBaseName, substitutions);
	}

	private String getApplicationClassName(String baseName) {
		return baseName + "Application";
	}

	/**
	 * Generates a base name (beginning of the class name) for classes to be
	 * created based on the last part of a package path.
	 *
	 * @param applicationPackage
	 * @return correctly capitalized beginning of a class name (not qualified
	 *         with the package name)
	 */
	private String generateApplicationBaseName(JavaPackage applicationPackage) {
		String pkg = applicationPackage.getFullyQualifiedPackageName();
		return StringUtils.capitalize(pkg.substring(pkg.lastIndexOf(".") + 1));
	}

	public void generateAll(JavaPackage viewPackage,
			boolean visuallyComposable, JavaType baseClass) {
		Assert.notNull(pathResolver, "Path resolver required");

		baseClass = prepareCreateEntityViews(baseClass);

		Set<ClassOrInterfaceTypeDetails> cids = typeLocationService
				.findClassesOrInterfaceDetailsWithAnnotation(new JavaType(
						RooEntity.class.getName()));
		for (ClassOrInterfaceTypeDetails cid : cids) {
			if (Modifier.isAbstract(cid.getModifier())) {
				continue;
			}

			JavaType javaType = cid.getName();
			Path path = PhysicalTypeIdentifier.getPath(cid
					.getDeclaredByMetadataId());
			EntityMetadata entityMetadata = (EntityMetadata) metadataService
					.get(EntityMetadata.createIdentifier(javaType, path));

			if (entityMetadata == null || (!entityMetadata.isValid())) {
				continue;
			}

			// Check to see if this entity metadata has a web scaffold metadata
			// listening to it
			String downstreamEntityViewMetadataId = VaadinEntityViewMetadata
					.createIdentifier(javaType, path);
			if (dependencyRegistry.getDownstream(entityMetadata.getId())
					.contains(downstreamEntityViewMetadataId)) {
				// There is already a view for this entity
				// TODO check if valid?
				continue;
			}

			// To get here, there is no view, so add one
			JavaType view = new JavaType(
					viewPackage.getFullyQualifiedPackageName() + "."
							+ javaType.getSimpleTypeName() + "View");
			JavaType form = new JavaType(
					viewPackage.getFullyQualifiedPackageName() + "."
							+ javaType.getSimpleTypeName() + "Form");
			JavaType entity = javaType;
			Set<String> disallowedOperations = new HashSet<String>();
			createEntityView(view, form, visuallyComposable, entity, baseClass,
					disallowedOperations);
		}
		return;
	}

	private List<String> findValidAbstractEntityViewMids() {
		String providesType = VaadinAbstractEntityViewMetadata
				.getMetadataIdentiferString();
		List<String> allMids = VaadinRooUtils.scanPotentialTypeMids(
				metadataService, fileManager, providesType);

		List<String> entityViewMids = new ArrayList<String>();
		for (String mid : allMids) {
			if (VaadinAbstractEntityViewMetadata.isValid(mid)
					&& metadataService.get(mid) instanceof VaadinAbstractEntityViewMetadata) {
				entityViewMids.add(mid);
			}
		}

		return entityViewMids;
	}

	private List<String> findValidEntityManagerViewMids() {
		String providesType = VaadinEntityManagerViewMetadata
				.getMetadataIdentiferString();
		List<String> allMids = VaadinRooUtils.scanPotentialTypeMids(
				metadataService, fileManager, providesType);

		List<String> entityViewMids = new ArrayList<String>();
		for (String mid : allMids) {
			if (VaadinEntityManagerViewMetadata.isValid(mid)
					&& metadataService.get(mid) instanceof VaadinEntityManagerViewMetadata) {
				entityViewMids.add(mid);
			}
		}

		return entityViewMids;
	}

	public void createEntityView(JavaType view, boolean visuallyComposable,
			JavaType entity, JavaType baseClass,
			Set<String> disallowedOperations) {
		Assert.notNull(pathResolver, "Path resolver required");
		Assert.notNull(typeLocationService, "Type location service required");

		Assert.notNull(view, "View Java Type required");
		Assert.notNull(entity, "Entity Java Type required");
		Assert.notNull(disallowedOperations,
				"Set of disallowed operations required");

		baseClass = prepareCreateEntityViews(baseClass);
		JavaType form = new JavaType(view.getPackage()
				.getFullyQualifiedPackageName()
				+ "."
				+ entity.getSimpleTypeName() + "Form");
		createEntityView(view, form, visuallyComposable, entity, baseClass,
				disallowedOperations);
	}

	/**
	 * Prepare to create entity views: set up necessary dependencies etc.
	 *
	 * Changes (dependencies, helper classes etc.) specific to the use of
	 * JPAContainer are triggered by {@link VaadinAbstractEntityViewMetadata} by
	 * calling {@link #setupJpaContainer()}.
	 *
	 * @param baseClass
	 *            base class (AbstractEntityView) or null to find an unambiguous
	 *            base class that implements {@link RooVaadinAbstractEntityView}
	 * @return base class, looked up based on annotations if not given
	 *
	 * @throws IllegalArgumentException
	 *             if an unambiguous base class is not found or the parameters
	 *             are otherwise incorrect
	 */
	private JavaType prepareCreateEntityViews(JavaType baseClass) {
		Assert.notNull(pathResolver, "Path resolver required");
		Assert.notNull(typeLocationService, "Type location service required");

		// find base class if not given
		if (baseClass == null) {
			// if no explicitly specified base class, find classes with the
			// @RooVaadinAbstractEntityView annotation using metadata
			List<String> allMids = findValidAbstractEntityViewMids();
			if (allMids.size() == 0) {
				throw new IllegalArgumentException(
						"No entity view base class given and none found in the project.");
			} else if (allMids.size() > 1) {
				throw new IllegalArgumentException(
						"No entity view base class given and multiple candidates found in the project.");
			} else {
				// based on found annotation/metadata
				baseClass = VaadinAbstractEntityViewMetadata
						.getJavaType(allMids.get(0));
			}
		}

		// Parse the configuration.xml file
		Element configuration = XmlUtils.getConfiguration(getClass());
		addBeanValidationDependencies(configuration);

		return baseClass;
	}

	/**
	 * Internal creation of an entity view, not setting up external classes and
	 * dependencies etc.
	 *
	 * @param view
	 *            the view class to create (should not exist)
	 * @param form
	 *            the form class to create (should not exist)
	 * @param visuallyComposable
	 *            true for the form to be editable with the visual editor, false
	 *            to use an automatic form constructed at runtime
	 * @param entity
	 *            the entity type to manage in the view
	 * @param viewBaseClass
	 *            non-null base class (AbstractEntityView)
	 * @param disallowedOperations
	 * @param useJpaContainer
	 */
	private void createEntityView(JavaType view, JavaType form,
			boolean visuallyComposable, JavaType entity,
			JavaType viewBaseClass, Set<String> disallowedOperations) {
		Assert.notNull(pathResolver, "Path resolver required");
		Assert.notNull(typeLocationService, "Type location service required");

		Assert.notNull(view, "View Java Type required");
		Assert.notNull(entity, "Entity Java Type required");
		Assert.notNull(disallowedOperations,
				"Set of disallowed operations required");

		String annotationParameters = "formBackingObject = "
				+ entity.getFullyQualifiedTypeName() + ".class";
		for (String operation : disallowedOperations) {
			annotationParameters = annotationParameters + ", " + operation
					+ " = false";
		}

		Map<String, String> substitutions = new HashMap<String, String>();
		substitutions.put("__ENTITY_PACKAGE__", entity.getPackage()
				.getFullyQualifiedPackageName());
		substitutions.put("__ENTITY_CLASS__", entity.getSimpleTypeName());
		substitutions.put("__ABSTRACT_ENTITY_VIEW_PACKAGE__", viewBaseClass
				.getPackage().getFullyQualifiedPackageName());
		substitutions.put("__ABSTRACT_ENTITY_VIEW_CLASS__",
				viewBaseClass.getSimpleTypeName());

		substitutions.put(
				"__ABSTRACT_ENTITY_VIEW_IMPORT__",
				createImport(view, viewBaseClass.getPackage(),
						viewBaseClass.getSimpleTypeName()));
		substitutions.put(
				"__AUTOMATIC_ENTITY_FORM_IMPORT__",
				createImport(view, viewBaseClass.getPackage(),
						AUTOMATIC_ENTITY_FORM_CLASS));
		substitutions.put(
				"__ENTITY_EDITOR_IMPORT__",
				createImport(view, viewBaseClass.getPackage(),
						ENTITY_EDITOR_INTERFACE));

		substitutions.put("__ENTITY_VIEW_PACKAGE__", view.getPackage()
				.getFullyQualifiedPackageName());
		substitutions.put("__ENTITY_VIEW_CLASS__", view.getSimpleTypeName());
		substitutions.put("__ENTITY_VIEW_ANNOTATION_PARAMETERS__",
				annotationParameters);

		String formAnnotationParameters = "formBackingObject = "
				+ entity.getFullyQualifiedTypeName() + ".class";

		substitutions.put("__ENTITY_FORM_CLASS__", form.getSimpleTypeName());
		substitutions.put("__ENTITY_FORM_ANNOTATION_PARAMETERS__",
				formAnnotationParameters);

		String resourceIdentifier = typeLocationService
				.getPhysicalLocationCanonicalPath(view, Path.SRC_MAIN_JAVA);

		VaadinRooUtils.installFromTemplateIfNeeded(fileManager,
				resourceIdentifier, "EntityView.java", substitutions);

		String formResourceIdentifier = typeLocationService
				.getPhysicalLocationCanonicalPath(form, Path.SRC_MAIN_JAVA);

		if (visuallyComposable) {
			VisuallyComposableFormCreationHelper helper = new VisuallyComposableFormCreationHelper(
					metadataService, memberDetailsScanner, entity,
					VisuallyComposableFormMethodBuilder.INDENT,
					VisuallyComposableFormMethodBuilder.FIELD_LAYOUT_NAME);
			substitutions.put("__FIELD_INSERT_POINT__",
					helper.getFieldDeclarations());
			substitutions.put("__FIELD_CREATION_INSERT_POINT__",
					helper.getFieldCreationStatements());
			VaadinRooUtils.installFromTemplateIfNeeded(fileManager,
					formResourceIdentifier,
					"ConcreteVisuallyComposableEntityForm.java", substitutions);
		} else {
			VaadinRooUtils.installFromTemplateIfNeeded(fileManager,
					formResourceIdentifier, "ConcreteAutomaticEntityForm.java",
					substitutions);
		}
	}

	/**
	 * Create an import string for a class in the package of viewBaseClass.
	 *
	 * If importPackage is the same as the package of viewClass, an empty string
	 * is returned.
	 *
	 * @param viewClass
	 * @param importPackage
	 * @param className
	 * @return
	 */
	private String createImport(JavaType viewClass, JavaPackage importPackage,
			String className) {
		if (!importPackage.equals(viewClass.getPackage())) {
			return "import " + importPackage.getFullyQualifiedPackageName()
					+ "." + className + ";";
		} else {
			return "";
		}
	}

	/**
	 * If necessary, set up the project to enable using JPAContainer: add
	 * dependencies, helper classes etc.
	 *
	 * @param baseClass
	 * @param projectOperations
	 * @param fileManager
	 * @param typeLocationService
	 */
	public static void setupJpaContainer(JavaType baseClass,
			ProjectOperations projectOperations, FileManager fileManager,
			TypeLocationService typeLocationService) {
		// Parse the configuration.xml file
		Element configuration = XmlUtils
				.getConfiguration(VaadinOperationsImpl.class);
		addJpaContainerDependencies(configuration, projectOperations);

		// create helper class for JPAContainer
		String pkg = baseClass.getPackage().getFullyQualifiedPackageName();
		Map<String, String> substitutions = Collections.singletonMap("__ABSTRACT_ENTITY_VIEW_PACKAGE__", pkg);
		JavaType classToCreate = new JavaType(pkg + "." + ENTITY_PROVIDER_UTIL_CLASS);
		String classResourceIdentifier = typeLocationService
				.getPhysicalLocationCanonicalPath(classToCreate,
						Path.SRC_MAIN_JAVA);
		VaadinRooUtils.installFromTemplateIfNeeded(fileManager,
				classResourceIdentifier, "EntityProviderUtil.java", substitutions);
	}

	/**
	 * Configure GWT compilation for the project to support widgetset
	 * compilation.
	 *
	 * @param projectOperations
	 */
	public static void setupGwtCompilation(ProjectOperations projectOperations) {
		Element configuration = XmlUtils
				.getConfiguration(VaadinOperationsImpl.class);

		// Add dependencies
		List<Element> gwtDependencies = XmlUtils.findElements(
				"/configuration/gwt/dependencies/dependency", configuration);
		List<Dependency> dependencies = new ArrayList<Dependency>();
		for (Element dependency : gwtDependencies) {
			dependencies.add(new Dependency(dependency));
		}
		projectOperations.addDependencies(dependencies);

		// Add POM plugin
		// TODO this should go to dependency management (no auto-execution), but
		// no API in Roo for modifying it?
		List<Element> plugins = XmlUtils.findElements(
				"/configuration/gwt/plugins/plugin", configuration);
		for (Element plugin : plugins) {
			projectOperations.addBuildPlugin(new Plugin(plugin));
		}
	}

	public void createWidgetset(String browsers) throws IOException {
		// configure pom.xml for widgetset updates and compilation
		setupGwtCompilation(projectOperations);

		// find the entity manager view class location and name, use them for
		// the widgetset location and name determination

		JavaType entityManagerViewClass;
		List<String> allMids = findValidEntityManagerViewMids();
		if (allMids.size() == 0) {
			throw new IllegalArgumentException(
					"No entity manager base class found in the project. Please execute \"vaadin setup\" first.");
		} else {
			// based on found annotation/metadata
			entityManagerViewClass = VaadinEntityManagerViewMetadata
					.getJavaType(allMids.get(0));
		}

		// create widgetset file

		String widgetsetPackage = entityManagerViewClass.getPackage()
				.getFullyQualifiedPackageName();
		// if the user has given an application name, it is probably still the
		// beginning of the class name of the entity manager view
		String applicationBaseName = entityManagerViewClass.getSimpleTypeName()
				.replaceAll("EntityManagerView$", "");
		String widgetsetName = applicationBaseName + "Widgetset";
		String fullyQualifiedWidgetsetName = "".equals(widgetsetPackage) ? widgetsetName
				: (widgetsetPackage + "." + widgetsetName);

		String relativePath = fullyQualifiedWidgetsetName.replace('.',
				File.separatorChar) + ".gwt.xml";
		String resourceIdentifier = pathResolver.getIdentifier(
				Path.SRC_MAIN_JAVA, relativePath);

		String userAgentLine;
		if (browsers == null) {
			userAgentLine = "<!-- <set-property name=\"user.agent\" value=\"gecko1_8\"/> -->";
		} else {
			userAgentLine = "<set-property name=\"user.agent\" value=\""
					+ browsers + "\"/>";
		}

		VaadinRooUtils.installFromTemplateIfNeeded(fileManager,
				resourceIdentifier, "Widgetset.gwt.xml",
				Collections.singletonMap("__USER_AGENT_LINE__", userAgentLine));

		// update web.xml (reference to widgetset)
		addWidgetsetParamToWebXml(fullyQualifiedWidgetsetName);

		updateWidgetset();
	}

	public void updateWidgetset() throws IOException {
		mvn("vaadin:update-widgetset gwt:compile");
	}

	public void compileWidgetset() throws IOException {
		mvn("gwt:compile");
	}

	// org.springframework.roo.addon.maven.MavenCommands not found at injection
	// time, copied and adapted a little code

	// copied from MavenCommands, slightly modified
	public void mvn(String extra) throws IOException {

		File root = new File(pathResolver.getRoot(Path.ROOT));
		Assert.isTrue(root.isDirectory() && root.exists(),
				"Project root does not currently exist as a directory ('"
						+ root.getCanonicalPath() + "')");

		String cmd = (File.separatorChar == '\\' ? "mvn.bat " : "mvn ") + extra;
		Process p = Runtime.getRuntime().exec(cmd, null, root);

		// Ensure separate threads are used for logging, as per ROO-652
		LoggingInputStream input = new LoggingInputStream(p.getInputStream(),
				processManager);
		LoggingInputStream errors = new LoggingInputStream(p.getErrorStream(),
				processManager);

		input.start();
		errors.start();

		try {
			p.waitFor();
		} catch (InterruptedException e) {
			throw new IllegalStateException(e);
		}
	}

	// copied from MavenCommands, slightly modified
	private class LoggingInputStream extends Thread {
		private BufferedReader inputStream;
		private ProcessManager processManager;

		public LoggingInputStream(InputStream inputStream,
				ProcessManager processManager) {
			this.inputStream = new BufferedReader(new InputStreamReader(
					inputStream));
			this.processManager = processManager;
		}

		@Override
		public void run() {
			ActiveProcessManager.setActiveProcessManager(processManager);
			String line;
			try {
				while ((line = inputStream.readLine()) != null) {
					if (line.startsWith("[ERROR]")) {
						logger.severe(line);
					} else if (line.startsWith("[WARNING]")) {
						logger.warning(line);
					} else {
						logger.info(line);
					}
				}
			} catch (IOException ioe) {
				if (ioe.getMessage().contains("No such file or directory") || // *nix/Mac
						ioe.getMessage().contains("CreateProcess error=2")) { // Windows
					logger.severe("Could not locate Maven executable; please ensure mvn command is in your path");
				}
			} finally {
				if (inputStream != null) {
					try {
						inputStream.close();
					} catch (IOException ignore) {
					}
				}
				ActiveProcessManager.clearActiveProcessManager();
			}
		}
	}

}