package com.vaadin.spring.roo.addon.abstractentityview;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.List;
import java.util.logging.Logger;

import org.springframework.roo.classpath.PhysicalTypeIdentifierNamingUtils;
import org.springframework.roo.classpath.PhysicalTypeMetadata;
import org.springframework.roo.classpath.TypeLocationService;
import org.springframework.roo.classpath.details.MethodMetadata;
import org.springframework.roo.classpath.details.MethodMetadataBuilder;
import org.springframework.roo.classpath.details.annotations.AnnotationMetadataBuilder;
import org.springframework.roo.classpath.itd.AbstractItdTypeDetailsProvidingMetadataItem;
import org.springframework.roo.classpath.itd.InvocableMemberBodyBuilder;
import org.springframework.roo.classpath.itd.ItdSourceFileComposer;
import org.springframework.roo.metadata.MetadataIdentificationUtils;
import org.springframework.roo.model.JavaSymbolName;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.process.manager.FileManager;
import org.springframework.roo.project.Path;
import org.springframework.roo.project.ProjectOperations;
import org.springframework.roo.support.style.ToStringCreator;
import org.springframework.roo.support.util.Assert;

import com.vaadin.spring.roo.addon.VaadinOperationsImpl;
import com.vaadin.spring.roo.addon.VaadinRooUtils;
import com.vaadin.spring.roo.addon.annotations.RooVaadinAbstractEntityView;
import com.vaadin.spring.roo.addon.entityform.VaadinClassNames;

/**
 * Metadata item for the {@link RooVaadinAbstractEntityView} annotation, used to
 * identify and list such classes, and to keep track of whether JPAContainer is
 * used.
 *
 * Some methods related to the use of JPAContainer or an in-memory container are
 * generated in the related aspect.
 */
public class VaadinAbstractEntityViewMetadata extends
		AbstractItdTypeDetailsProvidingMetadataItem {

	private static final String PROVIDES_TYPE_STRING = VaadinAbstractEntityViewMetadata.class
			.getName();
	private static final String PROVIDES_TYPE = MetadataIdentificationUtils
			.create(PROVIDES_TYPE_STRING);
	private static final JavaType ABSTRACT_ENTITY_VIEW_ANNOTATION_TYPE = new JavaType(
			RooVaadinAbstractEntityView.class.getName());

	// for JPAContainer
	private static final String TRANSACTIONAL_ANNOTATION_TYPE = "org.springframework.transaction.annotation.Transactional";

	public static final String DO_COMMIT_METHOD = "doCommit";
	public static final String DO_DELETE_METHOD = "doDelete";

	private static Logger logger = Logger
			.getLogger(VaadinAbstractEntityViewMetadata.class.getName());

	private final VaadinAbstractEntityViewAnnotationValues annotationValues;

	public VaadinAbstractEntityViewMetadata(String identifier,
			JavaType aspectName,
			PhysicalTypeMetadata governorPhysicalTypeMetadata,
			ProjectOperations projectOperations, FileManager fileManager,
			TypeLocationService typeLocationService,
			VaadinAbstractEntityViewAnnotationValues annotationValues) {
		super(identifier, aspectName, governorPhysicalTypeMetadata);
		this.annotationValues = annotationValues;
		Assert.isTrue(isValid(identifier), "Metadata identification string '"
				+ identifier + "' does not appear to be a valid");
		Assert.notNull(projectOperations, "Project operations required");
		Assert.notNull(fileManager, "File manager required");

		if (!isValid()) {
			return;
		}

		// add dependencies and helper classes if necessary
		if (annotationValues.isUseJpaContainer()) {
			JavaType governorType = governorTypeDetails.getName();
			VaadinOperationsImpl.setupJpaContainer(governorType,
					projectOperations, fileManager, typeLocationService);
		}

		addMethods();

		// Create a representation of the desired output ITD
		itdTypeDetails = builder.build();

		new ItdSourceFileComposer(itdTypeDetails);
	}

	@Override
	public String toString() {
		ToStringCreator tsc = new ToStringCreator(this);
		tsc.append("identifier", getId());
		tsc.append("valid", valid);
		tsc.append("aspectName", aspectName);
		tsc.append("destinationType", destination);
		tsc.append("governor", governorPhysicalTypeMetadata.getId());
		tsc.append("itdTypeDetails", itdTypeDetails);
		return tsc.toString();
	}

	public static final String getMetadataIdentiferString() {
		return PROVIDES_TYPE_STRING;
	}

	public static final String getMetadataIdentiferType() {
		return PROVIDES_TYPE;
	}

	public static final String createIdentifier(JavaType javaType, Path path) {
		return PhysicalTypeIdentifierNamingUtils.createIdentifier(
				PROVIDES_TYPE_STRING, javaType, path);
	}

	public static final JavaType getJavaType(String metadataIdentificationString) {
		return PhysicalTypeIdentifierNamingUtils.getJavaType(
				PROVIDES_TYPE_STRING, metadataIdentificationString);
	}

	public static final Path getPath(String metadataIdentificationString) {
		return PhysicalTypeIdentifierNamingUtils.getPath(PROVIDES_TYPE_STRING,
				metadataIdentificationString);
	}

	public static boolean isValid(String metadataIdentificationString) {
		return PhysicalTypeIdentifierNamingUtils.isValid(PROVIDES_TYPE_STRING,
				metadataIdentificationString);
	}

	public static JavaType getAnnotationType() {
		return ABSTRACT_ENTITY_VIEW_ANNOTATION_TYPE;
	}

	public boolean isUseJpaContainer() {
		return annotationValues.isUseJpaContainer();
	}

	private void addMethods() {
		// use JPAContainer for table
		boolean useJpaContainer = isUseJpaContainer();

		builder.addMethod(getDoCommitMethod(useJpaContainer));
		builder.addMethod(getDoDeleteMethod(useJpaContainer));
	}

	private MethodMetadata getDoCommitMethod(boolean useJpaContainer) {
		JavaSymbolName methodName = new JavaSymbolName(DO_COMMIT_METHOD);

		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		List<AnnotationMetadataBuilder> annotations = new ArrayList<AnnotationMetadataBuilder>();
		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		String entityClassName = "E";

		bodyBuilder.appendFormalLine("try {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("getForm().commit();");

		if (useJpaContainer) {
			annotations.add(createTransactionalAnnotation());

			bodyBuilder
					.appendFormalLine(entityClassName
							+ " entity = getEntityForItem(getForm().getItemDataSource());");
			bodyBuilder
					.appendFormalLine("if (!getTableContainer().containsId(getIdForEntity(entity))) {");
			bodyBuilder.indent();
			bodyBuilder.appendFormalLine("(("
					+ getClassName(VaadinClassNames.JPA_CONTAINER_CLASS) + "<"
					+ entityClassName
					+ ">) getTableContainer()).addEntity(entity);");
			bodyBuilder.indentRemove();
			bodyBuilder.appendFormalLine("}");
		} else {
			bodyBuilder
					.appendFormalLine("saveEntity(getEntityForItem(getForm().getItemDataSource()));");
		}

		bodyBuilder.appendFormalLine("return true;");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("} catch ("
				+ getClassName(VaadinClassNames.INVALID_VALUE_EXCEPTION_CLASS)
				+ " e) {");
		bodyBuilder.indent();
		bodyBuilder
				.appendFormalLine("// show validation error also on the save button");
		bodyBuilder
				.appendFormalLine("getForm().setCommitErrorMessage(e.getMessage());");
		bodyBuilder.appendFormalLine("return false;");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, methodName,
				JavaType.BOOLEAN_PRIMITIVE, bodyBuilder);
		methodBuilder.setAnnotations(annotations);
		return methodBuilder.build();
	}

	private MethodMetadata getDoDeleteMethod(boolean useJpaContainer) {
		JavaSymbolName methodName = new JavaSymbolName(DO_DELETE_METHOD);

		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		List<AnnotationMetadataBuilder> annotations = new ArrayList<AnnotationMetadataBuilder>();
		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		if (useJpaContainer) {
			annotations.add(createTransactionalAnnotation());

			bodyBuilder
					.appendFormalLine("Object id = getIdForEntity(getEntityForItem(getForm().getItemDataSource()));");
			bodyBuilder.appendFormalLine("if (id != null) {");
			bodyBuilder.indent();
			bodyBuilder.appendFormalLine("getTable().removeItem(id);");
			bodyBuilder.indentRemove();
			bodyBuilder.appendFormalLine("}");
		} else {
			bodyBuilder
					.appendFormalLine("deleteEntity(getEntityForItem(getForm().getItemDataSource()));");
		}

		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, methodName, JavaType.VOID_PRIMITIVE,
				bodyBuilder);
		methodBuilder.setAnnotations(annotations);
		return methodBuilder.build();
	}

	private AnnotationMetadataBuilder createTransactionalAnnotation() {
		JavaType transactionalType = new JavaType(TRANSACTIONAL_ANNOTATION_TYPE);
		AnnotationMetadataBuilder transactionalAnnotationBuilder = new AnnotationMetadataBuilder(
				transactionalType);
		return transactionalAnnotationBuilder;
	}

	/**
	 * Imports a type if necessary and returns the properly qualified type name
	 * including type parameters.
	 *
	 * @param typeName
	 * @return
	 */
	private String getClassName(String typeName) {
		JavaType type = new JavaType(typeName);
		return type.getNameIncludingTypeParameters(false,
				builder.getImportRegistrationResolver());
	}

}
