package com.vaadin.spring.roo.addon.entityform;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;

import org.springframework.roo.addon.plural.PluralMetadata;
import org.springframework.roo.classpath.details.BeanInfoUtils;
import org.springframework.roo.classpath.details.ClassOrInterfaceTypeDetails;
import org.springframework.roo.classpath.details.FieldMetadata;
import org.springframework.roo.classpath.details.ItdTypeDetailsBuilder;
import org.springframework.roo.classpath.details.MethodMetadata;
import org.springframework.roo.classpath.details.MethodMetadataBuilder;
import org.springframework.roo.classpath.itd.InvocableMemberBodyBuilder;
import org.springframework.roo.classpath.scanner.MemberDetails;
import org.springframework.roo.classpath.scanner.MemberDetailsScanner;
import org.springframework.roo.metadata.MetadataService;
import org.springframework.roo.model.DataType;
import org.springframework.roo.model.ImportRegistrationResolver;
import org.springframework.roo.model.JavaPackage;
import org.springframework.roo.model.JavaSymbolName;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.project.Path;
import org.springframework.roo.support.util.Assert;

import com.vaadin.spring.roo.addon.VaadinRooUtils;
import com.vaadin.spring.roo.addon.entityview.AbstractVaadinEntityViewMethodBuilder;
import com.vaadin.spring.roo.addon.entityview.ContainerMethodBuilder;
import com.vaadin.spring.roo.addon.entityview.ContainerMethodBuilder.JpaContainerParameters;
import com.vaadin.spring.roo.addon.entityview.VaadinEntityMetadataDetails;

public class FormMethodBuilder extends AbstractVaadinEntityViewMethodBuilder {

	public static final String MULTI_SELECT_FIELD_CLASS = VaadinClassNames.TWIN_COL_SELECT_CLASS;

	private static final String GET_ENTITY_CLASS_METHOD = "getEntityClass";
	protected static final String REFRESH_METHOD = "refresh";

	private final ItdTypeDetailsBuilder builder;

	// map from property name to the underlying (usually entity) type
	private Map<JavaSymbolName, JavaType> specialDomainTypes;

	private Map<JavaType, String> pluralCache;

	private static Logger logger = Logger.getLogger(FormMethodBuilder.class
			.getName());
	private final boolean useJpaContainer;

	private final JavaPackage utilPackage;

	public FormMethodBuilder(ClassOrInterfaceTypeDetails governorTypeDetails,
			String metadataId, VaadinEntityMetadataDetails vaadinEntityDetails,
			JavaPackage utilPackage, ImportRegistrationResolver importResolver,
			Map<JavaSymbolName, JavaType> specialDomainTypes,
			MetadataService metadataService,
			MemberDetailsScanner memberDetailsScanner,
			ItdTypeDetailsBuilder builder, boolean useJpaContainer) {
		super(governorTypeDetails, metadataId, vaadinEntityDetails,
				importResolver, metadataService, memberDetailsScanner);
		this.utilPackage = utilPackage;
		this.builder = builder;
		this.useJpaContainer = useJpaContainer;
		this.specialDomainTypes = specialDomainTypes;

		pluralCache = new HashMap<JavaType, String>();
	}

	protected boolean isSimpleDomainType(JavaType type,
			FieldMetadata fieldMetadata) {
		return VaadinRooUtils.isDomainTypeInProject(getMetadataService(), type)
				&& !VaadinRooUtils.isEmbeddedFieldType(fieldMetadata);
	}

	protected boolean isSimpleUnorderedDomainTypeCollection(JavaType type,
			FieldMetadata fieldMetadata) {
		logger.fine("Is " + type + " " + fieldMetadata
				+ " a simple unordered domain type collection?");
		if (Set.class.getName().equals(type.getFullyQualifiedTypeName())
				|| HashSet.class.getName().equals(
						type.getFullyQualifiedTypeName())) {
			List<JavaType> parameters = type.getParameters();
			if (parameters != null
					&& parameters.size() == 1
					&& VaadinRooUtils.isDomainTypeInProject(getMetadataService(),
							parameters.get(0))) {
				logger.fine("  Yes");
				return true;
			}
		}
		logger.fine("  No");
		return false;
	}

	protected Map<String, JavaType> getFieldSpecialTypes() {
		Map<String, JavaType> fieldSpecialTypes = new HashMap<String, JavaType>();
		
		MemberDetails memberDetails = getMemberDetails();

		Map<JavaSymbolName, MethodMetadata> accessors = VaadinRooUtils
				.getAccessors(getEntityType(), getEntityMetadata(),
						memberDetails, true);

		for (JavaSymbolName propertyName : accessors.keySet()) {
			FieldMetadata fieldMetadata = BeanInfoUtils
					.getFieldForPropertyName(memberDetails, propertyName);
			if (null == fieldMetadata) {
				continue;
			}

			JavaType type = accessors.get(propertyName).getReturnType();

			if (VaadinRooUtils.isEnumType(getMetadataService(), type)
					|| isSimpleDomainType(type,
							fieldMetadata) || isSimpleUnorderedDomainTypeCollection(
							type, fieldMetadata)) {
				fieldSpecialTypes.put(fieldMetadata.getFieldName()
						.getSymbolName(), type);
			}
		}
		return fieldSpecialTypes;
	}

	protected MethodMetadata buildDomainTypeComboBoxMethod(JavaType type,
			JavaSymbolName methodName) {
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		JavaType comboBoxFieldType = new JavaType(
				VaadinClassNames.COMBO_BOX_CLASS);
		String comboBoxClassName = getClassName(comboBoxFieldType);
		String containerMethod = getContainerCreationMethodName(type)
				.getSymbolName();
		bodyBuilder.appendFormalLine(comboBoxClassName + " combo = new "
				+ comboBoxClassName + "(null, " + containerMethod + "());");
		bodyBuilder.appendFormalLine("Object captionPropertyId = "
				+ getItemCaptionIdMethodName(type).getSymbolName() + "();");
		bodyBuilder.appendFormalLine("if (captionPropertyId != null) {");
		bodyBuilder.indent();
		bodyBuilder
				.appendFormalLine("combo.setItemCaptionPropertyId(captionPropertyId);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.appendFormalLine("return combo;");

		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, methodName, comboBoxFieldType,
				bodyBuilder);
		return methodBuilder.build();
	}

	protected MethodMetadata buildDomainTypeMultiSelectMethod(JavaType type,
			JavaSymbolName methodName) {
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);
		if (method != null) {
			return method;
		}

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		JavaType domainType = type.getParameters().get(0);

		JavaType selectFieldType = new JavaType(MULTI_SELECT_FIELD_CLASS);
		String selectClassName = getClassName(selectFieldType);
		String containerMethod = getContainerCreationMethodName(domainType)
				.getSymbolName();
		bodyBuilder.appendFormalLine(selectClassName + " select = new "
				+ selectClassName + "(null, " + containerMethod + "());");
		bodyBuilder.appendFormalLine("Object captionPropertyId = "
				+ getItemCaptionIdMethodName(domainType).getSymbolName()
				+ "();");
		bodyBuilder.appendFormalLine("if (captionPropertyId != null) {");
		bodyBuilder.indent();
		bodyBuilder
				.appendFormalLine("select.setItemCaptionPropertyId(captionPropertyId);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.appendFormalLine("return select;");

		MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
				getId(), Modifier.PUBLIC, methodName, selectFieldType,
				bodyBuilder);
		return methodBuilder.build();
	}

	public List<MethodMetadata> getDomainTypeContainerMethods() {
		List<MethodMetadata> methods = new ArrayList<MethodMetadata>();

		ContainerMethodBuilder containerBuilder = new ContainerMethodBuilder(
				governorTypeDetails, getId(), getVaadinEntityDetails(),
				getUtilPackage(), getBuilder()
						.getImportRegistrationResolver(), getMetadataService(),
				getMemberDetailsScanner(), getBuilder());

		JpaContainerParameters jpaParameters = useJpaContainer ? new JpaContainerParameters(
				false) : null;

		for (JavaType type : new LinkedHashSet<JavaType>(
				specialDomainTypes.values())) {
			if (type.equals(getEntityType())) {
				continue;
			}

			JavaSymbolName populateMethodName = getContainerCreationMethodName(type);
			MethodMetadata populateMethod = VaadinRooUtils.methodExists(
					populateMethodName, governorTypeDetails);
			if (populateMethod != null) {
				continue;
			}

			// as a side effect, this may create a field for the container
			populateMethod = containerBuilder.buildGetContainerMethod(type,
					populateMethodName, null, jpaParameters);

			if (populateMethod != null) {
				methods.add(populateMethod);
			}

		}
		return methods;
	}

	public MethodMetadata getGetEntityClassMethod() {
		String entityClassName = getClassName(getEntityType());
		JavaType returnType = new JavaType("java.lang.Class", 0, DataType.TYPE,
				null, Collections.singletonList(getEntityType()));
		return createReturnLiteralMethod(governorTypeDetails, getId(),
				GET_ENTITY_CLASS_METHOD, returnType,
				entityClassName + ".class");
	}

	protected JavaSymbolName getContainerCreationMethodName(JavaType type) {
		return new JavaSymbolName("getContainerFor" + getPlural(type));
	}

	private String getPlural(JavaType type) {
		if (pluralCache.get(type) != null) {
			return pluralCache.get(type);
		}
		PluralMetadata pluralMetadata = (PluralMetadata) getMetadataService()
				.get(PluralMetadata.createIdentifier(type, Path.SRC_MAIN_JAVA));
		Assert.notNull(
				pluralMetadata,
				"Could not determine the plural for the '"
						+ type.getFullyQualifiedTypeName() + "' type");
		pluralCache.put(type, pluralMetadata.getPlural());
		return pluralMetadata.getPlural();
	}

	protected ItdTypeDetailsBuilder getBuilder() {
		return builder;
	}

	private JavaPackage getUtilPackage() {
		return utilPackage;
	}

}
