package com.vaadin.spring.roo.addon.entityform;

import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.springframework.roo.addon.entity.EntityMetadata;
import org.springframework.roo.classpath.details.BeanInfoUtils;
import org.springframework.roo.classpath.details.ClassOrInterfaceTypeDetails;
import org.springframework.roo.classpath.details.FieldMetadata;
import org.springframework.roo.classpath.details.FieldMetadataBuilder;
import org.springframework.roo.classpath.details.ItdTypeDetailsBuilder;
import org.springframework.roo.classpath.details.MethodMetadata;
import org.springframework.roo.classpath.details.MethodMetadataBuilder;
import org.springframework.roo.classpath.details.annotations.AnnotatedJavaType;
import org.springframework.roo.classpath.itd.InvocableMemberBodyBuilder;
import org.springframework.roo.classpath.scanner.MemberDetails;
import org.springframework.roo.classpath.scanner.MemberDetailsScanner;
import org.springframework.roo.metadata.MetadataService;
import org.springframework.roo.model.DataType;
import org.springframework.roo.model.ImportRegistrationResolver;
import org.springframework.roo.model.JavaPackage;
import org.springframework.roo.model.JavaSymbolName;
import org.springframework.roo.model.JavaType;
import org.springframework.roo.project.Path;
import org.springframework.roo.support.util.StringUtils;

import com.vaadin.spring.roo.addon.VaadinRooUtils;
import com.vaadin.spring.roo.addon.entityview.VaadinEntityMetadataDetails;

public class VisuallyComposableFormMethodBuilder extends FormMethodBuilder {

	// for use from outside
	public static final String FIELD_LAYOUT_NAME = "fieldLayout";
	public static final String INDENT = "    ";

	// from CustomField add-on
	private static final String PROPERTY_CONVERTER_CLASS = "org.vaadin.addon.customfield.PropertyConverter";
	private static final String CONVERTING_VALIDATOR_CLASS = "org.vaadin.addon.customfield.ConvertingValidator";

	// from BeanValidation add-on
	private static final String BEAN_VALIDATION_VALIDATOR_CLASS = "com.vaadin.addon.beanvalidation.BeanValidationValidator";

	private static final String BEAN_CONVERTER_CLASS_SUFFIX = "BeanFieldPropertyConverter";
	private static final String BEAN_SET_CONVERTER_CLASS_SUFFIX = "BeanSetFieldPropertyConverter";

	// field, container, converter and validator configuration

	private static final String CONFIGURE_METHOD = "configure";

	private static final String FIELD_MAP_FIELD = "fieldMap";
	private static final String CONVERTER_MAP_FIELD = "converterMap";

	private static final String GET_BEAN_PROPERTIES_METHOD = "getBeanPropertyIds";
	private static final String GET_FIELD_METHOD = "getField";
	private static final String CONFIGURE_FIELD_MAP_METHOD = "configureFieldMap";

	private static final String CONFIGURE_FIELDS_METHOD = "configureFields";
	private static final String CONFIGURE_FIELD_CONTAINERS_METHOD = "configureContainersForFields";

	private static final String GET_CONVERTER_METHOD = "getConverter";
	private static final String CONFIGURE_CONVERTER_MAP_METHOD = "configureConverters";

	private static final String CONFIGURE_VALIDATORS_METHOD = "configureValidators";

	private static final String IS_MODIFIED_METHOD = "isModified";

	// form logic

	private static final String GET_FIRST_FIELD_METHOD = "getFirstField";
	private static final String VALIDATE_FIELDS_METHOD = "validateFields";
	private static final String COMMIT_FIELDS_METHOD = "commitFields";
	private static final String SET_PROPERTY_DATA_SOURCE_METHOD = "setFieldPropertyDataSource";
	private static final String SET_FIELD_VALUES_METHOD = "setFieldValues";

	public VisuallyComposableFormMethodBuilder(
			ClassOrInterfaceTypeDetails governorTypeDetails, String metadataId,
			VaadinEntityMetadataDetails vaadinEntityDetails,
			JavaPackage utilPackage, ImportRegistrationResolver importResolver,
			Map<JavaSymbolName, JavaType> specialDomainTypes,
			MetadataService metadataService,
			MemberDetailsScanner memberDetailsScanner,
			ItdTypeDetailsBuilder builder, boolean useJpaContainer) {
		super(governorTypeDetails, metadataId, vaadinEntityDetails,
				utilPackage, importResolver, specialDomainTypes,
				metadataService, memberDetailsScanner, builder, useJpaContainer);
	}

	/**
	 * Create a getField() method and (as a side effect) the corresponding field
	 * (Map), as well as method to initialize the map based on class fields and
	 * configure them.
	 *
	 * @return
	 */
	public List<MethodMetadata> getFormLogicMethods() {
		List<MethodMetadata> methods = new ArrayList<MethodMetadata>();

		methods.add(getValidateFieldsMethod());
		methods.add(getCommitFieldsMethod());
		methods.add(getSetDataSourceMethod());
		methods.add(getSetFieldValuesMethod());
		methods.add(getGetFirstFieldMethod());

		return methods;
	}

	/**
	 * Create a getField() method and (as a side effect) the corresponding field
	 * (Map), as well as method to initialize the map based on class fields and
	 * configure them.
	 *
	 * @return
	 */
	public List<MethodMetadata> getFieldMapMethods() {
		List<MethodMetadata> methods = new ArrayList<MethodMetadata>();

		JavaSymbolName methodName = new JavaSymbolName(GET_FIELD_METHOD);
		JavaType returnType = new JavaType(VaadinClassNames.FIELD_CLASS);

		JavaType objectType = new JavaType("java.lang.Object");

		getBuilder().addField(getFieldMapFieldBuilder());

		methods.add(getSingleLineGetterMethod(methodName, returnType,
				Collections.singletonList(objectType),
				Collections.singletonList(new JavaSymbolName("propertyId")),
				"return " + FIELD_MAP_FIELD + ".get(propertyId);"));

		methods.add(getConfigureMethod());
		methods.add(getRefreshMethod());
		methods.add(getIsModifiedMethod());
		methods.add(getConfigureFieldMapMethod());
		methods.add(getConfigureFieldsMethod());
		methods.add(getConfigureFieldContainersMethod());
		methods.add(getConfigureConvertersMethod());
		methods.add(getConfigureValidatorsMethod());

		return methods;
	}

	protected MethodMetadata getConfigureMethod() {
		JavaSymbolName methodName = new JavaSymbolName(CONFIGURE_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();
		bodyBuilder.appendFormalLine(CONFIGURE_FIELD_MAP_METHOD + "();");
		bodyBuilder.appendFormalLine(CONFIGURE_FIELDS_METHOD + "();");
		bodyBuilder.appendFormalLine(CONFIGURE_FIELD_CONTAINERS_METHOD + "();");
		bodyBuilder.appendFormalLine(CONFIGURE_CONVERTER_MAP_METHOD + "();");
		bodyBuilder.appendFormalLine(CONFIGURE_VALIDATORS_METHOD + "();");

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName,
					JavaType.VOID_PRIMITIVE, bodyBuilder);
			return methodBuilder.build();
		}
	}

	protected MethodMetadata getSingleLineGetterMethod(
			JavaSymbolName methodName, JavaType returnType,
			List<JavaType> parameterTypes, List<JavaSymbolName> parameterNames,
			String content) {
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();
		bodyBuilder.appendFormalLine(content);

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName, returnType,
					AnnotatedJavaType.convertFromJavaTypes(parameterTypes),
					parameterNames, bodyBuilder);
			return methodBuilder.build();
		}
	}

	private FieldMetadataBuilder getFieldMapFieldBuilder() {
		JavaType objectType = new JavaType("java.lang.Object");

		JavaType fieldType = new JavaType(VaadinClassNames.FIELD_CLASS);
		List<JavaType> fieldMapTypeParameters = Arrays.asList(new JavaType[] {
				objectType, fieldType });
		JavaType fieldMapFieldType = new JavaType("java.util.Map", 0,
				DataType.TYPE, null, fieldMapTypeParameters);
		JavaType fieldMapConcreteType = new JavaType("java.util.LinkedHashMap",
				0, DataType.TYPE, null, fieldMapTypeParameters);

		String fieldMapInitializer = "new "
				+ getClassName(fieldMapConcreteType) + "()";
		return new FieldMetadataBuilder(getId(), Modifier.PRIVATE,
				new JavaSymbolName(FIELD_MAP_FIELD), fieldMapFieldType,
				fieldMapInitializer);
	}

	protected MethodMetadata getConfigureFieldMapMethod() {
		JavaSymbolName methodName = new JavaSymbolName(CONFIGURE_FIELD_MAP_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();
		boolean fieldsAdded = false;
		// loop over entity properties, find corresponding fields
		for (String property : getBeanProperties()) {
			if (VaadinRooUtils.fieldExists(new JavaSymbolName(property
					+ "Field"), governorTypeDetails) != null) {
				bodyBuilder.appendFormalLine("fieldMap.put(\""+property+"\", "+property+"Field);");
				fieldsAdded = true;
			}
		}
		if (!fieldsAdded) {
			bodyBuilder
					.appendFormalLine("// no properties mapped - override this method to customize");
		}

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName,
					JavaType.VOID_PRIMITIVE, bodyBuilder);
			return methodBuilder.build();
		}
	}

	protected List<String> getBeanProperties() {
		List<String> beanProperties = new ArrayList<String>();

		MemberDetails memberDetails = getMemberDetails();
		Map<JavaSymbolName, MethodMetadata> accessors = VaadinRooUtils
				.getAccessors(getEntityType(), getEntityMetadata(),
						memberDetails, false);

		for (JavaSymbolName propertyName : accessors.keySet()) {
			beanProperties.add(StringUtils.uncapitalize(propertyName
					.getSymbolName()));
		}
		return beanProperties;
	}

	public MethodMetadata getGetBeanPropertiesMethod() {
		JavaType objectType = new JavaType("java.lang.Object");
		JavaType returnType = new JavaType("java.util.Collection", 0,
				DataType.TYPE, null, Collections.singletonList(objectType));

		StringBuilder content = new StringBuilder();
		content.append(getClassName("java.util.Arrays")
				+ ".asList(new Object[] { ");
		Iterator<String> it = getBeanProperties().iterator();
		while (it.hasNext()) {
			String propertyId = it.next();
			content.append("\"" + propertyId + "\"");
			if (it.hasNext()) {
				content.append(", ");
			}
		}
		content.append(" })");

		return createReturnLiteralMethod(governorTypeDetails, getId(),
				GET_BEAN_PROPERTIES_METHOD, returnType, content.toString());
	}

	protected MethodMetadata getConfigureFieldsMethod() {
		JavaSymbolName methodName = new JavaSymbolName(CONFIGURE_FIELDS_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();
		bodyBuilder
				.appendFormalLine("for (Object propertyId : "+GET_BEAN_PROPERTIES_METHOD+"()) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine(getClassName(VaadinClassNames.FIELD_CLASS)
				+ " field = getField(propertyId);");
		String textFieldClass = getClassName(VaadinClassNames.TEXT_FIELD_CLASS);
		bodyBuilder.appendFormalLine("if (field == null) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("continue;");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.appendFormalLine("if (field instanceof " + textFieldClass
				+ ") {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("((" + textFieldClass
				+ ") field).setNullRepresentation(\"\");");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.appendFormalLine("field.setWriteThrough(false);");
		// workaround to resolve #6406: see #6407
		bodyBuilder.appendFormalLine("field.setInvalidAllowed(true);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName,
					JavaType.VOID_PRIMITIVE, bodyBuilder);
			return methodBuilder.build();
		}
	}

	protected MethodMetadata getConfigureFieldContainersMethod() {
		JavaSymbolName methodName = new JavaSymbolName(
				CONFIGURE_FIELD_CONTAINERS_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		// enums, collections, domain type references, ... - anything that
		// requires customized field generation
		Map<String, JavaType> fieldSpecialTypes = getFieldSpecialTypes();

		if (!fieldSpecialTypes.isEmpty()) {
			String abstractSelectClass = getClassName(VaadinClassNames.ABSTRACT_SELECT_CLASS);
			MemberDetails memberDetails = getMemberDetails();

			bodyBuilder
					.appendFormalLine(getClassName(VaadinClassNames.FIELD_CLASS)
							+ " field;");
			for (Map.Entry<String, JavaType> entry : fieldSpecialTypes
					.entrySet()) {
				String property = entry.getKey();
				JavaType type = entry.getValue();

				bodyBuilder.appendFormalLine("");

				FieldMetadata fieldMetadata = BeanInfoUtils
						.getFieldForPropertyName(memberDetails, new JavaSymbolName(property));
				if (null == fieldMetadata) {
					continue;
				}
				boolean enumType = VaadinRooUtils.isEnumType(
						getMetadataService(), type);
				boolean simpleDomainType = isSimpleDomainType(type,
						fieldMetadata);
				boolean collectionType = isSimpleUnorderedDomainTypeCollection(
						type, fieldMetadata);
				if (enumType || simpleDomainType || collectionType) {
					JavaType entityType = collectionType ? type.getParameters()
							.get(0) : type;
					bodyBuilder.appendFormalLine("field = getField(\""+property+"\");");
					bodyBuilder.appendFormalLine("if (field instanceof "
							+ abstractSelectClass + ") {");
					bodyBuilder.indent();
					bodyBuilder.appendFormalLine("((" + abstractSelectClass
							+ ") field).setContainerDataSource("
							+ getContainerCreationMethodName(entityType)
							+ "());");

					bodyBuilder.appendFormalLine("Object captionId = "
							+ getItemCaptionIdMethodName(entityType) + "();");
					bodyBuilder.appendFormalLine("if (captionId != null) {");
					bodyBuilder.indent();
					bodyBuilder.appendFormalLine("((" + abstractSelectClass
							+ ") field).setItemCaptionPropertyId(captionId);");
					bodyBuilder.indentRemove();
					bodyBuilder.appendFormalLine("} else {");
					bodyBuilder.indent();
					// TODO uses BeanItem.toString(), not MyBean.toString()
					String modeName = enumType ? "ITEM_CAPTION_MODE_EXPLICIT_DEFAULTS_ID"
							: "ITEM_CAPTION_MODE_ITEM";
					bodyBuilder.appendFormalLine("((" + abstractSelectClass
							+ ") field).setItemCaptionMode("
							+ abstractSelectClass + "." + modeName + ");");
					bodyBuilder.indentRemove();
					bodyBuilder.appendFormalLine("}");

					bodyBuilder.indentRemove();
					bodyBuilder.appendFormalLine("}");
				}
			}
		} else {
			bodyBuilder.appendFormalLine("// no fields require special containers");
		}

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName,
					JavaType.VOID_PRIMITIVE, bodyBuilder);
			return methodBuilder.build();
		}
	}

	/**
	 * Create a getField() method and (as a side effect) the corresponding field
	 * (Map), as well as a method to initialize the map based on class fields.
	 *
	 * @return
	 */
	public List<MethodMetadata> getConverterMapMethods() {
		List<MethodMetadata> methods = new ArrayList<MethodMetadata>();

		JavaSymbolName methodName = new JavaSymbolName(GET_CONVERTER_METHOD);
		JavaType returnType = new JavaType(PROPERTY_CONVERTER_CLASS);

		JavaType objectType = new JavaType("java.lang.Object");

		getBuilder().addField(getConverterMapFieldBuilder());

		methods.add(getSingleLineGetterMethod(methodName, returnType,
				Collections.singletonList(objectType),
				Collections.singletonList(new JavaSymbolName("propertyId")),
				"return " + CONVERTER_MAP_FIELD + ".get(propertyId);"));

		// configure converters method is added by getFieldMapMethods()

		return methods;
	}

	private FieldMetadataBuilder getConverterMapFieldBuilder() {
		JavaType objectType = new JavaType("java.lang.Object");
		JavaType converterType = new JavaType(PROPERTY_CONVERTER_CLASS);
		List<JavaType> converterMapTypeParameters = Arrays
				.asList(new JavaType[] { objectType, converterType });
		JavaType converterMapFieldType = new JavaType("java.util.Map", 0,
				DataType.TYPE, null, converterMapTypeParameters);
		JavaType converterMapConcreteType = new JavaType(
				"java.util.LinkedHashMap", 0, DataType.TYPE, null,
				converterMapTypeParameters);

		String converterMapInitializer = "new "
				+ getClassName(converterMapConcreteType) + "()";
		return new FieldMetadataBuilder(getId(), Modifier.PRIVATE,
				new JavaSymbolName(CONVERTER_MAP_FIELD), converterMapFieldType,
				converterMapInitializer);
	}

	protected MethodMetadata getConfigureConvertersMethod() {
		JavaSymbolName methodName = new JavaSymbolName(
				CONFIGURE_CONVERTER_MAP_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		// loop over special domain properties, find corresponding fields, add
		// converters if field exists
		Map<String, JavaType> fieldSpecialTypes = getFieldSpecialTypes();

		if (!fieldSpecialTypes.isEmpty()) {
			String abstractSelectClass = getClassName(VaadinClassNames.ABSTRACT_SELECT_CLASS);
			String propertyConverterClass = getClassName(PROPERTY_CONVERTER_CLASS);
			String containerClass = getClassName(VaadinClassNames.CONTAINER_CLASS);

			MemberDetails memberDetails = getMemberDetails();

			bodyBuilder.appendFormalLine("// cannot parametrize "
					+ propertyConverterClass + " here due to an AJDT bug");
			bodyBuilder
					.appendFormalLine(propertyConverterClass + " converter;");
			bodyBuilder.appendFormalLine(containerClass + " container;");
			bodyBuilder
					.appendFormalLine(getClassName(VaadinClassNames.FIELD_CLASS)
							+ " field;");

			for (Map.Entry<String, JavaType> entry : fieldSpecialTypes
					.entrySet()) {
				bodyBuilder.appendFormalLine("");

				String property = entry.getKey();
				JavaType type = entry.getValue();
				FieldMetadata fieldMetadata = BeanInfoUtils
						.getFieldForPropertyName(memberDetails, new JavaSymbolName(property));
				if (null == fieldMetadata) {
					continue;
				}
				boolean enumType = VaadinRooUtils.isEnumType(
						getMetadataService(), type);
				boolean simpleDomainType = isSimpleDomainType(type,
						fieldMetadata);
				boolean collectionType = isSimpleUnorderedDomainTypeCollection(
						type, fieldMetadata);
				if (enumType) {
					// no conversion needed
				} else if (simpleDomainType || collectionType) {
					JavaType entityType = collectionType ? type.getParameters()
							.get(0) : type;
					EntityMetadata em = (EntityMetadata) getMetadataService()
							.get(EntityMetadata.createIdentifier(entityType,
									Path.SRC_MAIN_JAVA));

					bodyBuilder.appendFormalLine("field = getField(\""
							+ property + "\");");
					bodyBuilder.appendFormalLine("if (field instanceof "
							+ abstractSelectClass + ") {");
					bodyBuilder.indent();
					bodyBuilder.appendFormalLine("container = (("
							+ abstractSelectClass
							+ ") field).getContainerDataSource();");

					String suffix = simpleDomainType ? BEAN_CONVERTER_CLASS_SUFFIX
							: BEAN_SET_CONVERTER_CLASS_SUFFIX;

					FieldMetadata identifierField = em.getIdentifierField();
					List<JavaType> typeParams = Arrays.asList(entityType,
							identifierField.getFieldType());
					JavaType wrapperType = new JavaType(
							AutomaticFormMethodBuilder.BEAN_FIELD_WRAPPER_PACKAGE
									+ "." + suffix, 0, DataType.TYPE, null,
							typeParams);
					String converterClass = getClassName(wrapperType);

					bodyBuilder.appendFormalLine("converter = new "
							+ converterClass + "(" + getClassName(entityType)
							+ ".class, container, \""
							+ identifierField.getFieldName().toString()
							+ "\");");

					bodyBuilder.appendFormalLine(CONVERTER_MAP_FIELD
							+ ".put(\"" + property + "\", converter);");
					bodyBuilder.indentRemove();
					bodyBuilder.appendFormalLine("}");
				}
			}
		} else {
			bodyBuilder.appendFormalLine("// no converters needed");
		}

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName,
					JavaType.VOID_PRIMITIVE, bodyBuilder);
			return methodBuilder.build();
		}
	}

	protected MethodMetadata getConfigureValidatorsMethod() {
		JavaSymbolName methodName = new JavaSymbolName(
				CONFIGURE_VALIDATORS_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		bodyBuilder.appendFormalLine("for (Object propertyId : "
				+ GET_BEAN_PROPERTIES_METHOD + "()) {");
		bodyBuilder.indent();

		bodyBuilder.appendFormalLine(getClassName(VaadinClassNames.FIELD_CLASS)
				+ " field = " + GET_FIELD_METHOD + "(propertyId);");
		bodyBuilder.appendFormalLine("if (field != null) {");
		bodyBuilder.indent();

		String validatorClassName = getClassName(VaadinClassNames.VALIDATOR_CLASS);
		String beanValidatorClassName = getClassName(BEAN_VALIDATION_VALIDATOR_CLASS);
		String convertingValidatorClassName = getClassName(CONVERTING_VALIDATOR_CLASS);

		// remove old validators
		bodyBuilder.appendFormalLine(getClassName("java.util.Collection") + "<"
				+ validatorClassName + "> validators = field.getValidators();");
		bodyBuilder.appendFormalLine("if (validators != null) {");
		bodyBuilder.indent();
		JavaType validatorListType = new JavaType("java.util.ArrayList", 0,
				DataType.TYPE, null, Collections.singletonList(new JavaType(
						VaadinClassNames.VALIDATOR_CLASS)));
		bodyBuilder.appendFormalLine("for (" + validatorClassName
				+ " validator : new " + getClassName(validatorListType)
				+ "(field.getValidators())) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("if (validator instanceof "
				+ beanValidatorClassName + " || validator instanceof "
				+ convertingValidatorClassName + ") {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("field.removeValidator(validator);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		// prepare and add new validators, set "required", converters
		bodyBuilder.appendFormalLine(beanValidatorClassName
				+ " validator = new " + beanValidatorClassName
				+ "(getEntityClass(), String.valueOf(propertyId));");

		bodyBuilder.appendFormalLine("if (validator.isRequired()) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("field.setRequired(true);");
		bodyBuilder.appendFormalLine("field.setRequiredError(validator.getRequiredMessage());");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		bodyBuilder.appendFormalLine(getClassName(PROPERTY_CONVERTER_CLASS) + " converter = "+GET_CONVERTER_METHOD+"(propertyId);");

		bodyBuilder.appendFormalLine("if (converter == null) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("field.addValidator(validator);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("} else {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("field.addValidator(new "
				+ convertingValidatorClassName + "(validator, converter));");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		// if (field != null)
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		// for loop
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName,
					JavaType.VOID_PRIMITIVE, bodyBuilder);
			return methodBuilder.build();
		}
	}

	/**
	 * Create a refresh() method that updates all containers of the form fields.
	 *
	 * @return
	 */
	public MethodMetadata getRefreshMethod() {
		JavaSymbolName methodName = new JavaSymbolName(REFRESH_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();
		bodyBuilder.appendFormalLine(CONFIGURE_FIELD_CONTAINERS_METHOD + "();");
		bodyBuilder.appendFormalLine(CONFIGURE_CONVERTER_MAP_METHOD + "();");
		bodyBuilder.appendFormalLine(CONFIGURE_VALIDATORS_METHOD + "();");

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName,
					JavaType.VOID_PRIMITIVE, bodyBuilder);
			return methodBuilder.build();
		}
	}

	// methods for the form logic

	private MethodMetadata getDoForAllWriteableFieldsMethod(String methodName,
			String operation) {
		JavaSymbolName methodSymbolName = new JavaSymbolName(methodName);
		MethodMetadata method = VaadinRooUtils.methodExists(methodSymbolName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		bodyBuilder.appendFormalLine("if (getItemDataSource() != null) {");
		bodyBuilder.indent();
		bodyBuilder
				.appendFormalLine("for (Object propertyId : getItemDataSource().getItemPropertyIds()) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("Field field = getField(propertyId);");
		bodyBuilder
				.appendFormalLine("if (field != null && !field.isReadOnly()) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine(operation);
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodSymbolName,
					JavaType.VOID_PRIMITIVE, bodyBuilder);
			return methodBuilder.build();
		}
	}

	private MethodMetadata getCommitFieldsMethod() {
		return getDoForAllWriteableFieldsMethod(COMMIT_FIELDS_METHOD,
				"field.commit();");
	}

	private MethodMetadata getValidateFieldsMethod() {
		return getDoForAllWriteableFieldsMethod(VALIDATE_FIELDS_METHOD,
				"field.validate();");
	}

	private MethodMetadata getIsModifiedMethod() {
		JavaSymbolName methodSymbolName = new JavaSymbolName(IS_MODIFIED_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodSymbolName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		bodyBuilder.appendFormalLine("if (getItemDataSource() != null) {");
		bodyBuilder.indent();
		bodyBuilder
				.appendFormalLine("for (Object propertyId : getItemDataSource().getItemPropertyIds()) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("Field field = getField(propertyId);");
		bodyBuilder
				.appendFormalLine("if (field != null && field.isModified()) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("return true;");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.appendFormalLine("return false;");

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodSymbolName,
					JavaType.BOOLEAN_PRIMITIVE, bodyBuilder);
			return methodBuilder.build();
		}
	}

	private MethodMetadata getSetDataSourceMethod() {
		JavaSymbolName methodName = new JavaSymbolName(
				SET_PROPERTY_DATA_SOURCE_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		bodyBuilder.appendFormalLine("Field field = getField(propertyId);");
		bodyBuilder.appendFormalLine("if (field == null) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("return;");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		bodyBuilder.appendFormalLine("if (property == null) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("field.setPropertyDataSource(null);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("} else {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine(getClassName(PROPERTY_CONVERTER_CLASS)
				+ " converter = getConverter(propertyId);");
		bodyBuilder.appendFormalLine("if (converter != null) {");
		bodyBuilder.indent();
		bodyBuilder
				.appendFormalLine("converter.setPropertyDataSource(property);");
		bodyBuilder.appendFormalLine("field.setPropertyDataSource(converter);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("} else {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("if (field instanceof "
				+ getClassName(VaadinClassNames.CHECK_BOX_CLASS)
				+ " && property.getValue() == null) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("property.setValue(Boolean.FALSE);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.appendFormalLine("field.setPropertyDataSource(property);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		if (method != null) {
			return method;
		} else {
			List<AnnotatedJavaType> paramTypes = AnnotatedJavaType
					.convertFromJavaTypes(Arrays.asList(new JavaType[] {
							new JavaType("java.lang.Object"),
							new JavaType(VaadinClassNames.PROPERTY_CLASS) }));
			List<JavaSymbolName> paramNames = Arrays
					.asList(new JavaSymbolName[] {
							new JavaSymbolName("propertyId"),
							new JavaSymbolName("property") });
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName,
					JavaType.VOID_PRIMITIVE, paramTypes, paramNames,
					bodyBuilder);
			return methodBuilder.build();
		}
	}

	private MethodMetadata getSetFieldValuesMethod() {
		JavaSymbolName methodName = new JavaSymbolName(SET_FIELD_VALUES_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		bodyBuilder.appendFormalLine("if (item != null) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("// set values for fields in item");
		bodyBuilder
				.appendFormalLine("for (Object propertyId : item.getItemPropertyIds()) {");
		bodyBuilder.indent();
		bodyBuilder
				.appendFormalLine(SET_PROPERTY_DATA_SOURCE_METHOD+"(propertyId, item.getItemProperty(propertyId));");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder
				.appendFormalLine("// other fields are not touched by default");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("} else {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("// reset all fields");
		bodyBuilder
				.appendFormalLine("for (Object propertyId : "+GET_BEAN_PROPERTIES_METHOD+"()) {");
		bodyBuilder.indent();
		bodyBuilder
				.appendFormalLine(SET_PROPERTY_DATA_SOURCE_METHOD+"(propertyId, null);");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");

		if (method != null) {
			return method;
		} else {
			List<AnnotatedJavaType> paramTypes = AnnotatedJavaType
					.convertFromJavaTypes(Collections
							.singletonList(new JavaType(
									VaadinClassNames.ITEM_CLASS)));
			List<JavaSymbolName> paramNames = Collections
					.singletonList(new JavaSymbolName("item"));
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName,
					JavaType.VOID_PRIMITIVE, paramTypes, paramNames,
					bodyBuilder);
			return methodBuilder.build();
		}
	}

	private MethodMetadata getGetFirstFieldMethod() {
		JavaSymbolName methodName = new JavaSymbolName(GET_FIRST_FIELD_METHOD);
		MethodMetadata method = VaadinRooUtils.methodExists(methodName,
				governorTypeDetails);

		InvocableMemberBodyBuilder bodyBuilder = new InvocableMemberBodyBuilder();

		bodyBuilder.appendFormalLine(getClassName("java.util.Iterator")
				+ "<Object> it = " + GET_BEAN_PROPERTIES_METHOD
				+ "().iterator();");
		bodyBuilder.appendFormalLine("if (it.hasNext()) {");
		bodyBuilder.indent();
		bodyBuilder.appendFormalLine("return getField(it.next());");
		bodyBuilder.indentRemove();
		bodyBuilder.appendFormalLine("}");
		bodyBuilder.appendFormalLine("return null;");

		if (method != null) {
			return method;
		} else {
			MethodMetadataBuilder methodBuilder = new MethodMetadataBuilder(
					getId(), Modifier.PUBLIC, methodName, new JavaType(
							VaadinClassNames.FIELD_CLASS), bodyBuilder);
			return methodBuilder.build();
		}
	}

}
